/*********************************************************************\
 * tzgps.c -- GPS dohickey for the Palm/Handspring                   *
 *                                                                   *
 * This program is free software; you can redistribute it and/or     *
 * modify it under the terms of the GNU General Public License as    *
 * published by the Free Software Foundation; either version 2 of    *
 * the License, or (at your option) any later version.               *
 *                                                                   *
 * This program is distributed in the hope that it will be useful,   *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of    *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     *
 * GNU General Public License for more details.                      *
 *                                                                   *
 * You should have received a copy of the GNU General Public License *
 * along with this program; if not, contact:                         *
 *                                                                   *
 * Free Software Foundation           Voice:  +1-617-542-5942        *
 * 59 Temple Place - Suite 330        Fax:    +1-617-542-2652        *
 * Boston, MA  02111-1307,  USA       gnu@gnu.org                    *
 *                                                                   *
\*********************************************************************/


/*
 * FILE:
 * tzgps.c
 *
 * HISTORY:
 * Originally written by Tom Zerucha <tomz@users.sourceforge.net>
 *
 * This is hacked version of tzgps by Linas Vepstas
 * Copyright (C) 2002 Linas Vepstas <linas@linas.org> 
 *
 * This is a re-hacked version of tzgps by Daniel Seifert
 * Copyright (C) 2002 Daniel Seifert <dseifert@dseifert.com>
 *
 */


#include "tzgps.h"

#include "HandEra/Silk.h"
#include "HandEra/Vga.h"

#define mySrmReceive(a, b, c, d, e) (newSerial?SrmReceive(a,b,c,d,e):SerReceive(a,b,c,d,e))
#define mySrmReceiveCheck(a, b)     (newSerial?SrmReceiveCheck(a,b):SerReceiveCheck(a,b))
#define mySrmSend(a, b, c, d)       (newSerial?SrmSend(a,b,c,d):SerSend(a,b,c,d))
#define mySrmClearErr(a)            (newSerial?SrmClearErr(a):SerClearErr(a))
#define mySrmClose(a)               (newSerial?SrmClose(a):SerClose(a))

/* serial port */
UInt16 serRef;
Boolean newSerial = true;

/* graphic settings */
tzscreen myScreen;

VgaScreenModeType screenMode;
VgaRotateModeType rotateMode;

/* storage */
#define BUFMAX 512
static char bigbuf[BUFMAX];
static char param[20][20];

/* ========================================================= */
/* globally visible */
int lgtdeg, lgtmin, lgtfrac; /* longitude degrees, minutes, fraction */
int latdeg, latmin, latfrac; /* latitude degrees, minutes, fraction */
Int16 lat, lgt; /* degree-minutes */
char latns, lgtew;

int day, month, year;        /* UTC date */
int hour, min, sec, secfrac; /* UTC time */

char qual_gll='V';
char qual_gga='0';
char qual_gsa='1';

int rcvd_gll = 0;   /* received long/lat/time triplet */
int rcvd_gga = 0;   /* received an altitude recently */
int rcvd_rmc = 0;   /* received date/speed/heading recently */
int rcvd_sats = 0;  /* received satelite positions recently */

int altitude, altfrac, heading, speed, headingfrac, speedfrac;
char altunit;

satstat sats[12];

int totsats, fixsats;

char battery[5] = "    ";

int fastmode = 0;

/* ========================================================= */

/* Store last location before unit was powered off, used to 
 * initialize GPS when powered back on */
struct {
	Int32  timedif;  /* seconds between UTC and Palm Clock setting  */
	Int16  lat;      /* previously recorded position */
	Int16  lgt;
	char   latns;
	char   lgtew;
	UInt32 utc;      /* seconds UTC when last fix was taken */
} prf;

/* ========================================================= */

void init_gps_receiver (void) {
	DateTimeType dt;
	UInt16 pfsz = sizeof(prf);
	char initstr[128], ebuf[12];
	unsigned char sum, *cp;
	Err err;

	PrefGetAppPreferences(TZGPSAppID, 0, &prf, &pfsz, true);

	// printtd();

	TimSecondsToDateTime(TimGetSeconds() - prf.timedif, &dt);


	StrCopy(initstr, "$PRWIINIT,V,,," );

	StrIToA(ebuf, prf.lat);
	StrCat(initstr, ebuf);
	StrCat(initstr, ",");
	StrNCat(initstr, &prf.latns,1);
	StrCat(initstr, ",");
	StrIToA(ebuf, prf.lgt);
	StrCat(initstr, ebuf);
	StrCat(initstr, ",");
	StrNCat(initstr, &prf.lgtew,1);
	StrCat(initstr,",,,,,,");

	StrIToA(ebuf, dt.hour + 100);
	StrCat(initstr, &ebuf[1]);
	StrIToA(ebuf, dt.minute + 100);
	StrCat(initstr, &ebuf[1]);
	StrIToA(ebuf, dt.second + 100);
	StrCat(initstr, &ebuf[1]);
	StrCat(initstr, ",");
	StrIToA(ebuf, dt.day + 100);
	StrCat(initstr, &ebuf[1]);
	StrIToA(ebuf, dt.month + 100);
	StrCat(initstr, &ebuf[1]);
	StrIToA(ebuf, dt.year - 1900);
	StrCat(initstr, &ebuf[1]);
	sum = 0;
	cp = initstr;
	while (*++cp)
		sum ^= *cp;
	StrIToH(ebuf, sum);
	StrCat(initstr, "*");
	StrCat(initstr, &ebuf[6]);
	StrCat(initstr, "\r\n");


	/* open the serial port */
	#define BAUD 4800
	#define PORT 'u550' // 'u550' = CF, 0x8000 = cradle, iirc
 
	if (newSerial) {
		err = SrmOpen(PORT, BAUD, &serRef);
	} else {
		SerSettingsType serSettings;
		
		SysLibFind("Serial Library", &serRef);
		SerOpen(serRef, 0, BAUD);
		SerGetSettings(serRef, &serSettings);
		serSettings.flags = serSettingsFlagBitsPerChar8 | serSettingsFlagStopBits1;
		SerSetSettings(serRef, &serSettings);
	}

	mySrmSend(serRef, initstr, StrLen(initstr), &err);

}

void init_global_vars (void) {
	int i;

	latdeg = latmin = latfrac = lgtdeg = lgtmin = lgtfrac = 0;
	day = month = year = hour = min = sec = secfrac = 0;
	altitude = heading = speed = headingfrac = speedfrac = 0;

	latns = lgtew = altunit = ' ';
	totsats = fixsats = 0;

	for (i = 0; i < 12; i++)
		sats[i].num = sats[i].elev = sats[i].azim = sats[i].sn = sats[i].fix = 0;
}

/* ========================================================= */
/* make a note of the last good fix that we received */


UInt32 last_fix_utc = 0;
char last_fix_nmea[81] = "";

void
note_last_good_fix (char * buff)
{
	static UInt32 last_fix_gga = 0;
	DateTimeType dt;

	/* year needs to be set by a gga before we log anything */
	if (0 == year) return;

	dt.second = sec;
	dt.minute = min;
	dt.hour = hour;
	dt.day = day;
	dt.month = month;
	dt.year = year+2000;
	dt.weekDay = DayOfWeek(month, day, year);
	prf.utc = TimDateTimeToSeconds(&dt);
	prf.timedif = TimGetSeconds() - prf.utc;

	prf.lat = lat;
	prf.latns = latns;
	prf.lgt = lgt;
	prf.lgtew = lgtew;


	/* Any valid GGA message will do. */
	if ((0 == StrNCompare("$GPGGA", buff, 6)) &&
	'0' != qual_gga) 
	{
		last_fix_gga = prf.utc;
		last_fix_utc = prf.utc;
		StrCopy (last_fix_nmea, buff);
	}

	/* Prefer GGA messsages over GLL (because they have
	 * altitude), but only if the GGA is fairly fresh. 
	 * If GGA is old, note the GLL.
	 */
#define MAX_GGA_AGE 3

	else
	if (MAX_GGA_AGE < (prf.utc-last_fix_gga) && 
	(0 == StrNCompare("$GPGLL", buff, 6)) &&
	'A' == qual_gll) 
	{
		last_fix_utc = prf.utc;
		StrCopy (last_fix_nmea, buff);
	}

	// printtd();
}

Int32
get_data_age (void)
{
	Int32 age;

	age = TimGetSeconds() - prf.utc;
	age -= prf.timedif;

	return age;
}

UInt32
get_now_utc (void)
{
	UInt32 now;

	now = TimGetSeconds();
	now -= prf.timedif;

	return now;
}

/* ========================================================= */
/* log selected NMEA strings while the system is idling */

/* XXX fixme hack alert -- until we have a GUI that uses this
 * info, effectively disable logging by setting a huge idle value
 */
#define DEFAULT_LOG_IDLE_SECONDS 3600

static void
log_path_idle (char * buff)
{
	static UInt32 last_idle_log = 0;
	static int idle_log_cnt = 0;
	static int log_idle_seconds = DEFAULT_LOG_IDLE_SECONDS;

/* XXX hack to record a bunch of points quickly in succession */
static int fmc=0;
if (fastmode) {log_idle_seconds = 3; }

	/* log messsages only intermittently */
	if (log_idle_seconds > (prf.utc - last_idle_log)) return;

	/* log only GLL, RMC and GGA messages */
	if ((StrNCompare("$GPGLL", buff, 6)) &&
	(StrNCompare("$GPRMC", buff, 6)) &&
	(StrNCompare("$GPGGA", buff, 6))) 
	{
		 return;
	}

	/* log messages only if the quality is good */
	if ((rcvd_gll && 'A' == qual_gll) ||
	(rcvd_gga && '0' != qual_gga)) 
	{
		 StoreNmeaMsg (buff);
		 last_idle_log = prf.utc;
		 idle_log_cnt ++;

/* XXX hack to record a bunch of points */
if (fastmode) {fmc++; if (10<fmc) {fastmode=0; fmc=0;} }
	}
}

/* ========================================================= */

void set_latitude (char *parm)
{
	char editbuf[20];

	StrNCopy(editbuf, parm, 2);
	editbuf[2] = 0;
	latdeg = StrAToI(editbuf);

	StrNCopy(editbuf, &parm[2], 2);
	editbuf[2] = 0;
	latmin = StrAToI(editbuf);

	StrNCopy(editbuf, &parm[5], 4);
	StrCat(editbuf, "0000");
	editbuf[4] = 0;
	latfrac = StrAToI(editbuf);
}

void set_longitude(char *parm)
{
	char editbuf[20];

	StrNCopy(editbuf, parm, 3);
	editbuf[3] = 0;
	lgtdeg = StrAToI(editbuf);

	StrNCopy(editbuf, &parm[3], 2);
	editbuf[2] = 0;
	lgtmin = StrAToI(editbuf);

	StrNCopy(editbuf, &parm[6], 4);
	StrCat(editbuf, "0000");
	editbuf[4] = 0;
	lgtfrac = StrAToI(editbuf);
}

void set_utc_time(char *parm)
{
	char editbuf[20];

	StrNCopy(editbuf, parm, 2);
	editbuf[2] = 0;
	hour = StrAToI(editbuf);

	StrNCopy(editbuf, &parm[2], 2);
	editbuf[2] = 0;
	min = StrAToI(editbuf);

	StrNCopy(editbuf, &parm[4], 2);
	editbuf[2] = 0;
	sec = StrAToI(editbuf);

	StrNCopy(editbuf, &parm[7], 2);
	editbuf[2] = 0;
	secfrac = StrAToI(editbuf);
}

void 
getparms(char *buf, int max)
{
	char *bp, *pp;
	int i;

/* hack alert -- messages don't always start here -- fixme */
	pp = &buf[6];
	bp = &buf[7];
	i = 0;
	while (*bp) 
	{
		if ('\r' == *bp) *bp = 0;
		if ('*' == *bp) *bp = 0;
		if (',' == *bp++) 
		{
	bp[-1] = 0;
	StrCopy(param[i++], pp);
	pp = bp;
	if (i > max) break;
		}
	}
	StrCopy(param[i++], pp);
	while (i < max)
		param[i++][0] = 0;

}

void set_lat_long_utc(int a, int b, int c, int d, int e) {
	lat = StrAToI (param[a]);
	set_latitude (param[a]);
	latns = param[b][0];

	lgt = StrAToI (param[c]);
	set_longitude(param[c]);
	lgtew = param[d][0];

	set_utc_time(param[e]);
}

short checksum(char *buf) {
	unsigned char sum = 0;
	char *p = buf, hx[10];

	while (*++p != '*')
		sum ^= *p;
	StrIToH(hx, sum);
	return (StrNCompare(hx, ++p, 2) == 0);
}

/* ========================================================= */
/* Decodes the the NMEA messages, puts them into global vars. 
 * Returns 0 if message was decoded, returns 1 if message was not known
 *
 * hack alert -- we should check the checksum here, but we don't.
 */

int decode_NMEA0183 (char *buf) {
	int i, j, k;
	char editbuf[20];

	/* $GPGSV - GPS Satellites in view */
	/* Number of satellites (SV) in view, PRN numbers, elevation, 
	 * azimuth and SNR value. Four satellites maximum per transmission, 
	 * additional satellite data sent in second or third message. 
	 * Total number of messages being transmitted and the number of 
	 * messages being transmitted is indicated in the first two fields.
	 */
	/*        1 2 3  4  5  6   7           
	 * $--GSV,x,x,xx,xx,xx,xxx,xx..........xx,xx,xxx,xx*hh<CR><LF>
	 * 1    = Total number of messages of this type in this cycle, 1 to 3
	 * 2    = Message number, 1 to 3
	 * 3    = Total number of SVs in view
	 * 4    = SV PRN number
	 * 5    = Elevation in degrees, 90 maximum
	 * 6    = Azimuth, degrees from true north, 000 to 359
	 * 7    = SNR, 00-99 dB (null when not tracking)
	 * 8-11 = Information about second SV, same as field 4-7
	 * 12-15= Information about third SV, same as field 4-7
	 * 16-19= Information about fourth SV, same as field 4-7
	 */
	if (!StrNCompare("GPGSV", buf, 5)) 
	{
		getparms(buf, 20);
		totsats = StrAToI(param[2]);
		j = (StrAToI(param[1]) - 1) * 4;

		for (i = 3; i < 16; j++) {
			sats[j].num = StrAToI(param[i++]);
			sats[j].elev = StrAToI(param[i++]);
			sats[j].azim = StrAToI(param[i++]);
			sats[j].sn = StrAToI(param[i++]);
		}

		if (j > 10) rcvd_sats++;

	} else 

	/* $GPGLL - Geographic position, latitude / longitude */
	/*        1       2 3        4 5         6
	 * $--GLL,llll.ll,a,yyyyy.yy,a,hhmmss.ss,A 
	 *    1  = llll.ll = Latitude of position 
	 *    2  = a = N or S 
	 *    3  = yyyyy.yy = Longitude of position 
	 *    4  = a = E or W 
	 *    5  = hhmmss.ss = UTC of position 
	 *    6  = A = status: A = valid data; V = Data not valid
	 */
	if (!StrNCompare("GPGLL", buf, 5)) 
	{
		getparms(buf, 6);

		qual_gll = param[5][0];         // 'A'
		rcvd_gll ++;

		set_lat_long_utc(0, 1, 2, 3, 4);

	} else 

	/* $GPRMC - Recommended minimum specific GPS/Transit data */
	/*        1         2 3       4 5        6 7   8   9      10 11 12
	 * $GPRMC,hhmmss.ss,A,llll.ll,a,yyyyy.yy,a,x.x,x.x,ddmmyy,x.x,a*hh
	 * 1    = UTC of position fix
	 * 2    = Data status (V=navigation receiver warning, A=valid)
	 * 3    = Latitude of fix
	 * 4    = N or S
	 * 5    = Longitude of fix
	 * 6    = E or W
	 * 7    = Speed over ground in knots
	 * 8    = Course over ground in degrees true north
	 * 9    = UT date  (DDMMYY)
	 * 10   = Magnetic variation degrees (Easterly var. subtracts from true course)
	 * 11   = Magnetic variation, sense E or W
	 * 12   = Checksum (Mandatory in this sentence)
	 */
	if (!StrNCompare("GPRMC", buf, 5)) 
	{

		getparms(buf, 9);

		qual_gll = param[1][0];        
		rcvd_gll ++;
		rcvd_rmc ++;

		set_lat_long_utc(2, 3, 4, 5, 0);

		StrNCopy(editbuf, param[8], 2);
		editbuf[2] = 0;
		day = StrAToI(editbuf);

		StrNCopy(editbuf, &param[8][2], 2);
		editbuf[2] = 0;
		month = StrAToI(editbuf);

		StrNCopy(editbuf, &param[8][4], 2);
		editbuf[2] = 0;
		year = StrAToI(editbuf);

		speed = StrAToI(param[6]);
		{
			char *cp = param[6];
			while(*cp++ != '.' )
			{
				if( !*cp ) break;
			}
			speedfrac = StrAToI(cp);
		}

		heading = StrAToI(param[7]);
		headingfrac = StrAToI(&param[7][4]);

	} else

	/* $GPGGA - Global Positioning System Fix Data */
	/*        1         2       3 4        5 6 7  8   9  10 11  12 13  14
	 * $GPGGA,hhmmss.ss,llll.ll,a,yyyyy.yy,a,x,xx,x.x,x.x,M,x.x,M,x.x,xxxx
	 * 1    = UTC of Position
	 * 2    = Latitude
	 * 3    = N or S
	 * 4    = Longitude
	 * 5    = E or W
	 * 6    = GPS quality indicator 
	 *        0 = fix not available or invalid
	 *        1 = GPS SPS Mode, Fix valid
	 *        2 = Differential GPS or WAAS, SPS Mode, fix valid
	 *        3 = GPS PPS Mode, fix valid
	 * 7    = Number of satellites in use [not those in view]
	 * 8    = HDOP Horizontal dilution of position
	 * 9    = Antenna altitude above/below mean sea level (geoid)
	 * 10   = Meters  (Antenna height unit)
	 * 11   = Geoidal separation (Diff. between WGS-84 earth ellipsoid and
	 *           mean sea level.  -=geoid is below WGS-84 ellipsoid)
	 * 12   = Meters  (Units of geoidal separation)
	 * 13   = Age in seconds since last update from diff. reference station
	 * 14   = Diff. reference station ID#, 0000-1023
	 */

	if (!StrNCompare("GPGGA", buf, 5)) 
	{
		getparms(buf, 10);

		set_lat_long_utc(1, 2, 3, 4, 0);
		qual_gga = param[5][0];        

		fixsats = StrAToI(param[6]);
		altitude = StrAToI(param[8]);
		//    altfrac = StrAToI(&param[8][4]);
		altunit = param[9][0];

		rcvd_gga ++;

	} else 

	/* $GPGSA - GPS DOP and active satellites */
	/* GPS receiver operating mode, satelites used in the navigation 
	 * solution reported by the $--GGA sentence, and DOP values
	 * 
	 * DOP == Dilution of Precision
	 * 1    = Mode:
	 *        M=Manual, forced to operate in 2D or 3D
	 *        A=Automatic, 3D/2D
	 * 2    = Mode:
	 *        1=Fix not available
	 *        2=2D
	 *        3=3D
	 * 3-14 = PRN ID's of SVs used in position fix (null for unused fields)
	 * 15   = PDOP
	 * 16   = HDOP Horizontal DOP
	 * 17   = VDOP
	 */

	if (!StrNCompare("GPGSA", buf, 5)) 
	{
		getparms(buf, 17);
		qual_gsa = param[1][0];
	
		for (i = 0; i < 12; i++) sats[i].fix = 0;
	
		fixsats = 0;
		for (j = 0; j < 12; j++)
		{
			if ((k = StrAToI(param[j + 2])))
			{
				for (i = 0; i < totsats; i++)
			  	{
		   		if (sats[i].num == k) 
					{
						sats[i].fix = 1;
						fixsats ++;
						break;
					}
				}
			}
		}
	} else 

	/* 
	 * Magellan Proprietary
	 * $PMGNST,02.12,3,T,534,05.0,+03327,00*40 
	 * where:
	 *  ST      status information
	 *  02.12   Version number?
	 *  3       2D or 3D
	 *  T       True if we have a fix False otherwise
	 *  534     numbers change - unknown
	 *  05.0    time left on the gps battery in hours
	 *  +03327  numbers change (freq. compensation?)
	 *  00      PRN number receiving current focus
	 *  *40    checksum
	*/

	if (!StrNCompare("PMGNST", buf, 6)) 
	{
		getparms(buf, 7);
		battery[0] = param[4][0];
		battery[1] = param[4][1];
		battery[2] = param[4][2];
		battery[3] = param[4][3];
		battery[4] = 0x0;
	} else 

	/* 
	 * Magellan Proprietary
	 * $PMGNGO*1C
	 * 'GO' message, response to initializing unit.
	 */
	if (!StrNCompare("PMGNGO", buf, 6)) 
	{
	} else 
	{
		/* we don't know this message */
		return 1;
	}
	return 0;
}

/* ========================================================= */

/* Suck data out of serial port, store it in a temp buffer.
 * Returns 1 if we have one or more messages in the buffer,
 * otherwise retuirns 0.
 * If the 'discard' flag is set, the first message in teh buffer 
 * is discarded.
 * Whole messages are null-terminated by setting the last line-feed 
 * to zero. (the cr and checksum are left unharmed )
 */

int 
drain_serial_port(int discard_mesg)
{
	static int datalen = 0;
	int i;
	long nbytes;
	char *bp;
	Err err;

	/* If the discard message flag set, then we should 
	* throw away the first message in the buffer. */
	if (discard_mesg)
	{
	   bp = bigbuf;
	   i = 0;
	   while (*bp != '$' && i<datalen) { bp++, i++; }
	   if (i<=datalen) 
	   { 
	      MemMove ((void *) bigbuf, bp, datalen-i);
	      datalen -= i;
	   }
	}

	/* see how many bytes are waiting for us */
	if (mySrmReceiveCheck(serRef, &nbytes)) mySrmClearErr(serRef);
	
	/* check for buffer overflow */
	if (nbytes + datalen > BUFMAX) nbytes = BUFMAX - datalen;
	
	/* receive bytes, if there's room for them. */ 
	if (0 < nbytes) mySrmReceive(serRef, &bigbuf[datalen], nbytes, -1, &err);
	
	datalen += nbytes;
	
	/* shift so that start of message is at start of buffer */
	if (bigbuf[0] != '$') 
	{
		bp = bigbuf;
		while (*bp != '$' && datalen) bp++, datalen--;
	
		if (0 >= datalen) return 0;
	
		MemMove(bigbuf, bp, datalen);
	}

	/* if not enough data, we quit */
	if (datalen < 6) return 0;

	bp = bigbuf;
	i = datalen;

	/* hunt for * or CRLF (end of message); if not found, return. */
	bp++;
	i--;
	while (i) 
	{
		if (*bp == '\n') break;

		/* oops found start of message before we 
		 * found the end of the last one */
		if (*bp == '$') 
		{
			MemMove(bigbuf, bp, i);
			bp = bigbuf;
			datalen = i;
		}
		bp++;
		i--;
	}

	if (0 == i) return 0;

	/* null-terminate the message */
	*bp = 0;

	/* tell em we got a whole message waiting */
	return 1;
}

/* =============================================================== */

static Boolean ApplicationHandleEvent(EventPtr event) 
{
	FormPtr frm;
	UInt16  formID;
	Boolean handled = false;

	if (event->eType == frmLoadEvent) {
		formID = event->data.frmLoad.formID;
		frm = FrmInitForm(formID);
		FrmSetActiveForm(frm);

		switch (formID) {
 			case gupSplashPage:
				FrmSetEventHandler(frm, (FormEventHandlerPtr) gpsSplashPageHandler);
				break;
			case gupStatusPage:
				FrmSetEventHandler(frm, (FormEventHandlerPtr) gpsStatusPageHandler);
				break;
 			case gupNewWaypointPage:
				FrmSetEventHandler(frm, (FormEventHandlerPtr) gpsNewWaypointPageHandler);
				break;
			case gupReviewWaypointPage:
				FrmSetEventHandler(frm, (FormEventHandlerPtr) gpsReviewWaypointPageHandler);
				break;
			default:
				break;
		}
		handled = true;
	}

	return handled;
}

/* =============================================================== */

static void StopApplication(void) 
{
	UInt32 temp;

	/* close serial port and save prefs / last position */
	mySrmClose(serRef);
	PrefSetAppPreferences(TZGPSAppID, 0, 1, &prf, sizeof(prf), true);

	CloseWaypointDatabase();
	CloseNmeaDatabase();
	FreeWaypointFields();

	FrmSaveAllForms();
	FrmCloseAllForms();

	/* restore screen on Handera 330 */
	if (_TRGVGAFeaturePresent(&temp)) 
	{
		VgaSetScreenMode(screenMode, rotateMode);
	}
}


/* =============================================================== */

static void EventLoop(void) 
{
	EventType   e;
	UInt16      formID;
	Err         err;

	do {
		int got_msgs;

		EvtGetEvent(&e, SysTicksPerSecond()/4);
		EvtResetAutoOffTimer();

		/* rip through any messages in the serial port buffer */
		got_msgs = drain_serial_port(0);
		while (got_msgs) {
			char msg[90];   /* max nmea msg length is 80 chars */

			StrNCopy (msg, &bigbuf[1], 90);
			decode_NMEA0183(msg);
// if $!	WinDrawChars(bigbuf, StrLen(bigbuf), 0, 150);

 			/* If good data, then log the last good location */
			if ((rcvd_gll && 'A' == qual_gll) || (rcvd_gga && '0' != qual_gga)) {
 				note_last_good_fix(bigbuf);
				log_path_idle (bigbuf);
 			}

			got_msgs = drain_serial_port(1);
		}

		if ((! SysHandleEvent (&e))  
	      && (! MenuHandleEvent (NULL, &e, &err))
	      && (! ApplicationHandleEvent (&e)))
		{
			FrmDispatchEvent (&e);
		}

		/* -------------------------------------- */
		/* Update the display */

 		formID = FrmGetActiveFormID ();
		switch (formID) {
			case gupStatusPage:
				draw_sat_data();
				break;
 			case gupNewWaypointPage:
	     		draw_crypto();
				break;
		}

		rcvd_gll = 0;
		rcvd_gga = 0;
		rcvd_rmc = 0;
		rcvd_sats = 0;

	} while (e.eType != appStopEvent);
}


/* =============================================================== */

static UInt16 StartApplication(void) 
{
	UInt32   value;

	/* check for presence of New Serial Manager                       */
	if (FtrGet(sysFileCSerialMgr, sysFtrNewSerialPresent, &value))
 		newSerial = false;
	else
		newSerial = true;

	/* if we are running on a Handera, switch to high resolution      */
	if (_TRGVGAFeaturePresent(&value)) {
		VgaGetScreenMode(&screenMode, &rotateMode);
		VgaSetScreenMode(screenMode1To1, rotateMode);
		
		myScreen.highres = true;
		myScreen.scale = 15;
		myScreen.radius = 90;
		myScreen.cx = myScreen.cy = 120;
		myScreen.height = myScreen.width = 240;
	} else {
		myScreen.highres = false;
		myScreen.scale = 10;
		myScreen.radius = 60;
		myScreen.cx = myScreen.cy = 80;
		myScreen.height = myScreen.width = 160;
	}

	init_gps_receiver ();
	init_global_vars ();
	OpenNmeaDatabase ();
	OpenWaypointDatabase ();
	InitWaypointFields ();


	FrmGotoForm (gupStatusPage);
	return 0;
}


/* =============================================================== */

UInt32 PilotMain(UInt16 cmd, void *cmdPBP, UInt16 launchFlags) {
	UInt16 err;

	if (cmd == sysAppLaunchCmdNormalLaunch) {
		err = StartApplication();
		if (err) return err;
	
		EventLoop();
		StopApplication();
	} else {
		return sysErrParamErr;
	}

	return 0;
}

/* ========================== END OF FILE ============================ */
