
/*
 * FILE:
 * gui.c
 *
 * FUNCTION:
 * GUI elements
 *
 * HISTORY:
 * Linas Vepstas January 2002
 */

#include "tzgps.h"

// ==========================================================

typedef enum
{
	gpsWaypointName=0,
	gpsWaypointNote,
	gpsWaypointNmea,
	gpsNumFields
} gpsField;

/* The curent record will be stored on the heap, as a collection
 * of strings */
struct WaypointFields_s
{
	VoidHand field_handles [gpsNumFields];
	int lengths [gpsNumFields];
};

typedef struct WaypointFields_s WaypointFields;

WaypointFields curr_rec;
WinHandle sphereWin;

#define FSTR_CPY(i, str) {                      \
	int lenny;                                   \
	VoidHand h = curr_rec.field_handles[i];      \
	MemPtr mp = MemHandleLock (h);               \
	char * cp = (char *) mp;                     \
	StrCopy (cp, str);                           \
	lenny = StrLen (str);                        \
	curr_rec.lengths[i] = lenny;                 \
	MemPtrUnlock(mp);                            \
}

#define EMPTY(i) {                              \
	VoidHand h = curr_rec.field_handles[i];      \
	MemPtr mp = MemHandleLock (h);               \
	char * p = (char *) mp;                      \
	p[0] = 0;                                    \
	curr_rec.lengths[i] = 0;                     \
	MemPtrUnlock(mp);                            \
}

/* initialize memory on the heap for a set of strings */
Int16
InitWaypointFields (void)
{
	int i;
	VoidHand h;

	/* most fields aer 50 bytes max */
	for (i=0; i<gpsNumFields; i++)
	{
	   h = MemHandleNew (50);
	   curr_rec.field_handles[i] = h;
	}

	/* make these into zero-length strings */
	for (i=0; i<gpsNumFields; i++)
	{
	   EMPTY(i);
	}

	return 0;
}

void
FreeWaypointFields (void)
{
	int i;

	for (i=0; i<gpsNumFields; i++)
	{
	   MemHandleFree (curr_rec.field_handles[i]);
	}
}

void
ResetWaypointFields (void)
{
	/* we list most, but not all fields here */
	EMPTY (gpsWaypointName);
	EMPTY (gpsWaypointNote);
}

// ==========================================================

static int
PackedSize (void)
{
	int sz;
	int i;
	VoidHand h;
	MemPtr mp;
	char * p;

	sz = 0;
	for (i=0; i<gpsNumFields; i++)
	{
	   h = curr_rec.field_handles[i];
	   mp = MemHandleLock (h);
	   p = (char *) mp;
	   curr_rec.lengths[i] = StrLen (p) + 1;
	   sz += curr_rec.lengths[i];
	   MemPtrUnlock(mp);
	}

	sz ++;      // just for good luck
	return sz;
}

// ==========================================================
/* copy collection of strings to the database */

Int16
StoreWaypointRecord (void)
{
	UInt16       index = 0;
	VoidHand     RecHandle;
	MemPtr       RecPointer;
	int sz;
	int i;
	VoidHand h;
	MemPtr mp;
	char * p;

	RecHandle = DmNewRecord(WaypointDB, &index, PackedSize());
	RecPointer = MemHandleLock(RecHandle);

	sz = 0;
	for (i=0; i<gpsNumFields; i++)
	{
	   h = curr_rec.field_handles[i];
	   mp = MemHandleLock (h);
	   p = (char *) mp;
	   DmWrite(RecPointer, sz, p, curr_rec.lengths[i]);
	   sz += curr_rec.lengths[i];
	   MemPtrUnlock(mp);
	}

	MemPtrUnlock(RecPointer);
	DmReleaseRecord(WaypointDB, index, true);

	return index;
}

// ==========================================================
/* Copy from the indicated record to the struct */

void
GetWaypointRecord (UInt16 index)
{
	VoidHand     RecHandle;
	MemPtr       RecPointer;
	int i;
	char * p;

	RecHandle = DmGetRecord (WaypointDB, index);
	RecPointer = MemHandleLock(RecHandle);
	p = (char *) RecPointer;

	for (i=0; i<gpsNumFields; i++)
	{
		int len;
		FSTR_CPY (i, p);
	   len = curr_rec.lengths[i];
		p += len +1;
	}

	MemPtrUnlock(RecPointer);
	DmReleaseRecord(WaypointDB, index, true);
}

// ==========================================================

void 
show_review (int incr)
{
	char buff[80];
	int yloc =15;
	static int rec_idx = 0;
	int nr;
	char *clr="                                                   ";
	int clrlen = StrLen (clr);
	VoidHand h;
	MemPtr mp;
	char * p;
	int i;

	rec_idx += incr;
	nr = DmNumRecords (WaypointDB);
	if (0 > rec_idx) rec_idx = nr-1;
	if (nr <= rec_idx) rec_idx = 0;

	StrPrintF(buff, "Record  Number %d        ", rec_idx);
	WinDrawChars(buff, StrLen(buff), 10, yloc);  yloc +=12;

	GetWaypointRecord (rec_idx);

	for (i=0; i<gpsNumFields; i++)
	{
		int len = curr_rec.lengths[i];
	   h = curr_rec.field_handles[i];
	   mp = MemHandleLock (h);
	   p = (char *) mp;
		WinDrawChars(clr, clrlen, 10, yloc);  
		WinDrawChars(p, len, 10, yloc);  yloc +=12;
	   MemPtrUnlock(mp);
	}

}

// ==========================================================
// ==========================================================
// ==========================================================
// ==========================================================

static Boolean gpsPageFlip (Int16 butid) {
	switch (butid)	{
	   case 0:
	      FrmGotoForm(gupStatusPage);
	      return 1;
	   case 14:
	      FrmGotoForm(gupNewWaypointPage);
	      return 1;
	   case 15:
	      FrmGotoForm(gupReviewWaypointPage);
	      return 1;
	}
	return 0;
}

// ==========================================================

Boolean
gpsSplashPageHandler(EventPtr event)
{
	FormPtr   form;

	switch (event->eType)
	{
	   case frmOpenEvent:
	      form = FrmGetActiveForm();
	      FrmDrawForm(form);
	      return 1;

	   case ctlSelectEvent:
	      return gpsPageFlip (event->data.ctlEnter.controlID);

	   default:
	      return 0;
	}
	return 1;
}

// ==========================================================

Boolean gpsStatusPageHandler(EventPtr event) {
	FormPtr   form;
	Err err;
	Boolean handled = false;

	switch (event->eType) {
	   case frmOpenEvent:
	      form = FrmGetActiveForm();
	      FrmDrawForm(form);
			sphereWin = WinCreateOffscreenWindow(2*myScreen.radius+10, 2*myScreen.radius+10, genericFormat, &err);
			draw_sat_plot();
			handled = true;
			break;

		case frmCloseEvent:
			WinDeleteWindow(sphereWin, false);
			break;

      case popSelectEvent:
	      handled = gpsPageFlip(event->data.popSelect.controlID);
			break;
		
	   default:
	}

	return handled;
}

// ==========================================================

Handle
GetWaypointField (gpsField i)
{
	return (Handle) curr_rec.field_handles[i];
}


#define FSET(form,NAME) {                                            \
	FieldPtr fp;                                                      \
	fp = FrmGetObjectPtr(form, FrmGetObjectIndex(form, gup##NAME));   \
	FldSetTextHandle(fp, GetWaypointField (gps##NAME));               \
}

#define FRES(form,NAME) {                                            \
	FieldPtr fp;                                                      \
	fp = FrmGetObjectPtr(form, FrmGetObjectIndex(form, gup##NAME));   \
	FldSetTextHandle(fp, NULL);                                       \
}

extern int fastmode;

Boolean
gpsNewWaypointPageHandler(EventPtr event)
{
	FormPtr   form;

	switch (event->eType)
	{
	   case nilEvent: return 0;
	   case frmOpenEvent:
	      form = FrmGetActiveForm();

	      FSET (form, WaypointName);
	      FSET (form, WaypointNote);
	      FrmDrawForm(form);
	      return 1;

	   case frmCloseEvent:
	      form = FrmGetActiveForm();
	      FRES (form, WaypointName);
	      FRES (form, WaypointNote);
	      return 1;

	   case ctlSelectEvent:
	      if (gupLogWaypointButton == event->data.ctlEnter.controlID)
	      {

				/* quick -- grab a copy of a recent fix */
				/* hack alert -- we should not do this if the last fix is old. */
				FSTR_CPY (gpsWaypointNmea, last_fix_nmea);
	         StoreWaypointRecord();
	         // ResetWaypointFields();
	         return 1;
	      }
	      return gpsPageFlip (event->data.ctlEnter.controlID);

	   default:
	      return 0;
	}
	return 0;
}

// ==========================================================

Boolean gpsReviewWaypointPageHandler(EventPtr event) {
	FormPtr   form;

	switch (event->eType) {
	   case nilEvent:
			 return 0;

	   case frmOpenEvent:
	      form = FrmGetActiveForm();
	      FrmDrawForm(form);
			show_review (0);
	      return 1;

	   case frmCloseEvent:
	      form = FrmGetActiveForm();
	      return 1;

	   case ctlSelectEvent:
	      if (gupPrevWaypointButton == event->data.ctlEnter.controlID) {
				show_review (-1);
	         return 1;
	      }
	      if (gupNextWaypointButton == event->data.ctlEnter.controlID) {
				show_review (1);
	         return 1;
	      }
	      return gpsPageFlip(event->data.ctlEnter.controlID);

	   default:
	      return 0;

	}
	return 0;
}

// =================== END OF FILE ========================
