/********************************************************************\
 * graph.c -- GPS satellite graph & display routines                *
 *                                                                  *
 * This program is free software; you can redistribute it and/or    *
 * modify it under the terms of the GNU General Public License as   *
 * published by the Free Software Foundation; either version 2 of   *
 * the License, or (at your option) any later version.              *
 *                                                                  *
 * This program is distributed in the hope that it will be useful,  *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of   *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the    *
 * GNU General Public License for more details.                     *
 *                                                                  *
 * You should have received a copy of the GNU General Public License*
 * along with this program; if not, contact:                        *
 *                                                                  *
 * Free Software Foundation           Voice:  +1-617-542-5942       *
 * 59 Temple Place - Suite 330        Fax:    +1-617-542-2652       *
 * Boston, MA  02111-1307,  USA       gnu@gnu.org                   *
 *                                                                  *
\********************************************************************/


/*
 * FILE:
 * graph.c
 *
 * HISTORY:
 * Originally written by Tom Zerucha <tomz@users.sourceforge.net>
 *
 * This is hacked version of tzgps by Linas Vepstas
 * Copyright (C) 2002 Linas Vepstas <linas@linas.org> 
 *
 * This is a re-hacked version of tzgps by Daniel Seifert
 * Copyright (c) 2002 Daniel Seifert <dseifert@dseifert.com>
 */



#include "tzgps.h"

WinHandle sphereWin;

/* ========================================================= */
/* Sine and Cosine for compass type things */

long int sinetab[91] =
  { 0, 571, 1143, 1714, 2285, 2855, 3425, 3993, 4560, 5126, 5690, 6252, 6812,
  7371, 7927, 8480, 9032, 9580, 10125, 10668, 11207, 11743, 12275, 12803,
  13327, 13848, 14364, 14876, 15383, 15886, 16383, 16876, 17364, 17846, 18323,
  18794, 19260, 19720, 20173, 20621, 21062, 21497, 21926, 22347, 22762, 23170,
  23571, 23964, 24351, 24730, 25101, 25465, 25821, 26169, 26509, 26841, 27165,
  27481, 27788, 28087, 28377, 28659, 28932, 29196, 29451, 29697, 29935, 30163,
  30381, 30591, 30791, 30982, 31164, 31336, 31498, 31651, 31794, 31928, 32051,
  32165, 32270, 32364, 32449, 32523, 32588, 32643, 32688, 32723, 32748, 32763,
  32768
};

#define icos(x) isin((x)+90)

/* Takes angle in degrees, returns 32768*sine (pi*angle/180) */
UInt32 isin(Int16 angle) {
	while (angle >= 360)
		angle -= 360;
	while (angle < 0) 
		angle += 360; 

  if (angle <=  90) return sinetab[angle];
  if (angle <= 180) return sinetab[180 - angle];
  if (angle <= 270) return -sinetab[angle - 180];
  return -sinetab[360 - angle];
}

/* ========================================================= */

static RectangleType r;

#define DrawPixel(x,y) WinDrawLine(x,y,x,y);

static void DrawCircle(Int16 cx, Int16 cy, Int16 r) {
	Int16 d,x,y;

	d = -r; x = r; y = 0;
	while(x>=y) {
		DrawPixel(cx+x,cy+y);
		DrawPixel(cx+x,cy-y);
		DrawPixel(cx+y,cy+x);
		DrawPixel(cx+y,cy-x);
		DrawPixel(cx-x,cy+y);
		DrawPixel(cx-x,cy-y);
		DrawPixel(cx-y,cy+x);
		DrawPixel(cx-y,cy-x);
		d += y+y+1;
		if( d > 0 ) {
			--x;
			d -= x+x;
		}
		y++;
	}
}

/* ========================================================= */

void draw_sat_plot (void) {
	UInt16 i;
	RectangleType rect = { 
		{myScreen.cx-myScreen.radius-3, myScreen.cy-myScreen.radius-3},
		{2*myScreen.radius+5, 2*myScreen.radius+5} };

	/****** SATELLITE COMPASS VIEW ******/
	DrawCircle(myScreen.cx, myScreen.cy, myScreen.radius);
	DrawCircle(myScreen.cx, myScreen.cy, myScreen.radius/3);
	DrawCircle(myScreen.cx, myScreen.cy, myScreen.radius*2/3);

	for (i = 0; i < 360; i += 30) {
		WinDrawLine(myScreen.cx, myScreen.cy,
			myScreen.cx + (icos(90 - i) * myScreen.radius /	32768),
			myScreen.cy + (isin(90 - i) * myScreen.radius / 32768));
	}

	WinCopyRectangle(WinGetActiveWindow(), sphereWin, &rect, 0, 0, 0);
}

/* ========================================================= */

void draw_sat_data (void) {
	RectangleType rect = { 
		{0, 0},
		{2*myScreen.radius+10, 2*myScreen.radius+10} };

	/* Update the display */
	if (rcvd_gll && 'A' == qual_gll) {
		print_lat (); 
		print_lgt (); 
		print_utc (); 
		if (rcvd_rmc) {
			WinCopyRectangle(sphereWin, WinGetActiveWindow(), &rect,
				myScreen.cx-myScreen.radius-3, myScreen.cy-myScreen.radius-3, 0);

			print_speed(); 
			print_date(); 
			print_hdg(); 
			print_sats(); 
		} else if (rcvd_sats) {
				WinCopyRectangle(sphereWin, WinGetActiveWindow(), &rect,
					myScreen.cx-myScreen.radius-3, myScreen.cy-myScreen.radius-3, 0);
				print_sats(); 
			}
	}
	
	if (rcvd_gga && '0' != qual_gga) {
		print_lat (); 
		print_lgt (); 
		print_utc (); 
		print_alt (); 
	}
}

/* ========================================================= */

void printtd (UInt32 timedif) {
	char ebuf[40];
	// char ibuf[20];
	// unsigned long int td, td1;

	//      td = timedif / 3600;

	StrIToA(ebuf, timedif);
#if 0
	StrCat(ebuf, ":");

	td *= 3600;
	td1 = prf.timedif - td;

	td = td1 / 60;
	td1 %= 60;

	StrIToA(ibuf, 100 + td);
	StrCat(ebuf, &ibuf[1]);
	StrCat(ebuf, ":");
	StrIToA(ibuf, 100 + td1);
	StrCat(ebuf, &ibuf[1]);
#endif
	StrCat(ebuf, "   ");
	WinDrawChars(ebuf, StrLen(ebuf), 5*myScreen.scale, 15*myScreen.scale);
}


/* ========================================================= */
/* 
 * Draw a box whose size depends on the quality of the signal.
 * If we've got a fix on this satelite, then draw a square 
 * outline 
 */

void print_sats(void) {
	int i, j, a;
	int xv, yv;
	int fixc = 0;
	char ebuf[10];

	for (i = 0; i < totsats; i++) {
		if (sats[i].elev == 0)
			continue;

		a = 90 - sats[i].azim;

		xv = icos(a) * myScreen.radius / 32768;
		yv = isin(a) * myScreen.radius / 32768;

		r.topLeft.x = myScreen.cx + (90 - sats[i].elev) * xv / 90;
		r.topLeft.y = myScreen.cy - (90 - sats[i].elev) * yv / 90;

		r.extent.x = r.extent.y = 2*myScreen.scale/10;

		if (sats[i].sn) {
			j = (sats[i].sn - 34) / 3 + 1;
			r.topLeft.x -= j / 2;
			r.topLeft.y -= j / 2;
			r.extent.x += j;
			r.extent.y += j;
		}

		WinDrawRectangle(&r, 0);

		if (sats[i].fix) {
			r.topLeft.x++;
			r.topLeft.y++;
			r.extent.x -= 2;
			r.extent.y -= 2;
			WinEraseRectangle(&r, 0);
			++fixc;
		}
	}
	
	StrPrintF(ebuf, " Fix: %d ", fixc);
	WinDrawChars(ebuf, StrLen(ebuf), myScreen.cx+3+myScreen.radius-FntCharsWidth(ebuf,
	StrLen(ebuf)),myScreen.cy+myScreen.radius-FntCharHeight());
}

void print_lat (void) {
	char ebuf[40], ibuf[20];

	StrIToA(ibuf, latfrac + 10000);
	StrPrintF(ebuf, "     %d%d.%s' %c", latdeg, latmin, &ibuf[1], latns);
	WinDrawChars(ebuf, StrLen(ebuf), myScreen.cx-5-FntCharsWidth (ebuf, StrLen(ebuf)), myScreen.height-FntCharHeight()-2);
}

void print_lgt (void) {
	char ebuf[40], ibuf[20];
	
	StrIToA(ibuf, lgtfrac + 10000);
	StrPrintF(ebuf, "%d%d.%s' %c     ", lgtdeg, lgtmin, &ibuf[1], lgtew);
	WinDrawChars(ebuf, StrLen(ebuf), myScreen.cx+5, myScreen.height-FntCharHeight()-2);
}

void print_utc(void) {
	char ebuf[40], ibuf[20];

	StrPrintF(ebuf, "%d:", hour);
	StrIToA(ibuf, min + 100);
	StrCat(ebuf, &ibuf[1]);
	StrCat(ebuf, ":");
	StrIToA(ibuf, sec + 100);
	StrCat(ebuf, &ibuf[1]);
	StrCat(ebuf, ".");
	StrIToA(ibuf, secfrac + 100);
	StrCat(ebuf, &ibuf[1]);
	StrCat(ebuf, " ");
//  WinDrawChars(ebuf, StrLen(ebuf), 5*myScreen.scale, 14*myScreen.scale+5);
}


void print_date (void) {
	char ebuf[40];

	StrPrintF(ebuf, "%d/%d/%d   ", month, day, year);
//	WinDrawChars(ebuf, StrLen(ebuf), 11*myScreen.scale, 14*myScreen.scale+5);
}

void print_speed(void) {
	char ebuf[40];

	StrPrintF(ebuf, "     %d.%d Knts", speed, speedfrac);
	WinDrawChars(ebuf, StrLen(ebuf), myScreen.width - FntCharsWidth(ebuf, StrLen(ebuf)), 2);
}

void print_hdg(void) {
	char   ebuf[40];
	UInt16 a, xv, yv;

	StrPrintF(ebuf, "     %d.%d", heading, headingfrac);
	WinDrawChars(ebuf, StrLen(ebuf), myScreen.width - FntCharsWidth(ebuf, StrLen(ebuf)), 4+FntCharHeight());

	a = 90 - heading;
	xv = icos(a) * myScreen.radius / 32768;
	yv = isin(a) * myScreen.radius / 32768;

	WinDrawLine(myScreen.cx+xv-3,myScreen.cy+yv-3,myScreen.cx+xv+3,myScreen.cy+yv+3);
	WinDrawLine(myScreen.cx+xv-3,myScreen.cy+yv+3,myScreen.cx+xv+3,myScreen.cy+yv-3);
	WinDrawLine(myScreen.cx, myScreen.cy, myScreen.cx+xv, myScreen.cy+yv);
}

void print_alt(void) {
	char ebuf[40];

	StrPrintF(ebuf, "%d%c    ", altitude, altunit);
	// WinDrawChars(ebuf, StrLen(ebuf), 0, 14*myScreen.scale+5);
}


/* ========================== END OF FILE ============================ */
