/*
 * FILE:
 * testpipe.c
 *
 * FUNCTION:
 * Draw pipe.
 *
 * HISTORY:
 * Created by Linas Vepstas February 2002
 */

#include <math.h>
#include <stdlib.h>
#include <gtk/gtk.h>
#include "gtkstripchart.h"


typedef struct PipeSensor_s PipeSensor;
typedef struct PipeData_s PipeData;
typedef struct PipeChart_s PipeChart;

struct PipeSensor_s
{
  int location_angle;
  int npts;
  gdouble *dates;
  gdouble *thicks;

};


struct PipeData_s
{
  GList * sensors;
};


struct PipeChart_s
{
  GtkPlotCanvas *canvas;
  GtkPlot *polar;

  PipeData *pipe_data;

  // GtkPlotData *plot_data;
  GList *plot_data_list;

  int npts;
  gdouble *angles;
  gdouble *radii;

  GdkColor normal_color;
  GdkColor alarm_color;

  gdouble alarm_thickness;

  gdouble section_date;
};

/* ========================================================= */

PipeSensor *
pipe_sensor_new (void)
{
  PipeSensor *ps;

  ps = g_new (PipeSensor, 1);

  ps->location_angle = -1;
  ps->npts = 0;
  ps->dates = NULL;
  ps->thicks = NULL;

  return ps;
}

/* ========================================================= */

void
pipe_sensor_add_data (PipeSensor *ps, int angle, 
                 int npts, double *dates, double *thicks)
{
  if (!ps) return;
  ps->location_angle = angle;
  ps->npts = npts;
  ps->dates = dates;
  ps->thicks = thicks;

}

/* ========================================================= */

PipeData *
pipe_data_new (void)
{
  PipeData *pd = g_new(PipeData, 1);
  pd->sensors = NULL;
  return pd;
}

/* ========================================================= */

void
pipe_data_add_sensor (PipeData *pd, PipeSensor *ps)
{
  pd->sensors = g_list_append (pd->sensors, ps);
}

/* ========================================================= */

PipeChart *
pipe_chart_new (void)
{
  PipeChart *pc;

  GtkPlotCanvas *canvas;
  GtkPlot *pipe;

  canvas = GTK_PLOT_CANVAS (gtk_plot_canvas_new (300, 300, 1.0));
  gtk_widget_show (GTK_WIDGET(canvas));


  pipe = GTK_PLOT(gtk_plot_polar_new (NULL));
  gtk_widget_show (GTK_WIDGET(pipe));

  gtk_plot_hide_legends (pipe);
  gtk_plot_set_yrange (pipe, 0.0, 1.0);

  /* get rid of the scale on the left hand side */
  gtk_plot_axis_show_labels (pipe, GTK_PLOT_AXIS_LEFT, 0);
  gtk_plot_axis_show_ticks (pipe, GTK_PLOT_AXIS_LEFT, 0,0);
  gtk_plot_axis_set_visible (pipe, GTK_PLOT_AXIS_LEFT, 0);

  gtk_plot_resize (pipe, 0.8, 0.8);
  gtk_plot_canvas_add_plot (canvas, pipe, 0.1, 0.1);

  pc = g_new (PipeChart, 1);
  pc->canvas = canvas;
  pc->polar = pipe;
  pc->plot_data_list = NULL;

  pc->npts = 0;
  pc->angles = NULL;
  pc->radii = NULL;
  pc->section_date = 0.0;

  gdk_color_parse("green", &pc->normal_color);
  gdk_color_alloc(gdk_colormap_get_system(), &pc->normal_color);

  gdk_color_parse("orange", &pc->alarm_color);
  gdk_color_alloc(gdk_colormap_get_system(), &pc->alarm_color);



  return (pc);
}

/* ========================================================= */

void
pipe_chart_set_data (PipeChart *pc, PipeData *pd)
{
  PipeSensor *ps;
  GList *node;
  int i = 0;

  pc->pipe_data = pd;

  for (node = pd->sensors; node; node=node->next)
  {
    i ++;
  }

  i++;
  pc->npts = i;
  if (pc->angles) g_free (pc->angles);
  if (pc->radii) g_free (pc->radii);
  pc->angles = g_new (gdouble, i);
  pc->radii = g_new (gdouble, i);

  i = 0;
  for (node = pd->sensors; node; node=node->next)
  {
    ps = node->data;
    pc->angles[i] = ps->location_angle;
    pc->radii[i] = 1.0;
    i++;
  }

  /* last point is the same as the first */
  pc->angles[i] = pc->angles[0];
  pc->radii[i] = 1.0;

}


/* ========================================================= */

static double 
get_radius (PipeSensor *ps, time_t date)
{
  int j;
  gdouble r, ra, rb, da, db;

  /* xxx fixme use a faster search algo */
  for (j=0; j<ps->npts; j++)
  {
    if (date < ps->dates[j]) break;
  }
  if (0 == j) j = 1;
  if (j == ps->npts) j = j-1;
  ra = ps->thicks[j-1];
  rb = ps->thicks[j];

  da = ps->dates[j-1];
  db = ps->dates[j];

  /* linear interpolation */
  r = ra + ((rb-ra)/(db-da)) * (date-da);

  return r;
}

void
pipe_chart_show_slice (PipeChart *pc, time_t date)
{
  GdkColor *color;
  GtkPlotData *pld;
  PipeData *pd;
  PipeSensor *ps;
  GList *node;
  int i;
  int alrm=0, last_alrm=0, last_i;

  pc->section_date = date;

  /* first, get the pipe radii */
  pd = pc->pipe_data;
  i = 0;
  for (node = pd->sensors; node; node=node->next)
  {
    ps = node->data;

    pc->radii[i] = get_radius (ps, date);

    i++;
  }

  /* last point is the same as the first */
  pc->radii[i] = pc->radii[0];

  /* ---------------------------------- */
  /* Next, create multi-colored plot segments */

  last_i = 0;
  for (i=0; i<pc->npts; i++) 
  {
    if (pc->radii[i] <= pc->alarm_thickness) { alrm=1; } else { alrm=0; }
    if (0 == i) { last_alrm = alrm; continue; }

    if ((last_alrm == alrm) && 
        (i+1<pc->npts)) continue;

    /* if we are here, a mode change */
    pld = GTK_PLOT_DATA(gtk_plot_data_new ());
    gtk_plot_add_data (pc->polar, pld);
    gtk_widget_show (GTK_WIDGET(pld));

    if (1 == last_alrm) {
      color = &pc->alarm_color;
    } else {
      color = &pc->normal_color;
    }
#if (0 == GTKEXTRA_MAJOR_VERSION) && \
    (99 == GTKEXTRA_MINOR_VERSION) && \
    (17 >= GTKEXTRA_MICRO_VERSION)
    gtk_plot_data_set_line_attributes(pld,
                                    GTK_PLOT_LINE_SOLID,
                                    4, color);
#else
    gtk_plot_data_set_line_attributes(pld,
                                    GTK_PLOT_LINE_SOLID,
                                    GDK_CAP_BUTT,
                                    GDK_JOIN_MITER,
                                    4, color);
#endif

    
    gtk_plot_data_set_points (pld, 
        &pc->radii[last_i], &pc->angles[last_i], NULL, NULL, i-last_i+1);

    // fill_area is broken on polar plots
    // gtk_plot_data_fill_area (pld, 1);

    last_i = i;
    last_alrm = alrm;
  }

  /* ---------------------------------- */
  /* next, reverse the thickness in the polar plot */
  for (i=0; i<pc->npts; i++) 
  {
    pc->radii[i] = 1.0 - 0.5*pc->radii[i];
  }

  /* redraw now */
  gtk_plot_canvas_paint (pc->canvas);
  gtk_plot_canvas_refresh (pc->canvas);
}

/* ========================================================= */
/* create bogus pipe data */

void
fetch_bogus_data (int npts, gdouble *xpts, gdouble *ypts)
{
  int i;
  gdouble dr, th, d, s;
  static int angle = 0;

  s = sin (M_PI * angle/180.0);
  s += 1.0;
  s *= 0.5;
  s = 0.6 + 1.38*s;
  d = s / ((double) npts);
  th = 1.0;
  for (i=0; i<npts; i++)
  {
    xpts[i] = 32*365*24*3600 + 3*24*3600*i;
    ypts[i] = th;
    dr = ((double) rand ()) / ((double) RAND_MAX);
    dr *= d;
    th -= dr;
  }

  angle += 36;
}

/* ========================================================= */

void
add_data (PipeData *pd, GtkStripChartChart *chart)
{
  static int angle = 0;
  PipeSensor *ps;
  int npts = 20;
  gdouble *x, *y;


  x = g_new (gdouble, npts);
  y = g_new (gdouble, npts);

  fetch_bogus_data (npts, x, y);

  ps = pipe_sensor_new();
  pipe_sensor_add_data (ps, angle, npts, x, y);
  pipe_data_add_sensor (pd, ps);

  gtk_strip_chart_chart_add_strip (chart, npts, x, y);

  angle += 36;
}

/* ========================================================= */

static void
drag_cb (GtkStripChartChart *chart, GtkStripChartCursor *curs, gpointer user_data)
{
  PipeChart *pc = user_data;
  pipe_chart_show_slice (pc, curs->x_position);
}

/* ========================================================= */

int 
main(int argc, char *argv[])
{
  GdkColor color;
  GtkWidget *main_w, *hbox_w, *chart_w, *pipe_w;
  GtkStripChartChart *chart;
  GtkStripChartCursorSet *curset;
  PipeData *pd;
  PipeChart *pc;

  gtk_init(&argc,&argv);
  
  main_w = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW(main_w), "Strip Chart Demo");
  gtk_widget_set_usize (main_w,450,550);
  // gtk_container_border_width(GTK_CONTAINER(main_w),0);

  gtk_signal_connect (GTK_OBJECT (main_w), "destroy",
                 GTK_SIGNAL_FUNC (gtk_main_quit), NULL);

  hbox_w = gtk_hbox_new(FALSE,0);
  gtk_container_add (GTK_CONTAINER(main_w),hbox_w);
  gtk_widget_show (hbox_w);

/*
  scroll_w = gtk_scrolled_window_new(NULL, NULL);
  gtk_container_border_width (GTK_CONTAINER(scroll_w),0);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(scroll_w),
                             GTK_POLICY_ALWAYS,GTK_POLICY_ALWAYS);
  gtk_box_pack_start(GTK_BOX(hbox_w),scroll_w, TRUE, TRUE,0);
  gtk_widget_show(scroll_w);

  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scroll_w), chart_w);
*/

  chart_w = gtk_strip_chart_chart_new ();
  gtk_container_add (GTK_CONTAINER(hbox_w), chart_w);
  gtk_widget_show(chart_w);
  chart = GTK_STRIP_CHART_CHART (chart_w);

  /* override some attributes */
  gdk_color_parse("black", &color);
  chart->override_symbol_border_color = 0;
  chart->symbol.symbol_type = GTK_PLOT_SYMBOL_DIAMOND;
  chart->symbol.size = 8;
  chart->symbol.border.color = color;

  /* pipe */
  pc = pipe_chart_new();
  pipe_w = GTK_WIDGET (pc->canvas);
  gtk_container_add (GTK_CONTAINER(hbox_w), pipe_w);
  gtk_widget_show(pipe_w);

  gtk_signal_connect (GTK_OBJECT (chart_w), "cursor_drag",
                      GTK_SIGNAL_FUNC (drag_cb), pc);

  /* ------------------------------------------ */
  pd = pipe_data_new();

  add_data (pd, chart);
  add_data (pd, chart);
  add_data (pd, chart);
  add_data (pd, chart);
  add_data (pd, chart);
  add_data (pd, chart);
  add_data (pd, chart);
  add_data (pd, chart);
  add_data (pd, chart);
  add_data (pd, chart);


  gtk_strip_chart_chart_set_range (chart, 0.0, 1.0);

  gdk_color_parse("red", &color);
  gtk_strip_chart_chart_add_limit (chart, 0.25, color);
  gtk_strip_chart_chart_set_xrange (chart, 
             32*365*24*3600+ 6*24*3600, 
             32*365*24*3600 + 3*24*3600*20 + 6*24*3600);


  curset = gtk_strip_chart_chart_add_cursor (chart, NULL, 32*365*24*3600+ 6*24*3600+
                           2*24*3600*20, TRUE, 0);

  gtk_strip_chart_cursor_set_set_text (curset, "drag me");

  pc->alarm_thickness = 0.25;

  pipe_chart_set_data (pc, pd);
  pipe_chart_show_slice (pc, 32*365*24*3600+ 6*24*3600+ 2*24*3600*20);

  printf ("\n************************************\n");
  printf ("Use the arrow keys to zoom and scroll\n");
  printf ("Drag the green bar to chart the intersection\n");
  printf ("\n************************************\n");

  gtk_widget_show(main_w);
  gtk_main ();

  return 0;
}

/* ====================== END OF FILE ====================== */
