/*
 * FILE:
 * stripchart.c
 *
 * FUNCTION:
 * Display multiple strip charts
 *
 * HISTORY:
 * Lins Vepstas February 2002
 */

#include <float.h>
#include <math.h>
#include <time.h>
#include <glib.h>
#include <gdk/gdkkeysyms.h>

#include "gtkstripchart.h"
#include "gtkstripchartstrip.h"
#include "gtkstripchartcursorset.h"

/* ========================================================== */

enum  /* signal enumareation */
{
  CURSOR_SELECT,
  CURSOR_DRAG,
  LAST_SIGNAL
};

static guint chart_signals[LAST_SIGNAL] = {0};

/* ========================================================== */

/* class methods */
static void gtk_strip_chart_chart_class_init (GtkStripChartChartClass *);
static void gtk_strip_chart_chart_init (GtkStripChartChart *);
static void gtk_strip_chart_chart_destroy (GtkObject *object);
static void chart_draw    (GtkWidget *widget, GdkRectangle *area);
static void show_all  (GtkWidget *widget);

/* traditional callbacks */
static void size_allocate_cb  (GtkWidget *widget, 
                               GtkAllocation *allocation, 
                               gpointer *user_data);

static gint key_press_cb (GtkWidget *, GdkEventKey *, gpointer);
static gint butt_press_cb (GtkWidget *, GdkEventButton *, gpointer);
static gint motion_cb (GtkWidget *, GdkEventMotion *, gpointer);

/* ========================================================== */

GtkType
gtk_strip_chart_chart_get_type (void)
{
  static GtkType chart_type = 0;

  if (!chart_type)
    {
      GtkTypeInfo chart_info =
      {
        "GtkStripChartChart",
        sizeof (GtkStripChartChart),
        sizeof (GtkStripChartChartClass),
        (GtkClassInitFunc) gtk_strip_chart_chart_class_init,
        (GtkObjectInitFunc) gtk_strip_chart_chart_init,
        /* reserved 1*/ NULL,
        /* reserved 2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      chart_type = gtk_type_unique (gtk_plot_canvas_get_type(), &chart_info);
    }
  return chart_type;

}

/* ========================================================== */

static void
gtk_strip_chart_chart_class_init (GtkStripChartChartClass *klass)
{
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;

  object_class = (GtkObjectClass *) klass;
  widget_class = (GtkWidgetClass *) klass;

  widget_class->draw = chart_draw;

  widget_class->show_all = show_all;

  object_class->destroy = gtk_strip_chart_chart_destroy;

  klass->cursor_drag = NULL;

  chart_signals[CURSOR_SELECT] = 
    gtk_signal_new("cursor_select",
                   GTK_RUN_LAST,
                   object_class->type,
                   GTK_SIGNAL_OFFSET (GtkStripChartChartClass, cursor_select),
                   gtk_marshal_NONE__POINTER,
                   GTK_TYPE_NONE, 1, GTK_TYPE_STRIP_CHART_CURSOR);

  chart_signals[CURSOR_DRAG] = 
    gtk_signal_new("cursor_drag",
                   GTK_RUN_LAST,
                   object_class->type,
                   GTK_SIGNAL_OFFSET (GtkStripChartChartClass, cursor_drag),
                   gtk_marshal_NONE__POINTER,
                   GTK_TYPE_NONE, 1, GTK_TYPE_STRIP_CHART_CURSOR);

  gtk_object_class_add_signals (object_class, chart_signals, LAST_SIGNAL);

}

/* ========================================================== */

static void
gtk_strip_chart_chart_init (GtkStripChartChart *chart)
{
  GtkPlotCanvas *canvas;

  /* set up the canvas */
  canvas = &chart->plot_canvas;
  // GTK_PLOT_CANVAS_SET_FLAGS(canvas, GTK_PLOT_CANVAS_DND_FLAGS);

  // gtk_plot_canvas_set_magnification (canvas, 2.0);

  /* set the 'old width' and 'old height' */
  chart->ow = chart->oh = 1;

  chart->strip_list = NULL;
  chart->cursor_set_list = NULL;
  chart->drag_curs = NULL;

  chart->range_min = 0.0;
  chart->range_max = 1.0;

  chart->chart_min = time(0) - 30*24*3600;
  chart->chart_max = time(0);

  gdk_color_parse("blue", &chart->default_color);
  gdk_color_alloc(gdk_colormap_get_system(), &chart->default_color);

  /* -------------------------------- */
  /* drawing attributes */

  chart->override_symbol_color = 1;
  chart->override_symbol_border_color = 1;

  chart->line_attrs.line_style = GTK_PLOT_LINE_SOLID;
  chart->line_attrs.line_width = 2.0;
#if (0 == GTKEXTRA_MAJOR_VERSION) && \
    (99 == GTKEXTRA_MINOR_VERSION) && \
    (17 < GTKEXTRA_MICRO_VERSION)
  chart->line_attrs.cap_style = GDK_CAP_BUTT;
  chart->line_attrs.join_style = GDK_JOIN_MITER;
#endif

  chart->symbol.symbol_type = GTK_PLOT_SYMBOL_CIRCLE;
  chart->symbol.symbol_style = GTK_PLOT_SYMBOL_FILLED; 
  chart->symbol.size = 4;
  chart->symbol.border.line_width = 1;

  chart->line_connector = GTK_PLOT_CONNECT_STRAIGHT;

  /* -------------------------------- */
  
  chart->ylabel = g_strdup ("sn ");
  chart->title = NULL;

  chart->left_pixel_margin = 50;
  chart->right_pixel_margin = 33;
  chart->top_pixel_margin = 45;
  chart->bottom_pixel_margin = 60;
  chart->left_title_offset = 16;
  chart->strip_separation_pixels = 0;

  chart->x_label_spacing = 30;
  chart->y_label_spacing = 30;
  chart->nx_ticks = 1;
  chart->strip_height = 50;

  chart->use_date_labels = 1;

  gtk_signal_connect(GTK_OBJECT(chart), "size_allocate",
                     GTK_SIGNAL_FUNC(size_allocate_cb), NULL);

  gtk_signal_connect(GTK_OBJECT(chart), "key_press_event",
                     GTK_SIGNAL_FUNC(key_press_cb), NULL);

  gtk_signal_connect(GTK_OBJECT(chart), "button_press_event",
                     GTK_SIGNAL_FUNC(butt_press_cb), NULL);

  gtk_signal_connect(GTK_OBJECT(chart), "button_release_event",
                     GTK_SIGNAL_FUNC(butt_press_cb), NULL);

  gtk_signal_connect(GTK_OBJECT(chart), "motion_notify_event",
                     GTK_SIGNAL_FUNC(motion_cb), NULL);

}

/* ========================================================== */

GtkWidget *
gtk_strip_chart_chart_new (void)
{
  GtkStripChartChart *chart;

  chart = gtk_type_new (gtk_strip_chart_chart_get_type ());

  return GTK_WIDGET (chart);
}

/* ========================================================== */

static void 
gtk_strip_chart_chart_destroy (GtkObject *object)
{
  // GList *node;
  GtkStripChartChart *chart;

  g_return_if_fail (object != NULL);
  g_return_if_fail (GTK_IS_STRIP_CHART_CHART (object));

  chart = GTK_STRIP_CHART_CHART (object);

  // ?? hack alert fixme should we be walking the children, killing
  // them?
  g_list_free (chart->strip_list);
  g_list_free (chart->cursor_set_list);
}

/* ========================================================== */

static void
calc_y_tick_spacing (GtkStripChartChart *chart, gdouble *ret_sp, int *ret_sig)
{
  int i;
  gdouble tot_range, tick_spacing;
  gdouble log_sp, clog_sp, sp;

  /* List of allowed spacing values -- manually picked to look nice.
   * The final y-axis will have one of these times power of ten as the
   * major tick label. 
   */
  static const gdouble spacing[] = {0.1, 0.125, 0.15, 0.2, 0.25, 0.3, 0.4, 0.5, 0.75, 1.0 };

  /* Number of significant digits that need to be printed; must match
   * the number of digits in the list above */
  static const int sig_digs[] = {1, 3, 2, 1, 2, 1, 1, 1, 2, 1};

  /* the strip height is in pixels, as is the y_label_spacing */
  tot_range = chart->range_max - chart->range_min;
  tick_spacing = (gdouble) chart->y_label_spacing;
  tick_spacing /=  (gdouble) chart->strip_height;
  if (0.45 < tick_spacing) tick_spacing = 0.45;
  tick_spacing *= tot_range;

  /* We now have a tick spacing that gives us the right distance in pixels
   * between major tick marks.  But its a fractional value, and we want to 
   * 'round' it so it only has a few significant digits in it.  We use a bit 
   * of a trick for that. */

  log_sp = log10 (tick_spacing);
  clog_sp = floor (log_sp) +1;

  log_sp -= clog_sp;
  sp = pow (10.0, log_sp);

  i=0;
  while (sp > spacing[i]) i++;
  sp = spacing [i];

  sp *= pow (10.0, clog_sp);
  
  tick_spacing = sp;
  *ret_sp = tick_spacing;
  *ret_sig = sig_digs[i] + ((int) (-clog_sp));
}

/* ========================================================== */
/* Size setter */

static void 
set_size (GtkStripChartChart *chart)
{
  GtkPlotCanvas *canvas;
  GtkPlot *plot = NULL;
  GList *node;
  guint16 w,h;
  gdouble plot_height;
  int nstrips;
  gdouble yoff, ydelt;
  gdouble drange;

  int significant_digits;
  gdouble tick_spacing;
  gdouble one_over_w;
  gdouble one_over_h;
  gdouble new_left_margin;
  gdouble new_right_margin;
  gdouble new_bottom_margin;
  gdouble new_title_x;
  gdouble new_width;
  const gdouble squeeze = 0.82;

  nstrips=0;
  for (node=chart->strip_list; node; node=node->next) { nstrips ++; }
  if (0 == nstrips) return;

  canvas = GTK_PLOT_CANVAS (chart);
  /* w and h are in pixels */
  w = canvas->width;
  h = canvas->height;

  one_over_w = 1.0 / ((gdouble) w);
  one_over_h = 1.0 / ((gdouble) h);
  
  /* new left hand side of plot (expressed as fraction of total width 
   * of the canvas window) */
  new_left_margin = chart->left_pixel_margin * one_over_w;

  /* new right hand side of plot (expresssed as fraction of total width
   * of the canvas window) */
  new_right_margin = chart->right_pixel_margin * one_over_w;

  /* reposition the title */
  new_title_x = chart->left_title_offset * one_over_w;

  /* width of strips, as a fraction of total canvas width */
  new_width = 1.0 - new_right_margin - new_left_margin;
  if (0.6 > new_width)
  {
     new_left_margin *= squeeze;
     new_right_margin = 3.0 / ((gdouble) w);
     new_title_x *= squeeze;
     new_width = 1.0 - new_right_margin - new_left_margin;
  } 

  /* --------------------------- */
  /* strip y layout */
  yoff = chart->top_pixel_margin * one_over_h;
  new_bottom_margin = chart->bottom_pixel_margin * one_over_h;

  ydelt = 1.0 - yoff - new_bottom_margin;
  ydelt /= ((double) nstrips);
  plot_height = ydelt - chart->strip_separation_pixels *one_over_h;
  chart->strip_height = plot_height * (gdouble) h;

  /* --------------------------- */
  /* tick marks along y axis */

  calc_y_tick_spacing (chart, &tick_spacing, &significant_digits);

  /* --------------------------- */
  /* arrange the strips on the canvas */
  gtk_plot_canvas_freeze (canvas);
  for(node=chart->strip_list; node; node=node->next) 
  {
    GtkStripChartStrip *strip = node->data;
    plot = GTK_PLOT(strip);

    gtk_plot_axis_move_title (plot,GTK_PLOT_AXIS_LEFT, 90, 
            new_title_x, plot->left->title.y);

    if (GTK_WIDGET(plot)->parent)
    {
      gtk_plot_move_resize (plot, 
              new_left_margin, yoff,
              new_width, plot_height);
    }
    else 
    {
      gtk_plot_resize (plot, new_width, plot_height);
      gtk_plot_canvas_add_plot(canvas, plot, new_left_margin, yoff);
    }
    yoff += ydelt;

    /* --------------------------- */
    /* tick marks along y axis; two decimals, since that's our autoscaling */
    gtk_plot_axis_set_major_ticks (plot, GTK_PLOT_AXIS_Y, tick_spacing);
    gtk_plot_axis_set_labels_numbers (plot, GTK_PLOT_AXIS_LEFT, 
                                     GTK_PLOT_LABEL_FLOAT, 
                                     significant_digits);
    gtk_plot_axis_set_labels_numbers (plot, GTK_PLOT_AXIS_RIGHT, 
                                     GTK_PLOT_LABEL_FLOAT, 
                                     significant_digits);

    if (3 > chart->strip_separation_pixels)
    {
      gtk_plot_axis_set_ticks_limits (plot, GTK_PLOT_AXIS_Y, 
                chart->range_min, 
                chart->range_max -0.5 * tick_spacing);
    }
    else 
    {
      gtk_plot_axis_set_ticks_limits (plot, GTK_PLOT_AXIS_Y, 
                chart->range_min, 
                chart->range_max);
    }
  }

  /* --------------------------- */
  /* Set a suitable number of tick marks; but only on the 
   * topmost and bottommost strips */
  chart->nx_ticks = w * new_width / ((gdouble) chart->x_label_spacing);
  drange = chart->chart_max - chart->chart_min;
  drange /= chart->nx_ticks;
  
  gtk_plot_axis_set_major_ticks (plot, GTK_PLOT_AXIS_X, drange);
  plot = GTK_PLOT(chart->strip_list->data);
  gtk_plot_axis_set_major_ticks (plot, GTK_PLOT_AXIS_X, drange);

  gtk_plot_canvas_thaw (canvas);
}

/* ========================================================== */
/* This callback handles resize requests */

static void 
size_allocate_cb  (GtkWidget *widget,
                   GtkAllocation *allocation,
                   gpointer *user_data)
{
  guint16 w,h;
  GtkStripChartChart *chart;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_STRIP_CHART_CHART (widget));

  chart = GTK_STRIP_CHART_CHART (widget);

  w = allocation->width;
  h = allocation->height;

  /* need to protect against inf loop when setting size */
  if ((chart->ow != w) ||
      (chart->oh != h))
  {
    chart->ow = w;
    chart->oh = h;

    gtk_plot_canvas_set_size (GTK_PLOT_CANVAS(widget), w, h);
    set_size (chart);
  }
}

/* ========================================================== */

static gint 
key_press_cb (GtkWidget *w, GdkEventKey *ev, gpointer user_data)
{
  GtkStripChartChart *chart;
  gdouble width, delta;

  chart = GTK_STRIP_CHART_CHART (w);

  width = chart->chart_max - chart->chart_min;
  delta = width / 30;

  switch (ev->keyval)
  {
    case GDK_Page_Up: 
    case GDK_KP_Up:  /* numeric keypad */
    case GDK_Up:     /* arrow keys */
      gtk_strip_chart_chart_set_xrange (chart, chart->chart_min + delta,
                                         chart->chart_max - delta);
      break;
    case GDK_Page_Down:
    case GDK_KP_Down:
    case GDK_Down:
      gtk_strip_chart_chart_set_xrange (chart, chart->chart_min - delta,
                                         chart->chart_max + delta);
      break;
    case GDK_KP_Left:
    case GDK_Left:
      gtk_strip_chart_chart_set_xrange (chart, chart->chart_min + delta,
                                         chart->chart_max + delta);
      break;
    case GDK_KP_Right:
    case GDK_Right:
      gtk_strip_chart_chart_set_xrange (chart, chart->chart_min - delta,
                                         chart->chart_max - delta);
      break;

    default:
      return 0;
  }

  return 1;
}

/* ========================================================== */

/* How wide we treat a cursor, when we click on it */
#define CURS_DRAG_HALF_WIDTH 6.0

static gint 
butt_press_cb (GtkWidget *w, GdkEventButton *ev, gpointer user_data)
{
  GtkStripChartChart *chart = GTK_STRIP_CHART_CHART(w);

  if (GDK_BUTTON_PRESS == ev->type)
  {
    GtkStripChartStrip *strip = NULL;
    GtkPlot *plot = NULL;
    gdouble fx, fy, px, py;
    gint mouse_x, mouse_y;
  
    GList *node;
  
    mouse_x = ev->x;
    mouse_y = ev->y;
  
    /* find which plot this button press occured in */
    for (node=chart->strip_list; node; node=node->next)
    {
      strip = node->data;
      plot = GTK_PLOT(strip);
  
      gtk_plot_get_point (plot, mouse_x, mouse_y, &fx, &fy);
  
      if ((plot->xmin <= fx) && (fx <= plot->xmax) &&
          (plot->ymin <= fy) && (fy <= plot->ymax)) break;
    }
  
    if (NULL == node) return 0;  /* didn't find the plot */
  
    /* lets see if this plot has any cursors to drag */
    for (node=strip->curs_list; node; node=node->next)
    {
      GtkStripChartCursor *curs = node->data;
      fx = curs->x_position;
  
      gtk_plot_get_pixel (plot, fx, fy, &px, &py);
  
      if ((px-CURS_DRAG_HALF_WIDTH < ev->x) && 
          (px+CURS_DRAG_HALF_WIDTH > ev->x))
      {
         chart->drag_curs = curs;
         gtk_signal_emit (GTK_OBJECT(chart), 
              chart_signals[CURSOR_SELECT], chart->drag_curs);
      }
    }
    return 1;
  }
  
  if (GDK_BUTTON_RELEASE == ev->type)
  {
    chart->drag_curs = NULL;
    return 1;
  }
  return 0;
}

/* ========================================================== */

static gint 
motion_cb (GtkWidget *w, GdkEventMotion *ev, gpointer user_data)
{
  GtkStripChartCursor *curs;
  GtkPlot *plot;
  gdouble fx, fy;
  gint mouse_x, mouse_y;

  GtkStripChartChart *chart = GTK_STRIP_CHART_CHART(w);

  /* Ignore mouse motion when no button is depressed */
  if (0 == ev->state) return 0;

  /* handle only left-button-down */
  if (0 == (GDK_BUTTON1_MASK & ev->state)) return 0;

  /* nothing to drag ! */
  if (NULL ==  chart->drag_curs) return 1;

  /* not dragable ! */
  if (FALSE == chart->drag_curs->dragable) return 1;

  curs = chart->drag_curs;
  plot = curs->strip;
  mouse_x = ev->x;
  mouse_y = ev->y;

  gtk_plot_get_point (plot, mouse_x, mouse_y, &fx, &fy);

  if (fx < plot->xmin) fx = plot->xmin;
  if (fx > plot->xmax) fx = plot->xmax;
  gtk_strip_chart_cursor_set_pos (curs, fx);

  /* other cursors in this set get dragged too */
  gtk_strip_chart_cursor_set_set_pos (curs->cursor_set, fx);
  
  gtk_signal_emit (GTK_OBJECT(chart), chart_signals[CURSOR_DRAG], chart->drag_curs);
  gtk_plot_canvas_paint(&chart->plot_canvas);
  gtk_plot_canvas_refresh(&chart->plot_canvas);

  return 1;
}

/* ========================================================== */

static void
show_all (GtkWidget *widget)
{
  GList *node, *pldn;
  GtkStripChartChart *chart;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_STRIP_CHART_CHART (widget));

  chart = GTK_STRIP_CHART_CHART (widget);

  for (node=chart->strip_list; node; node=node->next)
  {
    GtkStripChartStrip *strip = node->data;
    // gtk_widget_show_all (GTK_WIDGET(strip->plot_data));

    for (pldn = strip->plot_data_list; pldn; pldn=pldn->next)
    {
       GtkPlotData *pld = pldn->data;
       gtk_widget_show (GTK_WIDGET(pld));
    }

    gtk_widget_show (GTK_WIDGET(strip));
  }
  gtk_widget_show (widget);

}

/* ========================================================== */

static void 
chart_draw (GtkWidget *widget, GdkRectangle *area)
{
  GtkStripChartChart *chart;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_STRIP_CHART_CHART (widget));

  chart = GTK_STRIP_CHART_CHART (widget);

  gtk_plot_canvas_paint(&chart->plot_canvas);
  gtk_plot_canvas_refresh(&chart->plot_canvas);

}

/* ========================================================== */

gboolean 
write_ticky (GtkPlotAxis *axis, gdouble *tick, gchar *label, gpointer user_data)
{
  GtkStripChartChart *chart  = user_data;
  time_t ltime;
  struct tm *ltm;

  ltime = (time_t) *tick;
  ltm = localtime (&ltime);

  /* if 24 hours or less then print hours not days */
  if (24*3600 > chart->chart_max - chart->chart_min)
  {
    snprintf (label, 100, "%02d:%02d:%02d", 
         ltm->tm_hour, ltm->tm_min, ltm->tm_sec);
  }
  else if (24*3600 > (chart->chart_max - chart->chart_min)/chart->nx_ticks)
  {
    /* hack alert -- this combo date time doesn't seem to work */
    snprintf (label, 100, "%02d/%02d/%02d\r\n%02d:%02d:%02d", 
         ltm->tm_mday, ltm->tm_mon+1, (ltm->tm_year)%100,
         ltm->tm_hour, ltm->tm_min, ltm->tm_sec);
  }
  else 
  {
    /* hack alert -- use strftime instead */
    snprintf (label, 100, "%02d/%02d/%02d", 
         ltm->tm_mday, ltm->tm_mon+1, (ltm->tm_year)%100);
  }
  return 1;
}

/* ========================================================== */
/* arrange the various strip charts on the canvas */

static void
place_strips (GtkStripChartChart *chart)
{
  GtkPlotCanvas *canvas = GTK_PLOT_CANVAS(chart);
  GtkPlotAxis *axis;
  GtkPlot *plot = NULL;
  GList *node;
  int i;

  if (NULL == chart->strip_list) return;

  /* set various strip attributes */
  i = 0;
  gtk_plot_canvas_freeze (canvas);
  for (node=chart->strip_list; node; node=node->next)
  {
    char buff[80];
    GtkStripChartStrip *strip = node->data;
    plot = GTK_PLOT(strip);

    gtk_plot_clip_data (plot, 1);
    gtk_plot_axis_show_labels (plot, GTK_PLOT_AXIS_BOTTOM, 0);
    gtk_plot_axis_show_ticks (plot, GTK_PLOT_AXIS_BOTTOM, 0, 0);

    i++;
    sprintf (buff, "%s%d", chart->ylabel, i);
    gtk_plot_axis_set_title (plot, GTK_PLOT_AXIS_LEFT, buff);
  }

  /* -------------------------------- */
  /* put the ticks only on the bottom graph */
  gtk_plot_axis_show_labels (plot, GTK_PLOT_AXIS_BOTTOM, GTK_PLOT_LABEL_OUT);
  gtk_plot_axis_show_ticks (plot, GTK_PLOT_AXIS_BOTTOM, 1, 1);

  gtk_plot_set_xrange (plot, chart->chart_min, chart->chart_max);

  if (chart->use_date_labels)
  {
    gtk_plot_axis_use_custom_tick_labels (plot, GTK_PLOT_AXIS_BOTTOM, 1);
    axis = gtk_plot_get_axis(plot, GTK_PLOT_AXIS_BOTTOM);
    gtk_signal_connect(GTK_OBJECT(axis), "tick_label",
                       GTK_SIGNAL_FUNC(write_ticky), chart);

  }

  gtk_plot_axis_set_labels_attributes (plot, GTK_PLOT_AXIS_BOTTOM,
           NULL, 10, 270, NULL, NULL, 0, GTK_JUSTIFY_LEFT);

  /* put ticks on the top of the top strip */
  plot = GTK_PLOT (chart->strip_list->data);
  gtk_plot_axis_show_ticks (plot, GTK_PLOT_AXIS_TOP, 1, 1);

  /* -------------------------------- */
  /* put the title on top of it all */
  if (chart->title)
  {
    gtk_plot_axis_set_title (plot, GTK_PLOT_AXIS_TOP, chart->title);  
    gtk_plot_axis_show_title (plot, GTK_PLOT_AXIS_TOP);  
  }
  else
  {
    gtk_plot_axis_hide_title (plot, GTK_PLOT_AXIS_TOP);  
  }

  /* -------------------------------- */
  /* arrange the strips on the canvas */
  set_size (chart);

  gtk_plot_canvas_thaw (canvas);
}

/* ========================================================== */

#define LINE_NEW(PLOT_SYMB) {                               \
    GdkColor *clr_sym, *clr_bord;                           \
    pld = GTK_PLOT_DATA (gtk_plot_data_new());              \
    gtk_plot_add_data (plot, pld);                          \
    gtk_plot_data_fill_area (pld, fill_area);               \
    strip->plot_data_list = g_list_append (strip->plot_data_list, pld); \
    gtk_widget_show (GTK_WIDGET(pld));                      \
                                                            \
    if (strip->override_symbol_color) {                     \
      clr_sym = color;                                      \
    } else {                                                \
      clr_sym = &strip->symbol.color;                       \
    }                                                       \
                                                            \
    if (strip->override_symbol_border_color) {              \
      clr_bord = color;                                     \
    } else {                                                \
      clr_bord = &strip->symbol.border.color;               \
    }                                                       \
                                                            \
    gtk_plot_data_set_symbol(pld,                           \
                             PLOT_SYMB,                     \
                             strip->symbol.symbol_style,    \
                             strip->symbol.size,            \
                             strip->symbol.border.line_width,\
                             clr_sym, clr_bord);            \
}

#if (0 == GTKEXTRA_MAJOR_VERSION) && \
    (99 == GTKEXTRA_MINOR_VERSION) && \
    (17 >= GTKEXTRA_MICRO_VERSION)

#define LINE_ATTR() {                                       \
    gtk_plot_data_set_line_attributes(pld,                  \
                             strip->line_attrs.line_style,  \
                             strip->line_attrs.line_width,  \
                             color);                        \
}

#else
#define LINE_ATTR() {                                       \
    gtk_plot_data_set_line_attributes(pld,                  \
                             strip->line_attrs.line_style,  \
                             strip->line_attrs.cap_style,   \
                             strip->line_attrs.join_style,  \
                             strip->line_attrs.line_width,  \
                             color);                        \
}
#endif



static void 
graph_data (GtkStripChartChart *chart, GtkStripChartStrip *strip)
{
  GList *stubble;
  gdouble *stub;
  gdouble delta;
  GtkPlot *plot;
  GList *node;
  int i;
  int istart, iend;
  GtkStripChartLimit *limit, *limit_last=NULL;
  gboolean fill_area = 0;

  if (0 >= strip->npts) return;

  plot = GTK_PLOT(strip);
  stubble = strip->stubs;

  /* Free old plot data */
  /* hack alert -- fixme -- we could save some CPU by recyling 
   * the old plotdata structs */
  for (node=strip->plot_data_list; node; node=node->next)
  {
    GtkPlotData *pld = node->data;
    gtk_plot_remove_data (plot, pld);
    gtk_object_destroy (GTK_OBJECT (pld));
  }
  g_list_free (strip->plot_data_list);
  strip->plot_data_list = NULL;

  if (0 >= strip->npts) return;

  /* loop over points until the range changes */
  istart = 1;  /* thats right, istart=1 */
  for (i=0; i<strip->npts; i++)
  {
    GList * lnode;
    GtkPlotData *pld;
    GdkColor *color = NULL;

    iend = i;
    color = &chart->default_color; 
    fill_area = 0;

    limit = NULL;
    for (lnode = strip->limit_list; lnode; lnode=lnode->next)
    {
      GtkStripChartLimit * next_limit = lnode->data;
      if  (next_limit->range_max < strip->ypts[i]) break;
      limit = next_limit;
    }

    if (0 == i) {limit_last = limit; continue; }

    /* if no crossings, continue */
    if ((limit_last == limit) && (i < strip->npts-1)) continue;


    /* ------------------------------------------------ */
    /* If we are here, then we have a limit-ine crossing */
    if (limit_last)
    {
      color = &limit_last->range_color;
      fill_area = limit_last->range_fill;
    }
    else
    {
      color = &chart->default_color; 
      fill_area = 0;
    }

    /* create a dataset; set dataset attributes -- this
     * for the main set of data points before the crossing */

    LINE_NEW(strip->symbol.symbol_type);
    LINE_ATTR();

    gtk_plot_data_set_points (pld, &strip->xpts[istart-1], 
                                   &strip->ypts[istart-1], 
                              NULL, NULL, iend - istart+1);

    if (iend < strip->npts-1)
    {
    /* And now do the stub */
    /* The stubs are two short lines, each of a different color,
     * drawn from the last point before a limit line, to the 
     * limit line (in the color of the preceeding segments), and
     * then from the limit line to the next 'true' datapoint,
     * (in the color of the limit line).  We do this fancy footwork
     * so that the line segment crossing the limit line changes
     * color as it crosses the limit line.  Wow ! 
     */
      g_return_if_fail (limit != NULL);

      if (stubble) 
      {
        stub = stubble->data;
        stubble = stubble->next;
      }  
      else 
      {
        stub = g_new (gdouble, 6);
        strip->stubs = g_list_prepend (strip->stubs, stub);
      }
  
      stub[0] = strip->xpts[iend-1];
      stub[2] = strip->xpts[iend];
      stub[3] = strip->ypts[iend-1];
      stub[4] = limit->range_max;
      stub[5] = strip->ypts[iend];

      /* linear interp to get x-intersection */
      delta = (strip->ypts[iend-1] - strip->ypts[iend]) ;
      if (delta) 
      {
        stub[1] = (strip->xpts[iend-1] - strip->xpts[iend]) ;
        stub[1] /= delta;
        stub[1] *= (limit->range_max - strip->ypts[iend]) ;
        stub[1] +=  strip->xpts[iend];
      }
      else
      {
         stub[1] = stub[0];
      }

      /* the line stub above the limit */
      LINE_NEW(GTK_PLOT_SYMBOL_NONE);
      LINE_ATTR();

      gtk_plot_data_set_points (pld, &stub[0], 
                                     &stub[3], 
                                NULL, NULL, 2);

      /* the line stub below the limit */
      color = &limit->range_color;
      fill_area = limit->range_fill;
      LINE_NEW(GTK_PLOT_SYMBOL_NONE);
      LINE_ATTR();

      gtk_plot_data_set_points (pld, &stub[1], 
                                     &stub[4], 
                                NULL, NULL, 2);

    }
    istart = iend+1;
    limit_last = limit;

  }

}

/* ========================================================== */

GtkPlot *
gtk_strip_chart_chart_add_strip (GtkStripChartChart *chart, 
                          int npts, gdouble *xarr, gdouble *yarr)
{
  GList *node;
  GtkStripChartStrip *strip;
  GtkPlot *plot;

  g_return_val_if_fail (chart != NULL, NULL);
  g_return_val_if_fail (GTK_IS_STRIP_CHART_CHART (chart), NULL);

  /* -------------------------------- */
  /* create a plot, add the plot to the canvas */
  strip = GTK_STRIP_CHART_STRIP(gtk_strip_chart_strip_new());
  plot = GTK_PLOT(strip);

  /* -------------------------------- */
  gtk_plot_set_yrange (plot, chart->range_min, chart->range_max);

  /* -------------------------------- */
  strip->xpts = xarr;
  strip->ypts = yarr;
  strip->npts = npts;

  chart->strip_list = g_list_append (chart->strip_list, strip);

  /* -------------------------------- */
  /* hack alert -- we should save limit lines previously set,
     and add them to the strip right here 
   */
  /* -------------------------------- */
  /* create cursors, add them to the strip */
  
  for (node=chart->cursor_set_list; node; node=node->next)
  {
    GtkStripChartCursorSet *cs = node->data;
    GtkStripChartCursor *curs;
    curs = GTK_STRIP_CHART_CURSOR(gtk_strip_chart_cursor_new());
    gtk_strip_chart_cursor_set_add_cursor (cs, curs);
    gtk_strip_chart_strip_add_cursor (strip, curs);

  }

  /* -------------------------------- */
  /* copy the line attributes over */

  strip->override_symbol_color = chart->override_symbol_color;
  strip->override_symbol_border_color = chart->override_symbol_border_color;

  strip->line_attrs.line_style = chart->line_attrs.line_style;
  strip->line_attrs.line_width = chart->line_attrs.line_width;

#if (0 == GTKEXTRA_MAJOR_VERSION) && \
    (99 == GTKEXTRA_MINOR_VERSION) && \
    (17 < GTKEXTRA_MICRO_VERSION)
  strip->line_attrs.cap_style =  chart->line_attrs.cap_style;
  strip->line_attrs.join_style =  chart->line_attrs.join_style;
#endif

  strip->symbol.symbol_type = chart->symbol.symbol_type;
  strip->symbol.symbol_style = chart->symbol.symbol_style;
  strip->symbol.size = chart->symbol.size;
  strip->symbol.border.line_width = chart->symbol.border.line_width;

  strip->line_connector = chart->line_connector;

  /* -------------------------------- */
  graph_data (chart, strip);
  place_strips (chart);

  gtk_widget_show (GTK_WIDGET(plot));

  return plot;
}

/* ========================================================== */

void
gtk_strip_chart_chart_set_range (GtkStripChartChart *chart, 
                           gdouble ymin, gdouble ymax)
{
  GList *node;
  gdouble tick_spacing;
  int significant_digits;

  g_return_if_fail (chart != NULL);
  g_return_if_fail (GTK_IS_STRIP_CHART_CHART (chart));

  /* reverse order if needed. This is arguably wrong, but all sorts
   * of code elsewhere assumes max > min and would break if it wasn't. */
  if (ymin > ymax) {gdouble tmp=ymin; ymin=ymax; ymax=tmp; }

  chart->range_min = ymin;
  chart->range_max = ymax;
  
  calc_y_tick_spacing (chart, &tick_spacing, &significant_digits);

  for (node=chart->strip_list; node; node=node->next)
  {
    GtkStripChartStrip *strip = node->data;
    GtkPlot *plot = GTK_PLOT(strip);

    gtk_strip_chart_strip_set_yrange (strip, chart->range_min, chart->range_max);
    /* tick marks along y axis; two decimals, since that's our autoscaling */
    gtk_plot_axis_set_major_ticks (plot, GTK_PLOT_AXIS_Y, tick_spacing);
    gtk_plot_axis_set_labels_numbers (plot, GTK_PLOT_AXIS_LEFT, 
                                     GTK_PLOT_LABEL_FLOAT, 
                                     significant_digits);
    gtk_plot_axis_set_labels_numbers (plot, GTK_PLOT_AXIS_RIGHT, 
                                     GTK_PLOT_LABEL_FLOAT, 
                                     significant_digits);

    if (3 > chart->strip_separation_pixels)
    {
      gtk_plot_axis_set_ticks_limits (plot, GTK_PLOT_AXIS_Y, 
                chart->range_min, 
                chart->range_max -0.5 * tick_spacing);
    }
    else 
    {
      gtk_plot_axis_set_ticks_limits (plot, GTK_PLOT_AXIS_Y, 
                chart->range_min, 
                chart->range_max);
    }

  }

}

/* ========================================================== */

void
gtk_strip_chart_chart_auto_range (GtkStripChartChart *chart)
{
  GList *node;

  for (node=chart->strip_list; node; node=node->next)
  {
    GtkStripChartStrip *strip = node->data;
    double ymin, ymax;
    int i;

    if (0 == strip->npts || NULL == strip->ypts) continue;

    ymax = -DBL_MAX;
    ymin = DBL_MAX;
  
    for (i=0; i<strip->npts; i++)
    {
      if (ymin > strip->ypts[i]) ymin = strip->ypts[i];
      if (ymax < strip->ypts[i]) ymax = strip->ypts[i];
    }
    gtk_strip_chart_chart_set_range (chart, ymin, ymax);
  }
}

/* ========================================================== */

void
gtk_strip_chart_chart_add_limit (GtkStripChartChart *chart, 
                           gdouble ymax, GdkColor color)
{
  GList *node;

  g_return_if_fail (chart != NULL);
  g_return_if_fail (GTK_IS_STRIP_CHART_CHART (chart));

  for (node=chart->strip_list; node; node=node->next)
  {
    GtkStripChartLimit *lim;
    GtkStripChartStrip *strip = node->data;

    lim = GTK_STRIP_CHART_LIMIT(gtk_strip_chart_limit_new());
    gtk_strip_chart_limit_set_range (lim, ymax);
    gtk_strip_chart_limit_set_color (lim, color);
    gtk_strip_chart_limit_set_width (lim,  chart->chart_min, chart->chart_max);
    gtk_strip_chart_strip_add_limit (strip, lim);

    graph_data (chart, strip);
  }
}

/* ========================================================== */

void   
gtk_strip_chart_chart_set_xrange (GtkStripChartChart *chart,
                             gdouble xmin, gdouble xmax)
{
  time_t drange;
  GList *node;

  g_return_if_fail (chart != NULL);
  g_return_if_fail (GTK_IS_STRIP_CHART_CHART (chart));

  chart->chart_min = xmin;
  chart->chart_max = xmax;

  drange = xmax - xmin;
  drange /= chart->nx_ticks;

  for (node=chart->strip_list; node; node=node->next)
  {
    GtkStripChartStrip *strip = node->data;
    GtkPlot *plot = GTK_PLOT(strip);

    gtk_strip_chart_strip_set_xrange (strip, xmin, xmax);

    gtk_plot_axis_set_major_ticks (plot, GTK_PLOT_AXIS_X, drange);

    // The default config for strip charts doesn't show legends, 
    // but if the user turned them on, we should keep them visibile
    // by sliding them into place  ...
    // gtk_plot_legends_move (plot, .....);
  }

  gtk_plot_canvas_paint(&chart->plot_canvas);
  gtk_plot_canvas_refresh(&chart->plot_canvas);
}

/* ========================================================== */

void
gtk_strip_chart_chart_auto_xrange (GtkStripChartChart *chart)
{
  GList *node;
  int got_data = 0;
  double earliest, latest;

  latest = -DBL_MAX;
  earliest = DBL_MAX;
  
  for (node=chart->strip_list; node; node=node->next)
  {
    GtkStripChartStrip *strip = node->data;
    if (0 == strip->npts || NULL == strip->xpts) continue;
    if (earliest > strip->xpts[0]) earliest = strip->xpts[0];
    if (latest < strip->xpts[strip->npts-1]) latest = strip->xpts[strip->npts-1];
    got_data = 1;
  }
  if (got_data) gtk_strip_chart_chart_set_xrange (chart, earliest, latest);
}

/* ========================================================== */

GtkStripChartCursorSet *
gtk_strip_chart_chart_add_cursor (GtkStripChartChart *chart, 
                            GdkColor *color, gdouble pos,
                            gboolean dragable, gint style)
{
  GList *node;
  GtkStripChartCursorSet *cs;
  GdkColor bogus;

  cs = GTK_STRIP_CHART_CURSOR_SET(gtk_strip_chart_cursor_set_new());

  /* set attributes on cursor set first; the cursor will inherit. */
  if (color)
  {
    GtkPlotData *pld = GTK_PLOT_DATA (cs);
    GtkPlotLineStyle line_style;
    gfloat line_width;
#if (0 == GTKEXTRA_MAJOR_VERSION) && \
    (99 == GTKEXTRA_MINOR_VERSION) && \
    (17 >= GTKEXTRA_MICRO_VERSION)
    gtk_plot_data_get_line_attributes   (pld,
                    &line_style, &line_width, &bogus);

    gtk_plot_data_set_line_attributes(pld,
                                    line_style,
                                    line_width,
                                    color);
#else
    GdkJoinStyle join_style;
    GdkCapStyle cap_style;
    gtk_plot_data_get_line_attributes   (pld,
                    &line_style, &cap_style, &join_style, &line_width, &bogus);

    gtk_plot_data_set_line_attributes(pld,
                                    line_style,
                                    cap_style,
                                    join_style,
                                    line_width,
                                    color);
#endif
  }
  
  for (node=chart->strip_list; node; node=node->next)
  {
    GtkStripChartCursor *curs;
    GtkStripChartStrip *strip = node->data;

    curs = GTK_STRIP_CHART_CURSOR(gtk_strip_chart_cursor_new());
    gtk_strip_chart_strip_add_cursor (strip, curs);
    gtk_strip_chart_cursor_set_add_cursor (cs, curs);

  }
  gtk_strip_chart_cursor_set_set_pos (cs, pos);
  gtk_strip_chart_cursor_set_set_height (cs, chart->range_min,
                                             chart->range_max);
  gtk_strip_chart_cursor_set_set_dragable (cs, dragable);

  chart->cursor_set_list = g_list_prepend (chart->cursor_set_list, cs);
  return cs;
}

/* ========================================================== */

void
gtk_strip_chart_chart_redraw (GtkStripChartChart *chart)
{
  GList *node;

  g_return_if_fail (chart != NULL);
  g_return_if_fail (GTK_IS_STRIP_CHART_CHART (chart));

  for (node=chart->strip_list; node; node=node->next)
  {
    GtkStripChartStrip *strip = node->data;
    graph_data (chart, strip);
  }

  /* place_strips is a bit overkill, but needed for things
   * like setting the title, etc. */
  place_strips (chart);
  gtk_plot_canvas_paint(&chart->plot_canvas);
  gtk_plot_canvas_refresh(&chart->plot_canvas);
}

/* ======================== END OF FILE ===================== */
