/***************************************************************************
 *
 * test_lkm.c
 *
 * LOMAC - Low Water-Mark Mandatory Access Control for Linux 
 * Copyright (C) 1999, 2000 NAI Labs
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU General Public
 * License as published by the Free Software Foundation.  This program
 * is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.  You should have received a copy of the
 * GNU General Public License along with this program; if not, write
 * to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 *
 *
 * Test lkm for kernel interface path-handling code.
 *
 ***************************************************************************/

#include <linux/module.h>  /* for module macros */
#include <asm/uaccess.h>    /* for verify_area() */

#include "kernel_interface.h"
#include "lomac_log.h"

/* Commands for ioctl() interface. */
#define GET_PID_LEVEL_IOCTL_CMD  0
#define GET_FILE_LEVEL_IOCTL_CMD 1


int do_ioctl( struct inode *, struct file *, unsigned int, unsigned long );
void init_driver( void );

/* This is the file operation jump table for the LOMAC *
 * character device.  We'll be using ioctl, only.      */
struct file_operations lomac_fops = { NULL, NULL, NULL, NULL, NULL,
				      do_ioctl, NULL, NULL, NULL, NULL,
				      NULL, NULL, NULL };

int lomac_major;           /* holds major number of lomac device */

/* do_ioctl()
 *
 * in:     p_inode_ignored - ignored
 *         p_file_ignored  - ignored
 *         command - GET_PID_LEVEL_IOCTL_CMD  - return level of pid in `arg'.
 *                   GET_FILE_LEVEL_IOCTL_CMD - return level of file named
 *                                              by the string `arg'.
 *         arg     - if `command' is GET_PID_LEVEL_IOCTL_CMD, then `arg' is
 *                   the pid of the process whose level we want to query.
 *                   if `command' is GET_FILE_LEVEL_IOCTL_CMD, then `cmd'
 *                   points to the name of the file whose level we want
 *                   to query, stored as a null-terminated string.
 * out:    nothing
 * return: the level of the process or file indicated by `command' and `arg',
 *         or -1 if no such process or file exists.  Also returns -1 on
 *         a bad `cmd' parameter.
 *
 *     User-space utility programs may query the state of the LOMAC LKM
 * through this ioctl() interface.
 *
 */

int do_ioctl( struct inode *p_inode_ignored, struct file *p_file_ignored,
	      unsigned int command, unsigned long arg ) {

  unsigned int filename_length;       /* length of file name from userspace */
  int ret_val;

  switch( command ) {
  case GET_PID_LEVEL_IOCTL_CMD:          /* look up process level */
    return( -1 );
    break;

  case GET_FILE_LEVEL_IOCTL_CMD:         /* look up file level */

    /* When a file level is queried, we expect `arg' to point to a     *
     * variable-sized block of user-space memory organized as follows: *
     * First, an unsigned int containing the size of a following       *
     * buffer, and second, the buffer itself.  The buffer is used to   *
     * contain the absolute path to the file in question, as a         *
     * null-terminated string.                                         */

    /* First, verify that we can read the string-length uint, *
     * and copy it into kernel memory as `filename_length'.   */
    if( verify_area( VERIFY_READ, (void *)arg, sizeof( unsigned int ) ) ) {
      return( -1 );
    }
    copy_from_user( &filename_length, (void *)arg, sizeof( int ) );
    arg = (unsigned int)((unsigned int *)arg + 1);  /* skip over size uint */
    /* ## testing here! */
    if( 0 <= ( ret_val = predict_object_canabspath( (char *)arg, log_s, 
						    PATH_MAX ) ) ) {
      printk( "%s\n", log_s );
    }
    return( ret_val );
    break;
    
  default:                               /* error - bad cmd value */

    return( -1 );

  } /* switch( command ) */

  return( -1 );

} /* do_ioctl() */



/* init_module()
 *
 * in:     nothing
 * out:    nothing
 * return: value      condition
 *         -----      ---------
 *         0          success
 *
 */

int init_module( void ) {

  /* initialize driver (user-space info interface) */
  if( 0 > ( lomac_major = register_chrdev( 0, "lomac", &lomac_fops ) ) ) {
    PANIC( "LOMAC/K: no major device number available for lomac" );
  }

  /* Print a victory message to the console, indicating a successful load. */
  printk( "loaded test LKM\n" ); 
  return( 0 );

} /* init_module() */



/* cleanup_module()
 * 
 * in:     nothing
 * out:    nothing
 * return: nothing
 *
 * This function is called by the kernel in response to a request to unload
 * the lomac lkm.  Someday, this function should unpatch the system call
 * vector so the kernel can keep running once the LKM is gone, but this
 * requires some method to rescue processes that are currently blocked
 * in lomac-intercepted system calls.
 */

void cleanup_module( void ) {

  unregister_chrdev( lomac_major, "lomac" );
  printk( "unloaded test LKM\n" );

} /* cleanup_module() */





