/***************************************************************************
 *
 * kernel_path.c
 *
 * LOMAC - Low Water-Mark Mandatory Access Control for Linux 
 * Copyright (C) 1999, 2000 NAI Labs
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU General Public
 * License as published by the Free Software Foundation.  This program
 * is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.  You should have received a copy of the
 * GNU General Public License along with this program; if not, write
 * to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 *
 * This file contains routines that use the kernel's namei() function
 * to produce absolute canonical-form paths.  These routines are placed
 * here, rather than in kernel_interface.c, primarily to make it easier
 * to test them separately from the rest of LOMAC.
 *
 ***************************************************************************/

#include "kernel_interface.h"
#include "lomac_log.h"

/* chop_last_component()
 *
 * in:     path_s     - a path
 *         null_index - index of '\0' that terminates `path_s'
 * out:    path_s     - `path_s', shortened by one component
 * return: new index of null that terminates `path_s'.
 *
 * Removes the last component of `path_s' by overwriting '/' with '\0'.
 * Examples:
 *
 * `path_s' in                 `path_s' out
 * -----------     ---->       ------------
 * /foo/bar/                   /foo
 * /foo/bar                    /foo
 * foo/bar                     foo
 * /foo                        /
 * /foo//bar                   /foo
 * /foo/../bar                 /foo/..
 * /foo/..                     /foo
 *
 */

int chop_last_component( char *path_s, int null_index ) {

  
} /* chop_last_component() */



/* predict_object_canabspath()
 *
 * in:     path_s       - path to object that may or may not yet exist;
 *                        path may or may not be canonical or absolute;
 *                        path may or may not be valid.
 *         len          - length of `canabspath_s' buffer.
 * out:    canabspath_s - absolute path in canonical form corresponding
 *                        to `path_s'.
 * return: 0 on success, else kernel namei()'s error codes.
 *
 * This function is used to predict the absolute, canonical-form
 * pathnames of objects that are about to be opened, based on the
 * not-necessarily absolute, canonical-form, or valid paths supplied
 * by the calling process.  Due to the nature of the operating system,
 * given a particular `path_s', there are three cases:
 *
 * 1)  `path_s' describes an existing object in the file system.
 *     In this case, we convert `path_s' into absolute, canonical form,
 *     and output this new canonical-form path out in `canabspath_s'.
 * 2)  `path_s' does not describe an existing object in the file system,
 *     but `path_s' with its last component removed describes a directory.
 *     This case may occur when the calling process wants to create a
 *     new filesystem object in that directory.  In this case, we
 *     convert the directory's path into absolute canonical form, append
 *     the last component of `path_s', and output this in `canabspath_s'.
 * 3)  `path_s' does not describe any object in the filesystem, and when
 *     its last component is removed, it still does not describe a
 *     directory in the filesystem.  In this case, we return an error
 *     code, following the normal error behavior of the open (etc.) 
 *     system calls.
 *
 */

int predict_object_canabspath( char *path_s, char *canabspath_s, int len ) {

  struct dentry *p_dentry;       /* points to dcache entries (inodes) */
  int null_index;                /* index of '\0' that terminates paths */
  int last_char_index;           /* index of last non-"/" char in `path_s' */
  int ret_val;                   /* holds error return codes */
  
#ifdef PARANOID
  if( !path_s ) {
    PANIC( "LOMAC/K: predict_object_canabspath() given null path_s.\n" );
  }
  if( !canabspath_s ) {
    PANIC( "LOMAC/K: predict_object_canabspath() given null canabspath_s.\n" );
  }
#endif

  /* Check for case 1: `path_s' refers to a valid filesystem object. */
  p_dentry = namei( path_s );
  ret_val = PTR_ERR( p_dentry );
  if( !IS_ERR( p_dentry ) ) {
    
    /* `path_s' does indeed refer to a valid object.  Place the         *
     * absolute canonical form version of `path_s' into `canabspath_s'. */
#ifdef PARANOID
    ret_val = 
#endif
      get_object_canabspath( p_dentry, canabspath_s, len );
#ifdef PARANOID
    if( ret_val ) {
      PANIC( "LOMAC/K: predict_object_canabspath failed to convert path.\n" );
    }
#endif
    dput( p_dentry );  /* we're done with `p_dentry', thank you. */
    return( 0 );       /* OK! */
  }

  /* If we reach here, we do not have a case 1 situation.  Check for  *
   * case 2 by removing last component of `path_s' and trying namei() *
   * again.                                                           */

  /* Copy input `path_s' into scratch space so we can mangle it. */
  STRNCPY( log_s, path_s, MAX_PATH_LEN );
  null_index = STRNLEN( log_s, MAX_PATH_LEN );

  /* First, remove any trailing slashes */
  for( null_index--;  /* start with char to the left of null terminator */
       ( null_index >= 0 && path_s[ null_index ] == '/' ); 
       null_index-- ) {
    path_s[ null_index ] = '\0';
  }

  /* save index of last non-"/" char so we can eliminate trailing *
   * slashes easily later on.                                     */
  last_char_index = null_index;

  /* Next, remove any non-"/" chars until we find another "/" */
  for( ;  /* start with the same char previous loop ended on */
       ( null_index >= 0 && path_s[ null_index ] != '/' ); 
       null_index-- ) {
    path_s[ null_index ] = '\0';
  }

  /* Last component removed.  Now try namei() again. */
  p_dentry = namei( path_s );
  ret_val = PTR_ERR( p_dentry );
  if( !IS_ERR( p_dentry ) ) {
  
    /* The path in the `log_s' scratch space does indeed refer *
     * to a valid filesystem object.  Is it a directory?       */
    if( S_ISDIR( p_dentry->d_inode->i_mode ) ) {

      /* Yes, the path does refer to a directory.  Convert it into   *
       * absolute, canonical form, append the last component of      *
       * `path_s' to it, and copy it into `canabspath_s' for output. */

      /* Start by putting canabspath for `p_dentry' into log_s. */
      STRNCPY( log_s, d_path( p_dentry, log_s, MAX_PATH_LEN ), MAX_PATH_LEN ); 

      /* Put a "/" between `p_dentry's path and the last component *
       * from path_s that we'll be adding in the next step.        */
      STRNCAT( log_s, "/", 1 );
 
      /* Append last component of `path_s'.  We're counting on      *
       * `null_index' to point to (the proper index -1) in `path_s' *
       * after coming out of the for loops, above.                  */
      STRNCAT( log_s, path_s[ null_index + 1 ], 
	       (last_char_index - null_index ) );

      printk( "%s\n", log_s );   /* for debugging! ## */

      ret_val = 0;   /* OK! */

    } else {

      /* Although the shortened path in `log_s' is valid, it does not *
       * describe a directory.  Return the appropriate error code.    */

      ret_val = -ENOTDIR;

    } /* if/else `p_dentry' is a directory */

    dput( p_dentry );                          /* all done, thanks! */
    return( ret_val );

  } /* if the shortened path in `log_s' refers to a valid FS object */


  /* If we reach here, we have a case 3 situation.  Return error code. */
  return( ret_val );  /* set by namei() */

} /* predict_object_canabspath() */










  do {
    /* attempt to find an existing file for `path_s'. */
    p_dentry = namei( log_s );

    /* If `path_s' corresponds to an existing file, leave namei() loop. */
    if( !ERR( p_dentry ) ) {
      break;
    }

    /* `path_s' does not describe an existing file.  Shorten `path_s' *
     * by one component, and try namei() again.                       */
    null_index = chop_last_component( log_s, null_index );

  } while( log_s[ 0 ] != '\0' );      /* repeat until exhaust `log_s' */


  /* For now, just print the existing prefix of `path_s' and be done. */
  log_start();
  log_append_string( log_s );
  log_append_string( "\n" );
  log_print();

  
} /* predict_object_canabspath() */


