#########################################################################
#
#
# LOMAC - Low Water-Mark Mandatory Access Control for Linux 
# Copyright (C) 1999 TIS Labs at Network Associates, Inc.
# Copyright (C) 2000 - 2001 NAI Labs
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of version 2 of the GNU General Public
# License as published by the Free Software Foundation.  This program
# is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or
# FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
# License for more details.  You should have received a copy of the
# GNU General Public License along with this program; if not, write
# to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.
#
# This module contains functions which allow Perl scripts to
# communicate with the LOMAC LKM via ioctl().
#
# Prior to LOMAC v1.0.4, these functions were implemented as
# individual Perl scripts.  The lls and lps invoked these scripts via
# system() - an arrangement which involved the considerable overhead
# of forking and exec'ing the Perl interpreter.  As a result, lls and
# lps ran very slowly.  The LOMAC v1.0.4 release introduced these
# functions as a module, allowing lls and lps to invoke them via
# function calls.  This arrangement has much less overhead, improving
# the speed of lls and lps immensely.  This module-implementation
# solution is due to John Thiltges <jthiltg1@bigred.unl.edu>.
#
# Tim Fraser <tfraser@tislabs.com> : original level and linfo scripts
# John Thiltges <jthiltg1@bigred.unl.edu> : combined scripts into lomac.pm
#
#########################################################################

package lomac;

use Exporter   ();
@ISA         = qw(Exporter lomac);
@EXPORT      = qw(level linfo);

########################################################################
#
# level - a perl script to determine the LOMAC level associated with
#         a specified process or file.  Uses the /dev/lomac pseudo-
#         device to retrieve this information from the LOMAC loadable
#         kernel module (LKM).
#
# usage:
# level -p <pid>
# level -f <pathname>
#
########################################################################

sub level ($$) {
my $mode = shift;
my $input = shift;
my $path;
my @ioctlarg;
my $ret_val;

#
# Step 4 - open "/dev/lomac" and make the appropriate ioctl.
# 

# Open /dev/lomac, the pseudodevice that allows us to query the LOMAC LKM.
open( LOMAC, "/dev/lomac" ) || die "Couldn't open /dev/lomac";

if( $mode eq "-p" ) {

    #
    # We use ioctl() through "/dev/lomac" to query the LOMAC LKM.
    # We have to do the scalar() and + 0 thing to get Perl to pass
    # by value rather than by reference.
    #

	$ret_val = ioctl( LOMAC, 0, ( scalar( $input ) + 0 ) ) ||
	($ret_val = -1);  # need this "or" when ioctl returns undefined

} elsif( $mode eq "-f" ) {

	$path = $input;
    $ioctlarg[ 0 ] = length( $path ) + 1;   # add 1 for null terminator.
    $ioctlarg[ 1 ] = $path;
    $ret_val = ioctl( LOMAC, 1, pack( "Ia*", @ioctlarg ) ) || ($ret_val = -1);

} else {
    print "USAGE: level -p <pid>\n       level -f <pathname>\n";
    return( -1 );
}

close LOMAC;

#
# Step 5 - report results.
#

if( $ret_val < 0 ) {
    print "Error: ", $!, "\n"; # prints error string corresponding to errno
    return( -1 );
}

return( $ret_val );
}


########################################################################
#
# linfo - the lomac info program.  The catch-all LOMAC utility.
#         Currently, linfo can be used only to determine if the
#         LOMAC LKM is loaded and the /dev/lomac device exists.
#
# USAGE:
# lomac -c
#
# OPTIONS:
# -c  checks to make sure the LOMAC LKM is loaded and /dev/lomac exists.
# 
#
# RETURNS:
# value       condition
# -----       ---------
#  -1         Incorrect command line parameters.
#   0         The LOMAC LKM is loaded and /dev/lomac exists.
#   1         The LOMAC LKM is not loaded.
#   2         /dev/lomac does not exist or is not a character device.  
#   3         /dev/lomac has incorrect device major number.
#
########################################################################

sub linfo ($) {
my $input = shift;
my $lomac_lkm_loaded_flag;
my $ignore;
my $lomac_major;
my @device_line_columns;
my $rdev;

#
# Step 1 - Process command-line arguments
#

# Bomb if we didn't get the proper number of arguments
if( not defined $input || $input ne "-c" ) {
    print( STDERR "USAGE: linfo -c\n" );
    return( -1 );
}

#
# Step 2 - Make sure the LOMAC LKM is loaded
#

# Pessimistically assume the LOMAC LKM is not loaded, and clear the
# $lomac_lkm_loaded_flag flag to false ("0"), accordingly

$lomac_lkm_loaded_flag = "0";

# Make <MODULES> a file handle to "/proc/modules", which is the kernel's
# list of currently loaded LKMs.
open( MODULES, "/proc/modules" ) || die "Couldn't open /proc/modules : $!";

# This while loop makes $_ equal each line in the <MODULES> file,
# on after another, until terminating at the end-of-file.
while( <MODULES> ) {

    # Seach for the name of the LOMAC LKM ("lomac_mod") in the current
    # line of the <MODULES> file, which is contained in $_.
    if( index( $_, "lomac_mod" ) >= 0 ) {

	# The LOMAC LKM is loaded!  Set the $lomac_lkm_loaded_flag
	# flag to true ("1") and break out of the while loop.
	$lomac_lkm_loaded_flag = "1";
	last;     # last is like break in C.

    } # if

} # while
close MODULES;

# Terminate if the LOMAC LKM is not loaded
unless( $lomac_lkm_loaded_flag ) {
    print STDERR "the lomac loadable kernel module is not loaded.\n";
    return( 1 );
}


#
# Step 3 - Verify that "/dev/lomac" exists and is a character special file.
#

unless( -c "/dev/lomac" ) {
    print STDERR "/dev/lomac character device does not exist.\n";
    return( 2 );
}


#
# Step 4 - Verify that "/dev/lomac" has the proper major number for the
#          lomac driver.
#

# pessimistically make $lomac_major = 0
$lomac_major = 0;

# Make <DEVICES> a file handle to "/proc/devices", which is the kernel's
# list of device major numbers.
open( DEVICES, "/proc/devices" ) || die "Couldn't open /proc/devices : $!";

# This while loop makes $_ equal each line in the <DEVICES> file,
# on after another, until terminating at the end-of-file.
while( <DEVICES> ) {

    # Seach for the name of the LOMAC device ("lomac") in the current
    # line of the <DEVICES> file, which is contained in $_.
    if( index( $_, "lomac" ) >= 0 ) {

	# `$_' contains the /proc/devices line corresponding to the
	# lomac device.  Extract the major number from this line
	# (it's in the first space-delimited column, possibly with
	# leading spaces) and put it in $lomac_major
	@device_line_columns = split( / +/, $_ );
	$lomac_major = $device_line_columns[ 0 ];
	last;     # last is like break in C.

    } # if

} # while
close DEVICES;

# Get the major number of the actual /dev/lomac device file.
($ignore, $ignore, $ignore, $ignore, $ignore, $ignore, $rdev, $ignore,
 $ignore, $ignore, $ignore, $ignore, $ignore) = stat "/dev/lomac";
$rdev = $rdev >> 8;

# make sure that the major number of /dev/lomac is the correct major
# number as listed in /proc/devices.

if( $lomac_major != $rdev ) {
    print STDERR sprintf( "/dev/lomac has incorrect major number (%u should be %u)\n", $rdev, $lomac_major );
    return( 3 );
}

#
# Everything's OK!
#

return( 0 );
}

1;
