/***************************************************************************
 *
 * LOMAC - Low Water-Mark Mandatory Access Control for Linux 
 * Copyright (C) 1999, 2000 NAI Labs
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU General Public
 * License as published by the Free Software Foundation.  This program
 * is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.  You should have received a copy of the
 * GNU General Public License along with this program; if not, write
 * to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * lup - LOMAC "trusted" file upgrader.
 *
 ***************************************************************************/

#include <sys/mman.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <syslog.h>

#define USAGE_STRING "lup <sourcefile>"
#define DEST_FILE "/root/upgraded"

int main( int argc, char *argv[] ) {

  int source_fd;        /* source file descriptor */
  struct stat stats;    /* source file stats */
  char *memory_region;  /* start of region to which source file is mmapped */
  int dest_fd;          /* destination file descriptor */

  /* check command-line arguments */
  if( argc != 2 ) {
    fprintf( stderr, "USAGE: %s\n", USAGE_STRING );
    exit( -1 );
  }

  /* open and mmap source file named by argv[ 1 ] */
  if( -1 == ( source_fd = open( argv[ 1 ], O_RDONLY ) ) ) {
    perror( "can't open source file" );
    exit( -1 );
  }
  if( -1 == fstat( source_fd, &stats ) ) {
    perror( "can't stat source file" );
    close( source_fd );
    exit( -1 );
  }
  memory_region = NULL;
  if( -1 == (int)( memory_region = mmap( NULL, stats.st_size, PROT_READ,
					 MAP_PRIVATE, source_fd, 0 ) ) ) {
    perror( "can't mmap source file" );
    close( source_fd );
    exit( -1 );
  }
  
  /* copy mmapped data out to destination file */
  if( -1 == ( dest_fd = creat( DEST_FILE, ( S_IRUSR | S_IWUSR ) ) ) ) {
    perror( "can't create destination file" );
    munmap( memory_region, stats.st_size );
    close( source_fd );
    exit( -1 );
  }
  if( stats.st_size != write( dest_fd, memory_region, stats.st_size ) ) {
    perror( "can't write to destination file" );
    munmap( memory_region, stats.st_size );
    close( source_fd );
    close( dest_fd );
    exit( -1 );
  }
   
  munmap( memory_region, stats.st_size );
  close( source_fd );
  close( dest_fd );

  /* log the upgrade */
  openlog( "lup", 0, LOG_AUTHPRIV );
  syslog( LOG_NOTICE, "%s(%d) upgraded %s\n", 
	  ( getlogin() ? getlogin() : "<unknown>" ),
	  getuid(), argv[ 1 ] );
  closelog();

  return( 0 );

} /* main */
