/**************************************************************************
 *
 * lomac_plm.c
 *
 * LOMAC - Low Water-Mark Mandatory Access Control for Linux 
 * Copyright (C) 1999, 2000 NAI Labs
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU General Public
 * License as published by the Free Software Foundation.  This program
 * is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.  You should have received a copy of the
 * GNU General Public License along with this program; if not, write
 * to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * This is the implementation of the Path-Level Map (PLM).  It
 * provides a mapping between canonical absolute pathnames and levels.
 *
 *
 **************************************************************************/

#include "kernel_interface.h"
#include "lomac_log.h"
#include "lomac_plm.h"

#define IS_PATH_PREFIX(plm_path_s, user_path_s) \
( (user_path_s) == STRSTR( (user_path_s), (plm_path_s) ) )


typedef struct plm_rule {
  level_t level;
  int     flags;
  char   *path_s;
} plm_rule_t;

#define PLM_NOFLAGS 0x00 /* rule applies to this node and its children */
#define PLM_CHILDOF 0x01 /* rule applies to node's children, not the node */

/* This is LOMAC's default policy.                                  *
 * If you would like to run without TRUST, put /var/log/messages at *
 * a low level, since syslogd must write it after reading /dev/log. */
#define NUM_PLM_RULES 26
plm_rule_t plm[ NUM_PLM_RULES ] = {
  { LOMAC_LOWEST_LEVEL,  PLM_NOFLAGS, "/var/run/ftp.pids-all" },
  { LOMAC_HIGHEST_LEVEL, PLM_NOFLAGS, "/var/log/messages" },
  { LOMAC_HIGHEST_LEVEL, PLM_NOFLAGS, "/var/log/lastlog" },
  { LOMAC_HIGHEST_LEVEL, PLM_NOFLAGS, "/var/log/secure" },
  { LOMAC_LOWEST_LEVEL,  PLM_NOFLAGS, "/dev/printer" },  /* sadly deletable */
  { LOMAC_HIGHEST_LEVEL, PLM_NOFLAGS, "/var/lib/nfs" },
  { LOMAC_HIGHEST_LEVEL, PLM_NOFLAGS, "/var/lib/rpm" },
  { LOMAC_HIGHEST_LEVEL, PLM_NOFLAGS, "/home/httpd" },
  { LOMAC_HIGHEST_LEVEL, PLM_NOFLAGS, "/mnt/floppy" }, /* floppy is high */
  { LOMAC_HIGHEST_LEVEL, PLM_NOFLAGS, "/home/samba" },
  { LOMAC_HIGHEST_LEVEL, PLM_NOFLAGS, "/mnt/cdrom" },  /* cdrom is high */
  { LOMAC_LOWEST_LEVEL,  PLM_CHILDOF, "/usr/local" }, /* dwnlded progs here */
  { LOMAC_HIGHEST_LEVEL, PLM_NOFLAGS, "/home/ftp" },
  { LOMAC_LOWEST_LEVEL,  PLM_NOFLAGS, "/dev/log" },      /* sadly deletable */
  { LOMAC_LOWEST_LEVEL,  PLM_CHILDOF, "/usr/src" },
  { LOMAC_LOWEST_LEVEL,  PLM_CHILDOF, "/usr/tmp" },
  { LOMAC_LOWEST_LEVEL,  PLM_CHILDOF, "/var/lib" },
  { LOMAC_HIGHEST_LEVEL, PLM_NOFLAGS, "/var/lib" },
  { LOMAC_LOWEST_LEVEL,  PLM_CHILDOF, "/var/log" },
  { LOMAC_HIGHEST_LEVEL, PLM_NOFLAGS, "/var/log" },
  { LOMAC_HIGHEST_LEVEL, PLM_NOFLAGS, "/var/run" },
  { LOMAC_LOWEST_LEVEL,  PLM_CHILDOF, "/home" }, 
  { LOMAC_LOWEST_LEVEL,  PLM_CHILDOF, "/mnt" },  /* all nfs mounts are low */
  { LOMAC_LOWEST_LEVEL,  PLM_CHILDOF, "/tmp" },
  { LOMAC_LOWEST_LEVEL,  PLM_CHILDOF, "/var" },
  { LOMAC_HIGHEST_LEVEL, PLM_NOFLAGS, "/" }
};


#define PATH_UNRELATED 0
#define PATH_PREFIX    1
#define PATH_MATCH     2
int compare_paths( const char *rulepath_s, const char *querypath_s );


#ifdef PARANOID
/* check_plm_constraints()
 *
 * in:     nothing
 * out:    nothing
 * return: nothing
 *
 * The PLM will operate correctly only if the `plm' array is properly
 * set up.  A properly set up `plm' array obeys the following
 * constraints:
 *
 * 1.  The rule entries in the array are sorted according to the length of
 *     their `path_s' fields, with longer-stringed rules coming first.
 * 2.  If two rules have the same `path_s' vaule, the first rule of the
 *     pair will have its "CHILDOF" flag set, and the second won't.  No
 *     more than two rules can share a pathname.
 * 3.  The last rule in the array must be for `path_s' "/" with no
 *     flags set.
 * 4.  All rule paths must be absolute in canonical form.
 *
 * In addition, we add the following constraint for the sake of
 * security (rather than for the proper operation of the PLM):
 *
 * *. The PLM shall be configured so that no child in the filesystem
 *    path hierarchy has a higher level than its parent.
 *
 * This last constraint prevents situation where /foo/bar has level 2,
 * and /foo has level 1, allowing a level-1 process to mv "foo" to "baz",
 * or to perform some other annoying operation that makes life difficult
 * for higher-level users of "bar".
 *
 * This last constraint is not yet implemented.  ##
 * Note that the constraint algorithm must remove the last component of
 * each rule and then get the level of that truncated path to do the test.
 *
 */

void check_plm_constraints() {

  int plm_index1, plm_index2;       /* indicies into PLM rule array */
  int found_second_rule_flag;       /* set when two rules have same path */

  /* Check constraint #1 (sorting of rule entries) *
   * and #4 (absolute canonical form).             */
  if( !is_canabspath( plm[ 0 ].path_s ) ) {
    PANIC( "LOMAC: non-absolute-canonical path in PLM rule.\n" );
  }
  for( plm_index1 = 1; plm_index1 < NUM_PLM_RULES; plm_index1++ ) {

    /* check constraint #1 (ordering) */
    if( STRNLEN( plm[ plm_index1 - 1 ].path_s, MAX_PATH_LEN ) <
	STRNLEN( plm[ plm_index1 ].path_s, MAX_PATH_LEN ) ) {
      PANIC( "LOMAC: PLM is not sorted properly.\n" );
    }

    /* check constraint #4 (absolute canonical form) */
    if( !is_canabspath( plm[ plm_index1 ].path_s ) ) {
      PANIC( "LOMAC: non-absolute-canonical path in PLM rule.\n" );
    }

  } /* for all rules starting with the second rule */

  /* Check constraint #2: sorting of rules with identical paths. */
  for( plm_index1 = 0; plm_index1 < NUM_PLM_RULES; plm_index1++ ) {
    
    /* haven't found match for `plm[ plm_index1 ].path_s' yet, clear flag. */
    found_second_rule_flag = 0;   

    /* search following rules for a match for `plm[ plm_index1 ].path_s' */
    for( plm_index2 = plm_index1 + 1; plm_index2 < NUM_PLM_RULES; 
	 plm_index2++ ) {
 
      if( !STRCMP( plm[ plm_index1 ].path_s, plm[ plm_index2 ].path_s ) ) {

	/* We've found a match.  If this is not the first match, there *
	 * must be more than two rules with the same path - an error.  */
	if( found_second_rule_flag ) {
	  PANIC( "LOMAC: more than two PLM rules share the same path.\n" );
	} else {

	  /* OK, this is the second rule, set flag and perform tests. */
	  found_second_rule_flag = 1;

	  if( !( plm[ plm_index1 ].flags & PLM_CHILDOF ) ) {
	    PANIC( "LOMAC: missing CHILDOF flag in PLM.\n" );
	  }
	  if( plm[ plm_index2 ].flags != PLM_NOFLAGS )  {
	    PANIC( "LOMAC: illegal CHILDOF flag in PLM.\n" );
	  }

	} /* if we've found the second rule of a pair. */

      } /* if two rules have same path */

    } /* for all possible second rules */

  } /* for all rules in PLM */


  /* Check constraint #3 (last rule is "/") */
  if( STRCMP( plm[ NUM_PLM_RULES - 1 ].path_s, "/" ) ) {
    PANIC( "LOMAC: last PLM rule not \"/\".\n" );
  }

  if( plm[ NUM_PLM_RULES - 1 ].flags != PLM_NOFLAGS ) {
    PANIC( "LOMAC: last PLM rule has illegal flags.\n" );
  }

  /* Check last security constraint ( child level <= parent level ). */
  /* not implemented yet! ## */

} /* check_plm_constraints() */

#endif


/* get_pathname_level()
 *
 * in:     path_s  - pathname whose level we wish to know.  This must
 *                   be an absolute, canonical-form path.
 * out:    p_level - will contain level of `path_s'
 * return: nothing
 *
 * This function takes a path `path_s' and searches through the rules
 * in the `plm' array looking for the rule that applies to `path_s'.
 * The decision whether or not a given rule applies to `path_s' is
 * made by compare_paths() according to the following truth table:
 *
 * result of
 * compare_               rule flags
 * paths()         PLM_NOFLAGS  PLM_CHILDOF
 *
 * PATH_UNRELATED       0           0
 * PATH_PREFIX          1           1
 * PATH_MATCH           1           0
 *
 * where 1 indicates that the rule applies.
 *
 */

void get_pathname_level( const char *path_s, level_t *p_level ) {

  int plm_index;         /* index into PLM rule array */

#ifdef PARANOID
  if( !is_canabspath( path_s ) ) {
    PANIC( "LOMAC: PLM queried with non-absolute-canonical path.\n" );
  }
#endif

  /* Iterate through the PLM array searching for the rule that applies *
   * to `path_s'.                                                      */

  for( plm_index = 0; plm_index < NUM_PLM_RULES; plm_index++ ) {

    switch( compare_paths( plm[ plm_index ].path_s, path_s ) ) {

    case PATH_MATCH:
      if( plm[ plm_index ].flags & PLM_CHILDOF ) {
	break;   /* this rule does not apply */
      }
      /* else it does apply, slide on into the next case. */
    case PATH_PREFIX:
      /* This rule applies, report the level indicated by the rule. */
      *p_level = plm[ plm_index ].level;
      return;

    case PATH_UNRELATED:
    default:
      /* This rule does not apply. */

    } /* switch on compare_paths() */

  } /* for each rule in the PLM */

  /* If we arrive here, no PLM rule matched.  This isn't supposed *
   * to happen, since the PLM is always supposed to end with a    *
   * rule for "/", and "/" should match all paths.  Hmmm...       */
  log_start();
  log_append_string( "LOMAC: PLM overrun on " );
  log_append_string( path_s );
  log_append_string( "\n" );
  log_print();
  PANIC( "LOMAC: PLM overrun; path in log" );

} /* get_pathname_level() */ 


/* is_canabspath()
 *
 * in:     path_s - path to examine
 * out:    nothing
 * return: value    condition
 *         -----    ---------
 *           0      `path_s' is not an absolute path in canonical form.
 *           1      `path_s' is an absolute path in canonical form.
 *
 * This function is a predicate for deciding whether or not a given path
 * is absolute and is in canonical form.
 *
 */

int is_canabspath( const char *path_s ) {

  int index;                /* index used to traverse `path_s'. */

  if( path_s[ 0 ] != '/' ) {
    return( 0 );  /* `path_s' is non-absolute */
  }

  /* Check `path_s' for further bad conditions.  Short `path_s' *
   * strings are handled first, and then the loop takes care of *
   * `path_s' strings longer than 3 non-null characters.        */

  if( path_s[ 1 ] == '\0' ) {

    /* string is "/", done! */

  } else if( path_s[ 2 ] == '\0' ) {

    /* No "//" and "/.", please! */
    if( path_s[ 1 ] == '/' ) {
      return( 0 );   /* `path_s' is "//" */
    }
    if( path_s[ 1 ] == '.' ) {
      return( 0 );   /* `path_s' is "/." */
    }
    
  } else {
    
    for( index = 3; index < MAX_PATH_LEN; index++ ) {
      
      /* Check for "/..\0" and "/../" in `path_s'. */
      if( ( path_s[ index - 3 ] == '/' ) &&
	  ( path_s[ index - 2 ] == '.' ) &&
	  ( path_s[ index - 1 ] == '.' ) &&
	  ( ( path_s[ index ] == '/' ) ||
	    ( path_s[ index ] == '\0' ) ) ) {
	return( 0 );  /* `path_s' contains ".." as a component */
      }

      /* Check for "/./ and "/.\0" in `path_s' */
      if( ( path_s[ index - 2 ] == '/' ) &&
	  ( path_s[ index - 1 ] == '.' ) &&
	  ( ( path_s[ index ] == '/' ) ||
	    ( path_s[ index ] == '\0' ) ) ) {
	return( 0 );   /* `path_s' contains "." as a component */
      }
      
      /* Check for "//" in `path_s' */
      if( ( path_s[ index - 1 ] == '/' ) &&
	  ( path_s[ index ] == '/' ) ) {
	return( 0 );   /* `path_s' contains "//" */
      }
      
      if( path_s[ index ] == '\0' ) {
	/* We've reached the end of path_s, just one last check... *
	 * make sure the string didn't end in '/'.                 */
	if( path_s[ index - 1 ] == '/' ) {
	  return( 0 );  /* `path_s' ends with "/" */
	}
	break;
      }
      
    } /* for all but the first 3 chars of `path_s' */

    if( index >= MAX_PATH_LEN ) {
      return( 0 );  /* `path_s' is longer than MAX_PATH_LEN */
    }

  } /* else `path_s' is longer than 2 chars. */

  return( 1 );   /* `path_s' is absoulte and in canonical form. */

} /* is canabspath() */






/* -------------------- local utility functions ---------------------- */

/* compare_paths()
 *
 * in:     rulepath_s  - path from PLM rulebase
 *         querypath_s - path that we want to compare `rulepath_s' to
 * out:    nothing
 * return: value             condition
 *         --------------    ---------
 *         PATH_UNRELATED    `rulepath_s' is unrelated to `querypath_s'.
 *         PATH_PREFIX       `rulepath_s' is a prefix of `querypath_s'.
 *         PATH_MATCH        `rulepath_s' is the same string as `querypath_s'.
 *
 *     This function compares `rulepath_s' to `querypath_s'.
 *
 */

int compare_paths( const char *rulepath_s, const char *querypath_s ) {

  int index;         /* index used to traverse path strings */

#ifdef PARANOID
  if( !rulepath_s ) {
    PANIC( "LOMAC: null rulepath passed to compare_paths().\n" );
  }
  if( !querypath_s ) {
    PANIC( "LOMAC: null querypath passed to compare_paths().\n" );
  }
#endif

  for( index = 0; index < MAX_PATH_LEN; index++ ) {

    if( ( rulepath_s[ index ] == querypath_s[ index ] ) ) {

      /* If we've reached the end of both strings, return PATH_MATCH, *
       * otherwise do nothing and let the for loop carry us to the    *
       * next character.                                              */
      if( rulepath_s[ index ] == '\0' ) {
	return( PATH_MATCH );
      }

    } else {

      /* The strings do not match at `index'.  If we have reached the *
       * end of `rulepath_s', then return PATH_PREFIX.  Otherwise,    *
       * return PATH_UNRELATED.                                       */
      if( rulepath_s[ index ] == '\0' ) {
	return( PATH_PREFIX );
      } else {
	return( PATH_UNRELATED );
      }

    } /* if/else match at `index' */
      
  } /* for all chars in max-length path string */

#ifdef PARANOID
  PANIC( "LOMAC: compare_paths() found both rule and query paths too long\n" );
#endif
  return( PATH_UNRELATED );  /* shouldn't reach here... see panic above. */

} /* compare_paths() */
