/*************************************************************************
 *
 * lomac_mediate.c
 *
 * LOMAC - Low Water-Mark Mandatory Access Control for Linux 
 * Copyright (C) 1999, 2000 NAI Labs
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU General Public
 * License as published by the Free Software Foundation.  This program
 * is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.  You should have received a copy of the
 * GNU General Public License along with this program; if not, write
 * to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 *
 * This file contains functions that make access control decisions
 * concerning wether or not given system calls should be allowed
 * or denied.  This activity is called "mediation".  These functions
 * generally consider both the parameters passed to a system call
 * and the current internal state of LOMAC in the course of making
 * a decision.  However, they do not modify these parameters or
 * LOMAC's internal state.  Functions for modifying LOMAC's internal
 * state can be found in lomac_monitor.c.
 *
 * mediate_open() gets used to mediate not only open operations
 * (system calls), but other operations like bind and unlink, too.
 * Unfortunately, its log message output always says "open/creat" -
 * this is somewhat misleading.  mediate_open() needs an operation_s
 * parameter like mediate_is_high() has.  In fact, all of the
 * mediate_* functions need this kind of parameter.
 *
 *************************************************************************/

#include "kernel_interface.h"
#include "lomac_mediate.h"
#include "lomac_plm.h"
#include "lomac_log.h"

/* mediate_pgrp_change()
 *
 * in:     p_src_subject  - subject that requested a pgrp change
 *         p_dest_subject - subject `p_src_subject' wants to join
 * out:    nothing
 * return: value   condition
 *         -----   ---------
 *           0     caller should prevent pgrp change
 *           1     caller should allow pgrp change
 *
 *     Processes can move from one pgrp to another using the setpgrp
 * system call.  In LOMAC terms, this means that processes can move
 * from one subject to another.  This movement does no harm, provided
 * that processes never move from one subject to another of a higher
 * level.  The setpgrp wrapper function can use this function to
 * detect these harmful cases, so that it may prevent them.  Note
 * that the setpgrp system call also allows a process to create a
 * new pgrp.  This is always harmless - the wrapper function should
 * not bother calling this function in these harmless cases.
 *
 */

int mediate_pgrp_change( lomac_subject_t *p_src_subject,
			 lomac_subject_t *p_dest_subject ) {

  level_t src_level;     /* level of `p_src_subject' */
  level_t dest_level;    /* level of `p_dest_subject' */

  get_subject_level( p_src_subject, &src_level );
  get_subject_level( p_dest_subject, &dest_level );
  
  if( src_level < dest_level ) {

    if( verbose & VERBOSE_DEMOTE_DENY ) {
      log_start();
      log_append_string( "LOMAC: denied level-" );
      log_append_int( src_level );
      log_append_string( " proc " );
      log_append_subject_id( p_src_subject );
      log_append_string( "'s move to level-" );
      log_append_int( dest_level );
      log_append_string( " proc " );
      log_append_subject_id( p_dest_subject );
      log_append_string( "'s subject\n" );
      log_print();
    }

#ifdef NO_MEDIATION
    return( 1 );
#else
    return( 0 );
#endif
  }

  return( 1 );

} /* mediate_pgrp_change() */


/* mediate_open()
 * 
 * in:     p_subject - subject performing open on object named `path_s'
 *         path_s    - name of object.  We can't pass a lomac_object_t *
 *                     here because the object may not exist, and
 *                     possibly should never exist.
 * out:    nothing
 * return: value    condition
 *         -----    ---------
 *           0      Caller should prevent open
 *           1      Caller should permit open
 *
 *     This function allows LOMAC to mediate open and creat system
 * calls.  LOMAC needs to mediate these calls in two cases: (1) where
 * the open will truncate an existing file, and (2) where the open
 * will create a new file that did not exist before.  LOMAC must
 * mediate in case 1 because truncation is effectively a write.  This
 * version of LOMAC must mediate in case 2 because it determines the
 * level of objects solely by the output of the PLM.  The PLM
 * determines an object's level based on its pathname.  LOMAC must
 * prevent subjects from creating objects at levels that, according to
 * their path and the PLM, are at levels above their own.  There is no
 * need to call this function outside of these two cases.
 *
 */

int mediate_open( lomac_subject_t *p_subject, const char *path_s ) {

  level_t subject_level;            /* holds level of `p_subject' */
  level_t object_level;             /* holds level of `p_object' */

  /* Get the levels of `p_subject' and `path_s'.  Deny open/truncate *
   * if `path_s's level is higher that `p_subject's.                 */ 
  get_subject_level( p_subject, &subject_level );
  get_pathname_level( path_s, &object_level );

  if( ! LEVEL_LEQ( object_level, subject_level ) ) {

    if( verbose & VERBOSE_DEMOTE_DENY ) {
      log_start();
      log_append_string( "LOMAC: " );
      log_append_subject_id( p_subject );
      log_append_string( " level " );
      log_append_int( subject_level );
      log_append_string( " denied creat/trunc of " );
      log_append_string( path_s );
      log_append_string( " level " );
      log_append_int( object_level );
      log_append_string( "\n" );
      log_print();
    }

#ifdef NO_MEDIATION
    return( 1 );
#else
    return( 0 ); /* return 0 to deny trunc */
#endif

  } /* if creat/trunc should be denied */

  return( 1 );  /* allow the creat/trunc */

} /* mediate_open() */


/* mediate_write()
 *
 * in:     p_subject - subject trying to write to `p_object'.
 *         p_object  - object that `p_subject' is trying to write to.
 * out:    nothing
 * return: value    condition
 *         -----    ---------
 *           0      Caller should prevent open
 *           1      Caller should permit open
 * 
 * This function allows LOMAC to mediate write and writev system calls on
 * objects that have names in the filesystem.
 *
 */

int mediate_write( lomac_subject_t *p_subject, lomac_object_t *p_object ) {

  level_t subject_level;     /* level of `p_subject' */
  level_t object_level;      /* level of `p_object' */

#ifdef PARANOID
  if( !p_subject ) {
    PANIC( "LOMAC: null subject pointer passed to mediate_write()\n" );
  }
  if( !p_object ) {
    PANIC( "LOMAC: null object pointer passed to mediate_write()\n" );
  }
#endif

  /* Get the levels of `p_subject' and `p_object' so we can compare them. */
  get_subject_level( p_subject, &subject_level );
  get_object_level(  p_object,  &object_level );

  /* If `p_subject's level is less than `p_object's level,  *
   * we must indicate that the write should not be allowed. */
  if( ! LEVEL_LEQ( object_level, subject_level ) ) {

    if( verbose & VERBOSE_DEMOTE_DENY ) {
      log_start();
      log_append_string( "LOMAC: level-" );
      log_append_int( subject_level );
      log_append_string( " subject " );
      log_append_subject_id( p_subject );
      log_append_string( " denied write to " );
      log_append_int( object_level );
      log_append_string( "-level object " );
      log_append_object_id( p_object );
      log_append_string( "\n" );
      log_print();
    }
    
#ifdef NO_MEDIATION
    return( 1 );
#else
    return( 0 );   /* return 0 here to indicate write should be denied */
#endif

  } /* if caller should deny write */

  return( 1 );  /* allow write */

} /* mediate_write() */


/* mediate_rename()
 *
 * in:     p_subject       - subject doing the rename
 *         canabsoldname_s - old name of file
 *         canabsnewname_s - proposed new name of file
 * out:    nothing
 * return: value   condition
 *         -----    ---------
 *           0      Caller should prevent rename
 *           1      Caller should permit rename
 * 
 * This function allows LOMAC to mediate rename system calls on
 * objects that have names in the filesystem.
 *
 */
 
int mediate_rename( lomac_subject_t *p_subject, const char *canabsoldname_s,
		    const char *canabsnewname_s ) {

  level_t subject_level;    /* level of `p_subject' */
  level_t oldname_level;    /* level of `canabsoldname_s' */
  level_t newname_level;    /* level of `canabsnewname_s' */

#ifdef PARANOID
  if( !p_subject ) {
    PANIC( "LOMAC: null subject pointer passed to mediate_rename()\n" );
  }
  if( !canabsoldname_s ) {
    PANIC( "LOMAC: null old name pointer passed to mediate_rename()\n" );
  }
  if( !is_canabspath( canabsoldname_s ) ) {
    PANIC( "LOMAC: non-canonical/absolute old name "
	   "passed to mediate_rename()\n" );
  }
  if( !canabsnewname_s ) {
    PANIC( "LOMAC: null new name passed to mediate_rename()\n" );
  }
  if( !is_canabspath( canabsnewname_s ) ) {
    PANIC( "LOMAC: non-canonical/absolute new name "
	   "passed to mediate_rename()\n" );
  }
#endif

  /* Get levels so we can compare them. */
  get_subject_level( p_subject, &subject_level );
  get_pathname_level( canabsoldname_s, &oldname_level );
  get_pathname_level( canabsnewname_s, &newname_level );

  /* If `p_subject' has a lower level than `p_canabsoldname_s', deny. */
  if( !( LEVEL_LEQ( oldname_level, subject_level ) ) ) {

    if( verbose & VERBOSE_DEMOTE_DENY ) {
      log_start();
      log_append_string( "LOMAC: level-" );
      log_append_int( subject_level );
      log_append_string( " subject " );
      log_append_subject_id( p_subject );
      log_append_string( " denied rename of " );
      log_append_int( oldname_level );
      log_append_string( "-level object " );
      log_append_string( canabsoldname_s );
      log_append_string( "\n" );
      log_print();
    }

#ifdef NO_MEDIATION
    return( 1 );
#else
    return( 0 );  /* deny */
#endif
  }

  /* If `canabsoldname_s' has a lower level than `canabsnewname_s', deny. */
  if( !( LEVEL_LEQ( newname_level, oldname_level ) ) ) {

    if( verbose & VERBOSE_DEMOTE_DENY ) {
      log_start();
      log_append_string( "LOMAC: level-" );
      log_append_int( subject_level );
      log_append_string( " subject " );
      log_append_subject_id( p_subject );
      log_append_string( " denied rename of " );
      log_append_int( oldname_level );
      log_append_string( "-level object " );
      log_append_string( canabsoldname_s );
      log_append_string( " to level-" );
      log_append_int( newname_level );
      log_append_string( " name " );
      log_append_string( canabsnewname_s );
      log_append_string( "\n" );
      log_print();
    }

#ifdef NO_MEDIATION
    return( 1 );
#else
    return( 0 );  /* deny */
#endif
  }
  
  /* OK, allow rename */
  return( 1 );   /* allow */

} /* mediate_rename() */


/* mediate_link()
 *
 * in:     p_subject - subject doing the link
 *         canabsoldname_s - old name of file
 *         canabsnewname_s - proposed new name of file
 * out:    nothing
 * return: value   condition
 *         -----    ---------
 *           0      Caller should prevent link
 *           1      Caller should permit link
 * 
 * This function allows LOMAC to mediate link system calls on
 * objects that have names in the filesystem.
 *
 */
 
int mediate_link( lomac_subject_t *p_subject, const char *canabsoldname_s,
		  const char *canabsnewname_s ) {

  level_t subject_level;    /* level of `p_subject' */
  level_t oldname_level;    /* level of `canabsoldname_s' */
  level_t newname_level;    /* level of `canabsnewname_s' */

#ifdef PARANOID
  if( !p_subject ) {
    PANIC( "LOMAC: null subject pointer passed to mediate_link()\n" );
  }
  if( !canabsoldname_s ) {
    PANIC( "LOMAC: null old name pointer passed to mediate_link()\n" );
  }
  if( !is_canabspath( canabsoldname_s ) ) {
    PANIC( "LOMAC: non-canonical/absolute old name passed "
	   "to mediate_link()\n" );
  }
  if( !canabsnewname_s ) {
    PANIC( "LOMAC: null new name passed to mediate_link()\n" );
  }
  if( !is_canabspath( canabsnewname_s ) ) {
    PANIC( "LOMAC: non-canonical/absolute new name passed "
	   "to mediate_link()\n" );
  }
#endif

  /* Get levels so we can compare them. */
  get_subject_level( p_subject, &subject_level );
  get_pathname_level( canabsoldname_s, &oldname_level );
  get_pathname_level( canabsnewname_s, &newname_level );

  /* If `p_subject' has a lower level than `canabsoldname_s', deny. */
  if( !( LEVEL_LEQ( oldname_level, subject_level ) ) ) {

    if( verbose & VERBOSE_DEMOTE_DENY ) {
      log_start();
      log_append_string( "LOMAC: level-" );
      log_append_int( subject_level );
      log_append_string( " subject " );
      log_append_subject_id( p_subject );
      log_append_string( " denied link to " );
      log_append_int( oldname_level );
      log_append_string( "-level object " );
      log_append_string( canabsoldname_s );
      log_append_string( "\n" );
      log_print();
    }

#ifdef NO_MEDIATION
    return( 1 );
#else
    return( 0 );  /* deny */
#endif
  }

  /* If `canabsoldname_s' has a different level than *
   * `canabsnewname_s', deny.                        */
  if( !( LEVEL_LEQ( newname_level, oldname_level ) &&
	 LEVEL_LEQ( oldname_level, newname_level ) ) ) {

    if( verbose & VERBOSE_DEMOTE_DENY ) {
      log_start();
      log_append_string( "LOMAC: level-" );
      log_append_int( subject_level );
      log_append_string( " subject " );
      log_append_subject_id( p_subject );
      log_append_string( " denied level-" );
      log_append_int( newname_level );
      log_append_string( " link " );
      log_append_string( canabsnewname_s );
      log_append_string( "to level-" );
      log_append_int( oldname_level );
      log_append_string( " object " );
      log_append_string( canabsoldname_s );
      log_append_string( "\n" );
      log_print();
    }

#ifdef NO_MEDIATION
    return( 1 );
#else
    return( 0 );  /* deny */
#endif
  }
  
  /* OK, allow link */
  return( 1 );   /* allow */

} /* mediate_link() */


/* mediate_kill()
 *
 * in:      p_subject - the signalling process.  Assumed to be `current'.
 *          p_target  - subject targetted by signal, or NULL if the
 *                      signal targets all processes on the system.
 *                      Note that if the signal targets an entire process
 *                      group, you may pass the address of  any process in
 *                      that group as `p_target'.
 * out:     nothing
 * return:  value   condition
 *          -----   ---------
 *            0     caller should prevent kill
 *            1     caller should permit kill
 *
 * Monitors the sys_kill system call.
 *
 */

int mediate_kill( lomac_subject_t *p_subject, lomac_subject_t *p_target ) {

  level_t subject_level;   /* level of `p_subject', the signalling process */
  level_t target_level;    /* level of process(es) targetted by signal */
  int ret_val;             /* value to be returned by this function */

#ifdef PARANOID
  if( !p_subject ) {
    PANIC( "LOMAC: null subject pointer in mediate_kill().\n" );
  }
#endif

  ret_val = 0;        /* pessimistically assume denial. */
  subject_level = LOMAC_INVALID_LEVEL;
  target_level = LOMAC_INVALID_LEVEL;

  /* Make `subject_level' the level of `p_subject'. */
  get_subject_level( p_subject, &subject_level );
#ifdef PARANOID
  if( subject_level == LOMAC_INVALID_LEVEL ) {
    PANIC( "LOMAC: bad subject level in mediate_kill().\n" );
  }
#endif

  /* Mediate based on who the target of the signal is. */
  if( p_target == NULL ) {
    /* Signal all processes.  We will allow this only if *
     * `p_subject' is at the highest level.              */
    if( subject_level == LOMAC_HIGHEST_LEVEL ) {
      ret_val = 1;   /* allow */
    } else {
      /* denial log message */
      if( verbose & VERBOSE_DEMOTE_DENY ) {
	log_start();
	log_append_string( "LOMAC: denied level-" );
	log_append_int( subject_level );
	log_append_string( " proc " );
	log_append_subject_id( p_subject );
	log_append_string( "'s signal to all processes.\n" );
	log_print();
      }
    }
  } else {
    /* Signal a given process or process group.  Since LOMAC treats  *
     * all the members of a given process group as a single subject, *
     * we handle both of these cases together, here.                 */

    /* Set `target_level' to level of `p_target'. */
    get_subject_level( p_target, &target_level );
#ifdef PARANOID
    if( target_level == LOMAC_INVALID_LEVEL ) {
      PANIC( "LOMAC: bad target level in mediate_kill().\n" );
    }
#endif
    
    if( LEVEL_LEQ( target_level, subject_level ) ) {
      ret_val = 1;  /* allow */
    } else {
      /* denial log message */
      if( verbose & VERBOSE_DEMOTE_DENY ) {
	log_start();
	log_append_string( "LOMAC: denied level-" );
	log_append_int( subject_level );
	log_append_string( " proc " );
	log_append_subject_id( p_subject );
	log_append_string( "'s signal to level-" );
	log_append_int( target_level );
	log_append_string( " process or pgrp " );
	log_append_subject_id( p_target );
	log_append_string( ".\n" );
	log_print();
      }
    }
  } /* if/else signal all processes */

#ifdef PARANOID
  if( !( ( ret_val == 0 ) || ( ret_val == 1 ) ) ) {
    PANIC( "LOMAC: bad return value in mediate_kill().\n" );
  }
#endif

#ifdef NO_MEDIATION
  return( 1 );
#else
  return( ret_val );
#endif

} /* mediate_kill() */


/* mediate_is_highest()
 *
 * in:     p_subject   - subject whose level we want to check
 *         operation_s - operation requested by `p_subject'
 * out:    nothing
 * return: value   condition
 *         -----   ---------
 *           0     `p_subject' is not at LOMAC_HIGHEST_LEVEL
 *           1     `p_subject' is at LOMAC_HIGHEST_LEVEL
 *
 * This function provides a predicate for determining whether or not
 * a subject is at the highest possible level.  `operation_s' should
 * be the name of the operation that prompted the test; it's used to
 * make the log messages more informative.
 *
 */

int mediate_is_highest( lomac_subject_t *p_subject, char *operation_s ) {

  level_t subject_level;   /* level of `p_subject', the signalling process */

#ifdef PARANOID
  if( !p_subject ) {
    PANIC( "LOMAC: null subject pointer in mediate_is_highest().\n" );
  }
  if( !operation_s ) {
    PANIC( "LOMAC: null operation_s pointer in mediate_is_highest().\n" );
  }
  subject_level = LOMAC_INVALID_LEVEL;
#endif

  /* Make `subject_level' the level of `p_subject'. */
  get_subject_level( p_subject, &subject_level );
#ifdef PARANOID
  if( subject_level == LOMAC_INVALID_LEVEL ) {
    PANIC( "LOMAC: bad subject level in mediate_is_highest().\n" );
  }
#endif

  if( LEVEL_LEQ( LOMAC_HIGHEST_LEVEL, subject_level ) ) {
    return( 1 );   /* yes, `p_subject' is at highest level */
  }

  /* denial log message */
  if( verbose & VERBOSE_DEMOTE_DENY ) {
    log_start();
    log_append_string( "LOMAC: denied level-" );
    log_append_int( subject_level );
    log_append_string( " proc " );
    log_append_subject_id( p_subject );
    log_append_string( "'s " );
    log_append_string( operation_s );
    log_append_string( ".\n" );
    log_print();
  }

#ifdef NO_MEDIATION
  return( 1 );
#else
  return( 0 );  /* no, `p_subject' is not at the highest level */
#endif

} /* mediate_is_highest() */
