/***************************************************************************
 *
 * lomac_log.c
 *
 * LOMAC - Low Water-Mark Mandatory Access Control for Linux 
 * Copyright (C) 1999, 2000 NAI Labs
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU General Public
 * License as published by the Free Software Foundation.  This program
 * is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.  You should have received a copy of the
 * GNU General Public License along with this program; if not, write
 * to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 *
 *
 *
 ***************************************************************************/

#include "lomac_log.h"


char log_s[ LOG_BUFFER_LENGTH ];    /* the log buffer use by all log fxns */
int terminator_index;               /* index of `\0' in `log_s' */

/* The following definition sets the global log verbosity level. */
unsigned int verbose = ( VERBOSE_DEMOTE_DENY );



/* SPACE_LEFT reports space left in `log_s' for data.  It is aware *
 * that we must reserve a byte in `log_s' for the null terminator; *
 * consequently, it subtracts 1 from the actual `log_s' length to  *
 * ensure that there is always room left for the terminator.       */
#define SPACE_LEFT ( ( LOG_BUFFER_LENGTH - terminator_index ) - 1 )




/* log_start()
 *
 * in:     nothing
 * out:    nothing, see description for side effects
 * return: nothing
 *
 *     This function prepares the `log_s' buffer to receive a new
 * message by setting the data insertion point to the first byte
 * in the `log_s' buffer.
 *
 */

void log_start( void ) {

  terminator_index = 0;
  log_s[ terminator_index ] = '\0';
  log_append_string( LOG_PREFIX );  /* hack - this sets kernel log verbosity */

} /* log_start() */


/* log_append_string()
 *
 * in:     data_s - null-terminated string to append to log
 * out:    nothing, see description for side-effects
 * return: nothing
 *
 *     This function appends `data_s' to `log_s', being careful to ensure
 * that there is sufficient room in `log_s' for the data and a null 
 * terminator.  If there is insufficient room in `log_s' for the entire
 * `data_s' string, this function will append only the prefix of `data_s'
 * which fits.
 *
 */

void log_append_string( const char *data_s ) {

  int allowable_length;     /* amount of `data_s' that will fit in `log_s' */

#ifdef PARANOID
  if( terminator_index >= LOG_BUFFER_LENGTH ) {
    PANIC( "LOMAC: log terminator found beyond end of log buffer" );
  }
#endif PARANOID
  
  allowable_length = STRNLEN( data_s, SPACE_LEFT );
  STRNCAT( log_s, data_s, allowable_length );
  terminator_index += allowable_length;
  log_s[ terminator_index ] = '\0';

#ifdef PARANOID
  if( terminator_index >= LOG_BUFFER_LENGTH ) {
    PANIC( "LOMAC: log terminator left beyond end of log buffer" );
  }
#endif PARANOID
  
} /* log_append_string */


/* log_append_int()
 *
 * in:     data - integer value to append to log
 * out:    nothing, see description for side-effects
 * return: nothing
 *
 *     This function determines the ASCII representation of the integer
 * value in `data' and, if there is sufficient room, appends this
 * ASCII representation to `log_s'.  If there is insufficient room,
 * this function behaves as log_append_string().
 *
 */

void log_append_int( int data ) {

  char ascii[ MAX_DIGITS ];  /* holds the ASCII representation of `data' */

  TOASCII( data, ascii );

#ifdef PARANOID
  if( STRNLEN( ascii, MAX_DIGITS ) == MAX_DIGITS ) {
    PANIC( "LOMAC: excessively long int fed to TOASCII()" );
  }
#endif PARANOID

  log_append_string( ascii );

} /* log_append_int() */


/* log_append_subject_id()
 *
 * in:     p_subject - subject whose ID we want to append to the log message
 * out:    nothing, see description for side-effects
 * return: nothing
 *
 *    This function appends a string describing the identity of `p_subject'
 * to `log_s'.  If there is insufficient room in `log_s' for the entire
 * ID string, only a (possibly empty) prefix of the ID string will be
 * appended.
 *
 */

void log_append_subject_id( lomac_subject_t *p_subject ) {

#ifdef PARANOID
  if( terminator_index >= LOG_BUFFER_LENGTH ) {
    PANIC( "LOMAC: log_append_subject_id found null beyond log buffer end" );
  }
#endif PARANOID

  terminator_index += get_subject_id( p_subject,
				      &(log_s[ terminator_index ]),
				      SPACE_LEFT );
  log_s[ terminator_index ] = '\0';

#ifdef PARANOID
  if( terminator_index >= LOG_BUFFER_LENGTH ) {
    PANIC( "LOMAC: log_append_subject_id left null beyond end of log buffer" );
  }
#endif PARANOID

} /* log_append_subject_id() */


/* log_append_object_id()
 *
 * in:     p_object - object whose ID we want to append to the log message
 * out:    nothing, see description for side-effects
 * return: nothing
 *
 *    This function appends a string describing the identity of `p_object'
 * to `log_s'.  If there is insufficient room in `log_s' for the entire
 * ID string, only a (possibly empty) prefix of the ID string will be
 * appended.
 *
 */

void log_append_object_id( lomac_object_t *p_object ) {

#ifdef PARANOID
  if( terminator_index >= LOG_BUFFER_LENGTH ) {
    PANIC( "LOMAC: log_append_object_id found null beyond end of log buffer" );
  }
#endif PARANOID

  terminator_index += get_object_id( p_object,
				     &(log_s[ terminator_index ]),
				     SPACE_LEFT );
  log_s[ terminator_index ] = '\0';

#ifdef PARANOID
  if( terminator_index >= LOG_BUFFER_LENGTH ) {
    PANIC( "LOMAC: log_append_object_id left null beyond end of log buffer" );
  }
#endif PARANOID

} /* log_append_object_id() */


/* log_print()
 *
 * in:     nothing
 * out:    nothing
 * return: nothing
 *
 *     This function prints `log_s' to the system log.
 *
 */

void log_print( void ) {

  LOG( log_s );

} /* log_print() */





