/***************************************************************************
 *
 * kernel_wrappers.c
 *
 * LOMAC - Low Water-Mark Mandatory Access Control for Linux 
 * Copyright (C) 1999, 2000 NAI Labs
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU General Public
 * License as published by the Free Software Foundation.  This program
 * is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.  You should have received a copy of the
 * GNU General Public License along with this program; if not, write
 * to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 *
 * This file contains functions which "wrap" kernel system calls, and
 * a number of utility functions to serve them.  When the LKM loads,
 * the initialization routines in kernel_lkm.c modify the system call
 * table jump table by replacing entries for system calls that are
 * interesting to the LOMAC LKM with entries for corresponding
 * "wrapper" functions in this file.  When a user-space program calls
 * one of these interesting system calls, the replacement wrapper
 * function gets called instead.
 *
 * The wrapper functions all have a similar structure.  They all take
 * exactly the same arguments as the system call functions they
 * replace.  Their first step is to allow the LOMAC LKM to examine
 * these parameters in order to do monitoring or mediation.  Second,
 * (if the mediation result allows it) they call the original system
 * call, giving the calling user-space program the service it expects.
 * Finally, they allow the LOMAC LKM a second chance to do monitoring
 * after the original system call has returned.
 *
 * There are some serious time-of-check/time-of-use bugs in these
 * functions that need to be addressed.
 *
 ***************************************************************************/

#include <linux/sched.h>   /* for `current' process */
#include <linux/fs.h>      /* for inode, dentry, file, iovec structures */
#include <linux/file.h>    /* for fcheck() */
#include <linux/net.h>     /* for SYS_BIND and friends... */
#include <linux/socket.h>  /* for struct sockaddr */
#include <linux/un.h>      /* for struct sockaddr_un */
#include <net/sock.h>      /* for struct sock, struct unix_opt */
#include <asm/uaccess.h>   /* for copy_from_user() */

#include "kernel_wrappers.h"
#include "kernel_util.h"
#include "lomac_monitor.h"
#include "lomac_mediate.h"
#include "lomac_log.h"


/* LOMAC always allows writes to character devices such as serial  *
 * ports, ttys, pseudo-ttys, and /dev/null.  This macro identifies *
 * such devices, for the convenience of the creat/write mediation  *
 * handling functions.                                             */

#define WRITE_EXEMPT( pd ) \
  ( ( (pd)->d_inode->i_mode & S_IFCHR ) &&     /* if char device */     \
    ( ( MAJOR( (pd)->d_inode->i_rdev ) == 5 )   ||     /* cua  */       \
      ( MAJOR( (pd)->d_inode->i_rdev ) == 4 )   ||     /* tty  */       \
      ( MAJOR( (pd)->d_inode->i_rdev ) == 3 )   ||     /* ttyp */       \
      ( MAJOR( (pd)->d_inode->i_rdev ) == 136 ) ||     /* pts  */       \
      ( MAJOR( (pd)->d_inode->i_rdev ) == 2 )   ||     /* pty  */       \
      ( MAJOR( (pd)->d_inode->i_rdev ) == 6 )   ||     /* lp  */        \
      ( ( MAJOR( (pd)->d_inode->i_rdev ) == 10 ) &&    /* /dev/psaux */ \
        ( MINOR( (pd)->d_inode->i_rdev ) == 1 ) ) ||                    \
      ( ( MAJOR( (pd)->d_inode->i_rdev ) == 1 ) &&     /* /dev/null */  \
        ( MINOR( (pd)->d_inode->i_rdev ) == 3 ) ) ) ) 



int handle_mediate_open( const char *path_s, int grant_exemption_flag );
void handle_monitor_read( struct dentry *p_dentry );
int handle_mediate_write( struct dentry *p_dentry );
int handle_accept( int call, unsigned long *args );
int handle_bind( int call, unsigned long *args );
int handle_connect( int call, unsigned long *args );
int handle_recv( int call, unsigned long *args );
int handle_socketpair( int call, unsigned long *args );

/**************************************************************************
 *
 * wrapper functions (functions that intercept system calls and call
 * the higher-level LOMAC routines to react to them)...
 *
 **************************************************************************/


/* wrap_setpgid() 
 *
 * LOMAC's "subject" concept is equivalent to a pgrp ("job").
 * Processes can use the setpgid syscall to move between pgrps, and
 * therefore between subjects.  This movement is fine, provided that
 * no process moves from one pgrp to another of higher level.  We must
 * wrap the setpgid system call to prevent this situation from
 * happening.
 *
 * Potential problem:  is it possible for the shell to fork a two-process
 * pipeline, setpgid the source proc, source proc reads, is demoted, 
 * writes to pipe, *exits and is removed from proc table*, all before
 * the shell setpgid's the sink proc?  If so, the sink proc could decide
 * that it is the first in the new pgrp, and avoid demotion.
 *
 * LOMAC attempts to provide the following guarantees:
 * o No process can increase its level by moving from one job to another
 *   existing job with a higher level.
 * o No processs can set the pgid of another process with a higher level.
 *
 */

int wrap_setpgid( pid_t pid, pid_t pgid ) {

  struct task_struct *p_changing_proc = NULL; /* proc whose pgid is changing */
  struct task_struct *p_dest_proc = NULL;   /* someproc already in dest pgrp */
  struct task_struct *p_proc = NULL;       /* used to traverse process table */
  int need_to_mediate_flag = 1; /* set if LOMAC needs to mediate pgrp change */
  int ret_val;                              /* value returned by system call */

  /* printk( "%u sets %u to  %u\n", current->pid, pid, pgid ); */

  /* Determine actual pgid from parms, as sys_setpgid does */  
  if( !pgid ) {
    pgid = pid;
  }
  if( pgid < 0 ) {
    return( -EINVAL );
  }

  /* If our parameters indicate that the `current' process is using    *
   * setpgid() to change a process into an existing process group,     *
   * then LOMAC must mediate the change.  On the other hand, if the    *
   * parameters indicate that the `current' process is using setpgid() *
   * to create a new process group, then LOMAC does not need to        *
   * mediate the change.  We must determine which case we have.  For   *
   * safety's sake, we will initially set `need_to_mediate_flag' to 1, *
   * and clear it if we decide we don't have to mediate, after all.    */

  need_to_mediate_flag = 1;  /* initially assume we must mediate setpgid */

  /* There are two cases where setpgid() will create a new process  *
   * group: first, if `pgid' is still zero (thanks to the above if( *
   * !pgid ) statement, this will happen only when `pid' is also    *
   * zero), and second, if `pgid' contains a number that is not a   *   
   * currently-valid process group ID.  If we have either of these  *
   * two cases, clear `need_to_mediate_flag' to 0.                  */
  if( !pgid ) {

    need_to_mediate_flag = 0;

  } else {
      
  /* Try to find a process in the `pgid' pgrp.  If we can find no such    *
   * process, then the `current' process is trying to create a new group. */

    /* read_lock( &tasklist_lock ); */
    p_dest_proc = NULL;
    p_proc = current;
    do {
      if( p_proc->pgrp == pgid ) {
	p_dest_proc = p_proc;
	break;
      }
    } while( ( p_proc = p_proc->next_task ) != current );
    /* read_unlock( &tasklist_lock ); */

    if( !p_dest_proc ) {
      /* No match found, this is a new group creation, don't mediate. */
      need_to_mediate_flag = 0;
    }

  } /* if/else `pgid' set */
    

  /* If the current process is attempting to move process `pid'       *
   * into an existing pgrp (rather than creating a new pgrp), then we *
   * must perform mediation.  We will leave it to the kernel's own    *
   * access controls to deal with the issue of whether or not the     *
   * current process should be allowed to change `p_changing_proc's   *
   * pgrp at all.                                                     */
  if( need_to_mediate_flag ) {

    /* Figure out which process is getting its pgrp changed.  This is *
     * determined by `pid', which either holds the pid of the process *
     * in question, or 0 for the `current' process.                   */
    p_changing_proc = NULL;   /* for neatness */
    if( pid ) {
      /* read_lock( &tasklist_lock ); */
      p_changing_proc = pid_to_task( pid );
      /* read_unlock( &tasklist_lock ); */
      if( !p_changing_proc ) {
	return( -ESRCH );   /* mimic sys_setpgid() behavior */
      }
    } else {
      p_changing_proc = current;
    }

#ifdef PARANOID
    if( !p_changing_proc ) {
      PANIC( "LOMAC/K: "
	     "wrap_setpgid failed to set changing proc before mediate.\n" );
    }
    if( !p_dest_proc ) {
      PANIC( "LOMAC/K: "
	     "wrap_setpgid failed to set dest proc before mediate.\n" );
    }
#endif

    /* Let LOMAC mediate */
    if( ! mediate_pgrp_change( p_changing_proc, p_dest_proc ) ) {
      return( -EPERM );   /* deny this pgrp change */
    }
      
  } /* mediate only changes to existing pgrps, not creates */

  /* Call original system call */
  ret_val = ((int (*)(pid_t, pid_t))orig_setpgid)( pid, pgid );

  /* If this was a pgrp move and not a create, and if the actual    *
   * system call succeeded, then we need to update LOMAC's internal *
   * state to reflect the move.                                     */
  if( ( ret_val == 0 ) && ( need_to_mediate_flag ) ) {

#ifdef PARANOID
    if( !p_changing_proc ) {
      PANIC( "LOMAC/K: "
	     "wrap_setpgid failed to set changing proc before monitor.\n" );
    }
    if( !p_dest_proc ) {
      PANIC( "LOMAC/K: "
	     "wrap_setpgid failed to set dest proc before monitor.\n" );
    }
#endif

    monitor_pgrp_change( p_changing_proc, p_dest_proc );

  } /* if we need to monitor */ 

  /* return to the caller */
  return( ret_val );
  
} /* wrap_setpgid() */


/* wrap_open()
 *
 * Processes can use the open system call to create new files, or to
 * open existing files.  In both cases, we must call `monitor_open()'
 * to prompt LOMAC to assign a level to the inode resulting from the
 * open (assuming the open succeeded).  When a process uses the open
 * system call to create a new file, we must also call
 * `monitor_creat()' to allow LOMAC to mediate (allow/deny) the
 * create.
 *
 * LOMAC attempts to provide the following guarantees:
 * o No process will create a file whose level is higher than its own.
 *   Exception: creats are always allowed on files like serial devices
 *   and pseudo-terminals.  See WRITE_EXEMPT macro for complete list of
 *   exempt files.
 * o No process will truncate a file whose level is higher than its own.
 *   Exception: truncates are always allowed on files like serial devices
 *   and pseudo-terminals.  See WRITE_EXEMPT macro for complete list of
 *   exempt files.
 *
 */

int wrap_open( const char *filename, int flags, int mode ) {

  int ret_val;               /* holds value returned by open system call */

  /* If `flags' has the O_TRUNC flag set, then the `current' process *
   * is trying to create a zero-length file (truncate) named by      *
   * `filename'.  This action is effectively a write, so we must     *
   * perform mediation before calling the kernel's open system call. */

  /* We also wish to prevent the `current' process from creating a     *
   * file with a level higher than its own.  So, we will also perform  *
   * mediation if `flags' contains O_CREAT.  This action is actually   *
   * more restrictive than we need to be - the O_CREAT flag causes     *
   * creation only if the file specified by `filename' does not        *
   * already exist.  A more precise choice would be to mediate ( if    *
   * O_CREAT is in `flags' AND if the `filename' file does not already *
   * exist).  However, merely checking for O_CREAT is less work, and   *
   * we will stick with this method unless it causes an unacceptable   *
   * number of application failures.                                   */

  if( ( flags & O_TRUNC ) || ( flags & O_CREAT ) ) { 
    if( 0 != ( ret_val = handle_mediate_open( filename, 1 ) ) ) {
      return( ret_val );
    }
  } /* If open wants to truncate */

  /* Make open system call. */ 
  ret_val = ((int (*)(const char *, int, int))orig_open)
    ( filename, flags, mode );

  /* If the open system call succeeded, prompt LOMAC to set the level *
   * of the inode it created to represent the opened file.            */
  if( ret_val >= 0 ) {
    monitor_open( current, fd_to_dentry( ret_val ) );
  }

  return( ret_val );

} /* wrap_open() */ 



/* wrap_creat()
 *
 * This function is similar to wrap_open, except that it handles the
 * creat system call.  Unlike the open system call, the creat system
 * call always requires us to call both monitor_open() and
 * monitor_creat().
 *
 * LOMAC attempts to provide the following guarantees:
 * o No process will create a file whose level is higher than its own.
 *   Exception: creats are always allowed on files like serial devices
 *   and pseudo-terminals.  See WRITE_EXEMPT macro for complete list of
 *   exempt files.
 * o No process will truncate a file whose level is higher than its own.
 *   Exception: truncates are always allowed on files like serial devices
 *   and pseudo-terminals.  See WRITE_EXEMPT macro for complete list of
 *   exempt files.
 */
 
int wrap_creat( const char *filename, int mode ) {

  int ret_val;               /* holds value returned by open system call */

  /* The `current' process is trying to create a zero-length file    *
   * (truncate) named by `filename'.  This action is effectively a   *
   * write, so we must perform mediation before calling the kernel's *
   * creat system call.                                              */
  if( 0 != ( ret_val = handle_mediate_open( filename, 1 ) ) ) {
    return( ret_val );
  }

  /* Make creat system call. */ 
  ret_val = ((int (*)(const char *, int))orig_creat)( filename, mode );

  /* If the creat system call succeeded, prompt LOMAC to set the level *
   * of the inode it created to represent the created file.            */
  if( ret_val >= 0 ) {
    monitor_open( current, fd_to_dentry( ret_val ) );
  }

  return( ret_val );

} /* wrap_creat() */


/* wrap_read()
 *
 * LOMAC attempts to provide the following guarantees:
 * o Upon reading from a file, local-domain socket, network
 *   socket, unnamed pipe, or a FIFO with a level lower than its own,
 *   the level of the current process's job will be reduced to match
 *   the object's level.
 *
 */

int wrap_read( unsigned int fd, char *buf, int count ) {

  int ret_val;               /* holds return value of original syscall */
  struct dentry *p_dentry;   /* dcache entry corresponding to `fd' */

  if( !( p_dentry = fd_to_dentry( fd ) ) ) {
    return( -EBADF );   /* fail as original system call does */
  }

  ret_val = ((int (*)(unsigned int, char *, int))orig_read)( fd, buf, count );

  if( ret_val >= 0 ) {  /* If read succeeded, we must monitor it. */

    handle_monitor_read( p_dentry );

  } /* if read succeeded */

  return( ret_val );

} /* wrap_read() */


/* wrap_readv()
 *
 */

int wrap_readv( unsigned long fd, const struct iovec *vector, long count ) {

  int ret_val;               /* holds return value of original syscall */
  struct dentry *p_dentry;   /* dcache entry corresponding to `fd' */

  if( !( p_dentry = fd_to_dentry( fd ) ) ) {
    return( -EBADF );   /* fail as original system call does */
  }

  ret_val = ((int (*)(unsigned long, const struct iovec *, long))orig_readv)
    ( fd, vector, count );

  if( ret_val >= 0 ) {  /* If readv succeeded, we must monitor it. */

      handle_monitor_read( p_dentry );

  } /* if readv succeeded */

  return( ret_val );

} /* wrap_readv() */


/* wrap_write()
 *
 * LOMAC attempts to provide the following guarantees:
 * o No process will write to a file, local-domain socket, or a FIFO with
 *   a level higher than its own.
 *   Exception: writes are always allowed on files like serial devices
 *   and pseudo-terminals.  See WRITE_EXEMPT macro for complete list of
 *   exempt files.
 * o If a process writes to a pipe, the pipe's level will be reduced to
 *   match the process's level. 
 *
 */

int wrap_write( unsigned int fd, char *buf, unsigned int count ) {

  struct dentry *p_dentry;   /* dcache entry corresponding to `fd' */
  int ret_val;               /* holds function return values */

  if( !( p_dentry = fd_to_dentry( fd ) ) ) {
    return( -EBADF );   /* fail as original system call does */
  }

  /* If `fd' is not an unnamed pipe, we must perform mediation on the write */
  if( !( IS_PIPE( p_dentry ) ) ) {
    if( ! handle_mediate_write( p_dentry ) ) {
      return( -EPERM );  /* LOMAC indicates write should be denied. */
    }
  }

  /* LOMAC indicates write should be allowed.  Make system call. */
  ret_val = ((int (*)(unsigned int, char *, int))orig_write)( fd, buf, count );

  /* If the write succeeded, and if `fd' is an unnamed pipe, we must monitor */
  if( ( ret_val >= 0 ) && ( IS_PIPE( p_dentry ) ) ) {
    monitor_pipe_write( current, p_dentry );
  }

  return( ret_val );

} /* wrap_write() */


/* wrap_writev()
 *
 */

int wrap_writev( unsigned long fd, const struct iovec *vector, long count ) {

  struct dentry *p_dentry;   /* dcache entry corresponding to `fd' */
  int ret_val;               /* holds function return values */

  if( !( p_dentry = fd_to_dentry( fd ) ) ) {
    return( -EBADF );   /* fail as original system call does */
  }

  /* If `fd' is not an unnamed pipe, we must perform mediation on the write */
  if( !( IS_PIPE( p_dentry ) ) ) {
    if( ! handle_mediate_write( p_dentry ) ) {
      return( -EPERM );    /* LOMAC indicates write should be denied. */
    }
  }

  /* LOMAC indicates write should be allowed.  Make system call. */
  ret_val = ((int (*)(unsigned long, const struct iovec *, long))orig_writev)
                     ( fd, vector, count );

  /* If the write succeeded, and if `fd' is an unnamed pipe, we must monitor */
  if( ( ret_val >= 0 ) && ( IS_PIPE( p_dentry ) ) ) {
    monitor_pipe_write( current, p_dentry );
  }

  return( ret_val );

} /* wrap_writev() */


/* wrap_pipe()
 *
 * LOMAC attempts to provide the following guarantees:
 * o A newly-created pipe will inherit the level of its creator.
 *
 */

int wrap_pipe( unsigned long *fildes ) {

  int ret_val;              /* holds return value */
  unsigned long fd_index;   /* file descriptor index for pipe inode */
  struct dentry *p_dentry;  /* points to dentry corresponding to pipe */


  ret_val = ((int (*)(unsigned long *))orig_pipe)( fildes );

  /* If this was a successful pipe creation, set initial pipe "level" */
  if( !ret_val ) {

    /* The kernel's sys_pipe() call writes the appropriate file      *
     * descriptor values directly out to the calling process's file  *
     * table.  We must copy one of the file descriptors back into    *
     * kernel space to figure out what inode the kernel is using for *
     * the pipe.                                                     */
    copy_from_user( &fd_index, fildes, sizeof( long ) );
    p_dentry = fd_to_dentry( fd_index );

#ifdef PARANOID
    if( !p_dentry ) { 
      /* This situation shouldn't occur, because syscall just succeeded. */
      PANIC( "LOMAC/K: wrap_pipe got bad file descriptor.\n" );
    }
#endif

    monitor_pipe_create( current, p_dentry );

  } /* if pipe was successfully created */

  return( ret_val );

} /* wrap_pipe() */


/* wrap_mknod()
 *
 * LOMAC attempts to provide the following guarantees:
 * o A low-level process cannot create a high-level FIFO.
 * o A low-level process can use mknod only to create FIFOs.
 *
 */

int wrap_mknod(const char *filename, int mode, dev_t dev) {

  int ret_val;               /* holds function return values */
  struct dentry *p_dentry;   /* dcache entry for new FIFOs */

#ifdef PARANOID
  if( !filename ) {
    PANIC( "LOMAC/K: sys_mknod passed null filename\n" );
  }
#endif

  /* If `current' is trying to create a FIFO, make sure LOMAC will *
   * allow `current' to create a file with name `filename'.        */
  if( S_ISFIFO( mode ) ) {
    if( 0 != ( ret_val = handle_mediate_open( filename, 0 ) ) ) {
      return( ret_val );
    }
  } else {

    /* If `current' wants to do something other than create a FIFO,   *
     * we must ensure that `current' is at the highest level.  David  *  
     * Wheeler of IDA has suggested a better scheme where low-level   *
     * processes can use mknod to create certain kinds of harmless    *
     * device special files, but this requires LOMAC to understand    *
     * device major and minor numbers.  Due to time constraints, this *
     * functionality will have to wait for a later version of LOMAC.  */

    if( !mediate_is_highest( current, "mknod" ) ) {
      return( -EPERM );
    }

  } /* if/else current is creating a FIFO */

  ret_val = ((int (*)(const char *, int, dev_t))orig_mknod)
    ( filename, mode, dev );

  /* If the mknod call succeeded in making a FIFO, prompt LOMAC to set *
   * the level of the inode it created to represent the new FIFO.      */
  if( S_ISFIFO( mode ) && ret_val >= 0 ) {

    /* Find the dir cache entry corresponding to the new FIFO *
     * and pass it to LOMAC's monitoring function.            */
    p_dentry = namei( filename );
    if( !IS_ERR( p_dentry ) ) {
      monitor_open( current, p_dentry );
      dput( p_dentry );
    } 
#ifdef PARANOID
    else {
      /* the namei() call shouldn't fail, since the mknod succeded. */
      PANIC( "LOMAC/K: unexpected namei() failure in wrap_mknod().\n" );
    }
#endif      

  } /* if mknod call succeeded in creating a new FIFO */

  return( ret_val );

} /* wrap_mknod() */


/* wrap_truncate()
 *
 * LOMAC attempts to provide the following guarantees:
 * o A process cannot tuncate a file, fifo, or local-domain socket with
 *   a level higher than its own.  
 *   Exception: truncates are always allowed on files that are exempt
 *              from write mediation, like serial devices and pseudo-
 *              terminals.  See WRITE_EXEMPT macro for complete list
 *              of exempt files.
 */

int wrap_truncate( const char *path, unsigned long length ) {

  int ret_val;               /* holds value returned by open system call */

  /* The `current' process is trying to truncate the file named by *
   * `filename'.  This action is effectively a write, much like a  *
   * creat.  We will handle this write in the same way we handle   * 
   * writes due to creat.                                          */
  if( 0 != ( ret_val = handle_mediate_open( path, 1 ) ) ) {
    return( ret_val );
  }

  /* Make truncate system call. */ 
  ret_val = ((int (*)(const char *, unsigned long))orig_truncate)
    ( path, length );

  return( ret_val );

} /* wrap_truncate() */


/* wrap_ftruncate()
 *
 * LOMAC attempts to provide the following guarantees:
 * o A process cannot tuncate a file, fifo, or local-domain socket with
 *   a level higher than its own.
 *   Exception: truncates are always allowed on files that are exempt
 *              from write mediation, like serial devices and pseudo-
 *              terminals.  See WRITE_EXEMPT macro for complete list
 *              of exempt files.
 * o If a process truncates an unnamed pipe with a level higher than its
 *   own, the unnamed pipe's level will be reduced to match the process's
 *   level.
 */

int wrap_ftruncate( unsigned int fd, unsigned long length ) {

  struct dentry *p_dentry;   /* dcache entry corresponding to `fd' */
  int ret_val;               /* holds function return values */

  if( !( p_dentry = fd_to_dentry( fd ) ) ) {
    return( -EBADF );   /* fail as original system call does */
  }

  /* If `fd' is not an unnamed pipe, we must perform mediation on the write */
  if( !( IS_PIPE( p_dentry ) ) ) {
    if( ! handle_mediate_write( p_dentry ) ) {
      return( -EPERM );  /* LOMAC indicates write should be denied. */
    }
  }

  /* LOMAC indicates write should be allowed.  Make system call. */
  ret_val = ((int (*)(unsigned int, unsigned long))orig_ftruncate)
    ( fd, length );

  /* If the write succeeded, and if `fd' is an unnamed pipe, we must monitor */
  if( ( ret_val >= 0 ) && ( IS_PIPE( p_dentry ) ) ) {
    monitor_pipe_write( current, p_dentry );
  }

  return( ret_val );

} /* wrap_ftruncate() */


/* wrap_unlink()
 *
 * LOMAC attempts to provide the following guarantees:
 * o No process can unlink a file, fifo, or local-domain socket whose
 *   level is higher than its own.
 *
 */

int wrap_unlink( const char *pathname ) {

  int ret_val;        /* holds return values from functions */

  /* The `current' process is trying to unlink the file named by  *
   * `filename'.  This action is effectively a write, much like a *
   * creat.  We will handle this write in the same way we handle  * 
   * writes due to creat, except we will not automatically allow  *
   * the unlinking of write-mediation-exempt files such as serial *
   * devices.                                                     */
  if( 0 != ( ret_val = handle_mediate_open( pathname, 0 ) ) ) {
    return( ret_val );
  }

  /* Make unlink system call. */ 
  ret_val = ((int (*)(const char *))orig_unlink)( pathname );

  return( ret_val );

} /* wrap_unlink() */


/* wrap_rename()
 *
 * LOMAC attempts to provide the following guarantees:
 * o a process cannot rename a file if the file's level is higher
 *   than the process's level
 * o a process cannot rename a file if the new name for the file
 *   would cause it to have a higher level than the old name.
 *   (recall that file levels are determined by filename in this
 *    version of LOMAC).
 *
 */

int wrap_rename( const char *oldname, const char *newname ) {

  char *canabsoldname_s;     /* canonical absolute version of `oldname' */
  char *canabsnewname_s;     /* canonical absolute version of `newname' */
  int ret_val;               /* holds values returned by functions */

  /* Allocate memory for `canabsoldname_s' and `canabsnewname_s' */
  if( !( canabsnewname_s = (char *)__get_free_page( GFP_KERNEL ) ) ) {
    return( -ENOMEM );
  }
  if( !( canabsoldname_s = (char *)__get_free_page( GFP_KERNEL ) ) ) {
    free_page( (unsigned long)canabsnewname_s ); /* done with this, thanks! */ 
    return( -ENOMEM );
  }

  /* mediate_rename() requires the canonical absolute forms of the     *
   * two paths `oldname' and `newname' in order to make its decision.  *
   * We'll call predict_object_canabspath() for each of these two      *
   * paths before calling mediate_rename().  If either of these two    *
   * calls fails before we get to mediate_rename(), we'll return the   *
   * error code and neither mediation nor the call to the kernel's     *
   * original sys_rename() function will occur.  The following         *
   * nested if's implement this behavior, whatever happens, we hit the *
   * memory-freeing code at the end of the if's, and have the correct  *
   * return value in `ret_val'.                                        */

  /* Get canonical absolute form of `oldname' into `canabsoldname'. */
  if( ! ( ret_val = predict_object_canabspath( oldname, canabsoldname_s,
					       LOG_BUFFER_LENGTH ) ) ) {

    /* Get canonical absolute form of `newname' into `canabsnewname'. */
    if( ! ( ret_val = predict_object_canabspath( newname, canabsnewname_s,
						 LOG_BUFFER_LENGTH ) ) ) {

      /* Perform mediation.  mediate_rename() returns 1 for allow *
       * and zero for deny.  Translate these return values into   *
       * the kernel-specific 0 for allow and -EPERM for deny.     */
      if( mediate_rename( current, canabsoldname_s, canabsnewname_s ) ) {
	ret_val = 0;       /* kernel-specific allow */
      } else {
	ret_val = -EPERM;  /* kernel-specific deny */
      }

    } /* if we got `canabsnewname_s' without error */

  } /* if we got `canabsoldname_s' without error */

  free_page( (unsigned long)canabsoldname_s ); /* done with this, thanks! */ 
  free_page( (unsigned long)canabsnewname_s ); /* done with this, thanks! */ 

  /* `ret_val' should contain 0 if no errors or mediation denials    *
   * occurred, or some non-zero value indicating that we should stop *
   * right here.                                                     */
  if( ret_val ) {
    return( ret_val );
  }
  
  /* mediation passed, found no errors, call original rename system call */
  return( ((int (*)(const char *, const char *))orig_rename)
	  ( oldname, newname ) );

} /* wrap_rename() */


/* wrap_link()
 *
 * LOMAC attempts to provide the following guarantees:
 * o a process cannot link to a file if the file's level is higher
 *   than the process's level
 * o a process can make a new link to an old file only if the file
 *   has the same level when named through both new and old links. 
 *
 */

int wrap_link( const char *oldname, const char *newname ) {

  char *canabsoldname_s;     /* canonical absolute version of `oldname' */
  char *canabsnewname_s;     /* canonical absolute version of `newname' */
  int ret_val;               /* holds values returned by functions */

  /* Allocate memory for `canabsoldname_s' and `canabsnewname_s' */
  if( !( canabsnewname_s = (char *)__get_free_page( GFP_KERNEL ) ) ) {
    return( -ENOMEM );
  }
  if( !( canabsoldname_s = (char *)__get_free_page( GFP_KERNEL ) ) ) {
    free_page( (unsigned long)canabsnewname_s ); /* done with this, thanks! */ 
    return( -ENOMEM );
  }

  /* mediate_link() requires the canonical absolute forms of the       *
   * two paths `oldname' and `newname' in order to make its decision.  *
   * We'll call predict_object_canabspath() for each of these two      *
   * paths before calling mediate_link().  If either of these two      *
   * calls fails before we get to mediate_link(), we'll return the     *
   * error code and neither mediation nor the call to the kernel's     *
   * original sys_link() function will occur.  The following           *
   * nested if's implement this behavior, whatever happens, we hit the *
   * memory-freeing code at the end of the if's, and have the correct  *
   * return value in `ret_val'.                                        */

  /* Get canonical absolute form of `oldname' into `canabsoldname'. */
  if( ! ( ret_val = predict_object_canabspath( oldname, canabsoldname_s,
					       LOG_BUFFER_LENGTH ) ) ) {

    /* Get canonical absolute form of `newname' into `canabsnewname'. */
    if( ! ( ret_val = predict_object_canabspath( newname, canabsnewname_s,
						 LOG_BUFFER_LENGTH ) ) ) {

      /* Perform mediation.  mediate_link() returns 1 for allow *
       * and zero for deny.  Translate these return values into *
       * the kernel-specific 0 for allow and -EPERM for deny.   */
      if( mediate_link( current, canabsoldname_s, canabsnewname_s ) ) {
	ret_val = 0;       /* kernel-specific allow */
      } else {
	ret_val = -EPERM;  /* kernel-specific deny */
      }

    } /* if we got `canabsnewname_s' without error */

  } /* if we got `canabsoldname_s' without error */

  free_page( (unsigned long)canabsoldname_s ); /* done with this, thanks! */ 
  free_page( (unsigned long)canabsnewname_s ); /* done with this, thanks! */ 

  /* `ret_val' should contain 0 if no errors or mediation denials    *
   * occurred, or some non-zero value indicating that we should stop *
   * right here.                                                     */
  if( ret_val ) {
    return( ret_val );
  }
  
  /* mediation passed, call original link system call */
  return( ((int (*)(const char *, const char *))orig_link)
	  ( oldname, newname ) );

} /* wrap_link */


/* wrap_socketcall()
 *
 */

int wrap_socketcall( int call, unsigned long *args ) {

  /* The kernel multiplexes many socket services in a single system     *
   * call: sys_socketcall.  Callers specify the service they want in    *
   * `call'.  LOMAC is concerned with only a few of these services.     *
   * The following switch sends calls to those services that LOMAC      *
   * cares about to the appropriate handling function.  The rest fall   *
   * through and are passed directly on to the kernel's sys_socketcall. */
  switch( call ) {
  case SYS_ACCEPT:
    return( handle_accept( call, args ) );
  case SYS_BIND:
    return( handle_bind( call, args ) );
  case SYS_CONNECT:
    return( handle_connect( call, args ) );
  case SYS_RECV:
  case SYS_RECVFROM:
  case SYS_RECVMSG:
    return( handle_recv( call, args ) );
  case SYS_SOCKETPAIR:
    return( handle_socketpair( call, args ) );
  }

  /* call original sys_socketcall() */
  return( ((int (*)(int, unsigned long *))orig_socketcall)( call, args ) );

} /* wrap_socketcall() */


/* wrap_kill()
 *
 */

int wrap_kill( int pid, int sig ) {

  struct task_struct *p_target = NULL;  /* task that is target of signal */

  /* We must set up the parms to mediate_kill() differently depending *
   * on who the target of the signal is.  The target is determined by *
   * the `pid' parm as follows: 0 means signal to the current         *
   * process's process group, -1 means signal all processes, other    *
   * negative numbers mean signal process group -pid, and positive    *
   * numbers means signal a particular process by pid.                */
  if( pid != 0 ) {  /* skip mediation on signals to own process group */

    if( pid == -1 ) {
      p_target = NULL;   /* signal to all processes */
    } else if( pid < -1 ) {
      /* Signal pgrp -pid.  Make `p_target' point to any *
       * process in pgrp number -pid.                    */
      p_target = pgrp_to_task( -pid );

      /* If there is no such task, return same error as sys_kill(). */
      if( !p_target ) {
	return( -ESRCH );
      }
    } else {
      /* Signal process pid.  Make `p_target' point to process `pid'. */
      p_target = pid_to_task( pid );

      /* If there is no such task, return same error as sys_kill(). */
      if( !p_target ) {
	return( -ESRCH );
      }
    }

    /* Now that we have `p_target' set, perform mediation. */
    if( !mediate_kill( current, p_target ) ) {
      return( -EPERM );
    }

  } /* if we need to mediate */

  return( ((int (*)(int, int))orig_kill)( pid, sig ) );

}


/* wrap_mkdir()
 *
 * LOMAC attempts to provide the following guarantees:
 * o No process can make a directory with a level is higher than its own.
 *
 */

int wrap_mkdir( const char *pathname, int mode ) {

  int ret_val;        /* holds return values from functions */

  /* The `current' process is trying to create the directory named   *
   * by `pathname'.  This action is effectively a write, much like   *
   * a creat.  We will handle this write in the same way we handle   *
   * writes due to creat, except we will not allow write-exemptions, *
   * which are intended for serial devices, etc., not directories.   */
  if( 0 != ( ret_val = handle_mediate_open( pathname, 0 ) ) ) {
    return( ret_val );
  }

  /* Make mkdir system call. */ 
  ret_val = ((int (*)(const char *, int))orig_mkdir)( pathname, mode );

  return( ret_val );

} /* wrap_mkdir() */


/* wrap_rmdir()
 *
 * LOMAC attempts to provide the following guarantees:
 * o No process can remove a directory with a level is higher than its own.
 *
 */

int wrap_rmdir( const char *pathname ) {

  int ret_val;        /* holds return values from functions */

  /* The `current' process is trying to remove the directory named   *
   * by `pathname'.  This action is effectively a write, much like   *
   * a creat.  We will handle this write in the same way we handle   *
   * writes due to creat, except we will not allow write-exemptions, *
   * which are intended for serial devices, etc., not directories.   */
  if( 0 != ( ret_val = handle_mediate_open( pathname, 0 ) ) ) {
    return( ret_val );
  }

  /* Make rmdir system call. */ 
  ret_val = ((int (*)(const char *))orig_rmdir)( pathname );

  return( ret_val );

} /* wrap_rmdir() */


/* wrap_mount()
 *
 * LOMAC attempts to provide the following guarantees:
 * o Only highest-level processes can mount.
 *
 */

int wrap_mount( char *dev_name, char *dir_name, char *type,
		unsigned long new_flags, void *data ) {

  if( !mediate_is_highest( current, "mount" ) ) {
    return( -EPERM );
  }

  return( ((int (*)( char *, char *, char *, unsigned long, void *))orig_mount)
	  ( dev_name, dir_name, type, new_flags, data ) );

} /* wrap_mount() */


/* wrap_umount()
 *
 * LOMAC attempts to provide the following guarantees:
 * o Only highest-level processes can umount.
 *
 */

int wrap_umount( char *name, int flags ) {

  if( !mediate_is_highest( current, "umount" ) ) {
    return( -EPERM );
  }

  return( ((int (*)( char *, int ))orig_umount)( name, flags ) );

} /* wrap_mount() */


/* wrap_reboot()
 *
 * LOMAC attempts to provide the following guarantees:
 * o Only highest-level processes can reboot.
 *
 */

int wrap_reboot( int magic1, int magic2, int cmd, void *arg ) {

  if( !mediate_is_highest( current, "reboot" ) ) {
    return( -EPERM );
  }

  return( ((int (*)( int, int, int, void * ))orig_umount)
	  ( magic1, magic2, cmd, arg ) );

} /* wrap_reboot() */



/**************************************************************************
 *
 * some utility functions...
 *
 **************************************************************************/


/* handle_mediate_open()
 *
 * in:     path_s              - absolute canonical path that current
 *                               wants to open.
 *         grant_exemption_flag - if set, do not perform mediation if `path_s'
 *                               corresponds to an existing object that
 *                               LOMAC deems to be exempt from write mediation,
 *                               (such as ttys, etc.).
 * out:    nothing
 * return: value    condition
 *         -----    ---------
 *           0      caller should permit open
 *         other    namei's error codes, which caller should return
 *                  immediately without calling the open system call.  
 *
 *     This is a convenience function for wrapper functions that wrap
 * system calls which can create files based on a pathname argument.
 * Examples include wrap_open() and wrap_creat().  This function
 * encapsulates some common code that these wrapper functions all
 * need.
 *
 */

int handle_mediate_open( const char *path_s, int grant_exemption_flag ) {

  int ret_val;              /* holds function return values */
  int exempt_flag = 0;      /* set if `path_s' is exempt from mediation */
  struct dentry *p_dentry;  /* dcache entry corresponding to `path_s' */
  char *scratch_path_s;     /* dynamically allocated page for path mangling */


  /* We'll use the `scratch_path_s' buffer as scratch * 
   * space to hold the canabspath.                    */
  if( !( scratch_path_s = (char *)__get_free_page( GFP_KERNEL ) ) ) {
    return( -ENOMEM );
  }

  if( ( ret_val = predict_object_canabspath( path_s, scratch_path_s,
					     LOG_BUFFER_LENGTH ) ) ) {

    /* We were unable to predict the canabspath.  Namei() couldn't  *
     * grok `filename' for us, and it won't be able to grok         *
     * `filename' for the creat system call below.  We'll just fail *
     * here, returning namei()'s error code just like open does.    */
    goto out;

  } /* if failed to predict canabspath */

  if( grant_exemption_flag ) {
    /* OK, so now we know we have to mediate, and we have the
     * canabspath.  Before we can go ahead and mediate, we have to
     * check for one exeption.  Some user programs enjoy
     * opening/truncating existing terminal devices.  LOMAC grants
     * such devices exemptions from write mediation.  So, in order to
     * determine whether or not the current process is trying to open
     * an exempt device (the actual open hasn't happened yet,
     * remember), we must effectively open it first by looking up its
     * inode.  Once we have the inode (well, dcache entry), we can
     * check its bits to see if it is exempt.  If the inode does not
     * exist (the user program is trying to create a new file), we
     * assume that it isn't exempt.  This is safe, because exempt
     * files can be created only with sys_mknod(), and wrap_mknod()
     * calls this function with the `grant_exemption_flag' clear.       */

    if( ( p_dentry = path_to_object( scratch_path_s ) ) ) {
      if( WRITE_EXEMPT( p_dentry ) ) {
	exempt_flag = 1;    /* this file is exempt from write mediation */
      }
      object_done( p_dentry );
    } /* if file named by `scratch_path_s' exists */

  } /* if we're honoring write mediation exemptions */
    
  /* OK, if exempt_flag is still 0, then we must mediate the open. */
  if( !exempt_flag ) {

    if( mediate_open( current, scratch_path_s ) ) {
      ret_val = 0;       /* allow open */
    } else {
      ret_val = -EPERM;  /* prevent open */
    }

  } /* if non-exempt */
  
 out:
  free_page( (unsigned long)scratch_path_s );
  return( ret_val );
  
} /* handle_mediate_open() */



/* handle_monitor_read()
 *
 * in:     p_dentry   - dentry for file being read by read() or readv()
 * out:    nothing
 * return: nothing
 *
 *  This is a convenience function for wrap_read() and wrap_readv() -
 * it encapsulates some common code that they both need.
 *
 */

void handle_monitor_read( struct dentry *p_dentry ) {

#ifdef PARANOID  
  /* This function is called after the read or readv system call has   *
   * completed.  If the call failed, this function should not be       *
   * called.  Consequently, this function assumes that the read/v call *
   * succeeded, and `p_dentry' is a valid dentry.  These assertions    *
   * test this assumption.                                             */
  if( !p_dentry ) {
    PANIC( "LOMAC/K: handle_monitor_read() called on null dentry\n" );
  }
  if( !(p_dentry->d_inode) ) {
    PANIC( "LOMAC/K: handle_monitor_read() called on dentry with no inode\n" );
  }
#endif

  /* Processes use the read system calls to access a variety of        *
   * abstractions that the kernel represents with inodes.  LOMAC       *
   * considers some of these inode abstractions (like files) objects,  *
   * and others (like unnamed pipes) non-objects.  Even among those    *
   * inode abstractions that are objects, LOMAC uses different methods *
   * of determining their levels.  For example, the level of a file    *
   * object depends on its name in the filesystem, but the level of a  *
   * non-local domain socket depends on the hardware device it least   *
   * recently read from.  The following if statement                   *
   * distinguishes between all the kinds of inode abstractions and     *
   * objects so LOMAC can moderate each kind appropriately.            */

  if( IS_NETWORK_SOCKET( p_dentry ) ) {

    /* non-local domain socket */
    monitor_read_nic( current, NULL );

  } else {
    
    /* file, FIFO, unnamed pipe, UNIX (local) domain socket */
    monitor_read_object( current, p_dentry );

  }

} /* handle_monitor_read() */


/* handle_mediate_write()
 *
 * in:     p_dentry - dentry for file being written by write() or writev()
 * out:    nothing
 * return: value    condition
 *         -----    ---------
 *           0      caller should prevent write
 *           1      caller should permit write
 *
 *  This is a convenience function for wrap_write() and wrap_writev() -
 * it encapsulates some common code that they both need.
 *
 */

int handle_mediate_write( struct dentry *p_dentry ) {

#ifdef PARANOID  
  /* We're relying on the caller to ensure that `p_dentry' is valid. */
  if( !p_dentry ) {
    PANIC( "LOMAC/K: handle_mediate_write() called on null dentry\n" );
  }
  if( !(p_dentry->d_inode) ) {
    PANIC( "LOMAC/K: handle_mediate_write() "
	   "called on dentry with no inode\n" );
  }

  /* LOMAC never prevents a write to an unnamed pipe.  This function *
   * should not be called on unnamed pipes.                          */
  if( IS_PIPE( p_dentry ) ) {
    PANIC( "LOMAC/K: handle_mediate_write() called on pipe\n" );
  }
#endif


  /* Processes use the write system calls to access a variety of       *
   * abstractions that the kernel represents with inodes.  LOMAC       *
   * considers some of these inode abstractions (like files) objects,  *
   * and others (like unnamed pipes) non-objects.  Even among those    *
   * inode abstractions that are objects, LOMAC uses different methods *
   * of determining their levels.  For example, the level of a file    *
   * object depends on its name in the filesystem, but the level of a  *
   * non-local domain socket depends on the hardware device it least   *
   * recently read from.  The following nested if statement            *
   * distinguishes between all the kinds of inode abstractions and     *
   * objects so LOMAC can moderate each kind appropriately.            */

  if( IS_NETWORK_SOCKET( p_dentry ) ) {
    /* non-local domain socket.  LOMAC always allows these writes. */
    return( 1 );  /* allow write */

  } else {

    /* FIFO or file */
    if( WRITE_EXEMPT( p_dentry ) ) {
      
      /* Exempt files - we always allow writes to serial devices, *
       * ttys, pseudoterminals, /dev/null.                        */
      return( 1 ); /* allow write */

    } else {

      /* non-exempt files */

      return( mediate_write( current, p_dentry ) );

    } /* if/else exempt file */

  } /* else file */

  /* Eventually, we will handle writes on other kinds of objects here. */
  return( 1 );  /* allow... for now! ## */

} /* handle_mediate_write() */



/* handle_accept()
 *
 *
 */

int handle_accept( int call, unsigned long *args ) {

  unsigned long arg;               /* kernel-space copy of args[ 0 ] */
  int server_socket_fd;            /* server socket file descriptor */
  int session_socket_fd;           /* session socket file descriptor */
  struct dentry *p_server_socket;  /* `current' wants to accept on this */
  struct dentry *p_session_socket; /* session socket created if accept works */

  /* The server socket's fd number is in args[ 0 ].  Use it to get *
   * dentry for server socket.                                     */
  copy_from_user( &arg, args, sizeof( unsigned long ) );
  server_socket_fd = (int)arg;
  p_server_socket = fd_to_dentry( server_socket_fd );

  /* We'll proceed without checking for errors - if there were errors, *
   * the following call to the real sys_socketcall will catch them.    */
  
  /* Call original sys_socketcall().  It should return fd of session socket */
  session_socket_fd = 
    ((int (*)(int, unsigned long *))orig_socketcall)( call, args );

  /* We're concerned only with successful accepts *
   * on UNIX (local) domain sockets.              */
  if( ( session_socket_fd >= 0 ) &&
      IS_LOCAL_SOCKET( p_server_socket ) ) {

      /* If this was a successful accept() call to a UNIX-domain socket,  *
       * then the kernel has created a new session socket for handling    *
       * the accepted connection.  We must make `p_session_socket' point  *
       * to this new session socket's dentry, and set its level to match  *
       * the server socket's level.                                       */

      if( !( p_session_socket = fd_to_dentry( session_socket_fd ) ) ) {
	/* we expect no error since sys_socketcall() succeeded. */
	PANIC( "LOMAC: unable to get session socket in handle_accept.\n" );
      }
      
      monitor_unix_socket_accept_connect( current, p_server_socket,
					  p_session_socket );
  }

  return( session_socket_fd );

} /* handle_accept() */


/* handle_bind()
 *
 *
 */

int handle_bind( int call, unsigned long *args ) {

  unsigned long arg_array[ 3 ];   /* kernel-space copy of args[ 0..2 ] */
  struct dentry *p_socket_dentry; /* `current' wants to bind this socket */
  struct dentry *p_name_dentry;   /* dentry of bound name in filesystem */
  struct sockaddr_un *p_sockaddr; /* name to bind to socket */
  int sockaddr_length;            /* length of name */
  int ret_val;                    /* value returned by function calls */

  /* The arguments to SYS_BIND are as follows:
   *  args[ 0 ]  fd of socket
   *  args[ 1 ]  address of name to bind to
   *  args[ 2 ]  length of name
   */
  copy_from_user( arg_array, args, ( sizeof( unsigned long ) * 3 ) );
  sockaddr_length = (int)arg_array[ 2 ];
  p_sockaddr = (struct sockaddr_un *)arg_array[ 1 ];
  p_socket_dentry = fd_to_dentry( (int)(arg_array[ 0 ]) );

  /* If this is a bind on a UNIX (local) domain socket, and we have    *
   * good socket dentry and address pointers, and the name to bind is  *
   * in the filesystem namespace (as opposed to the abstract namespace *
   * new in the 2.2 kernels), we'll mediate.  Otherwise, we'll let     *
   * control pass directly to the kernel's original sys_socketcall.    */
  if( p_sockaddr && p_socket_dentry && IS_LOCAL_SOCKET( p_socket_dentry ) &&
      ( p_sockaddr->sun_path[ 0 ] != '\0' ) ) {

    /* Unfortunately, handle_mediate_open() doesn't take a name length  *
     * parm.  handle_mediate_open() calls predict_object_canabspath(),  *
     * which calls Linux's getname().  We should make use of the length *
     * parameter to bound the length of the path string more closely. ##*/ 
    if( ( ret_val = handle_mediate_open( p_sockaddr->sun_path, 0 ) ) ) {
      return( ret_val );
    }

  } /* if we need to mediate */
  
  /* Call original sys_socketcall(). */
  ret_val = ((int (*)(int, unsigned long *))orig_socketcall)( call, args );

  /* We're concerned only with successful binds on local domain sockets. */
  if( ( !ret_val ) && IS_LOCAL_SOCKET( p_socket_dentry ) ) {

    /* We need to monitor this bind differently depending on whether     *
     * the bound name is in the filesystem or in the abstract namespace. */
    if( p_sockaddr->sun_path[ 0 ] == '\0' ) {

      /* The binding has created a name in the abstract namespace. We *
       * must set `p_socket_dentry's level appropriately.             */
      monitor_unix_socket_abstract( current, p_socket_dentry );

    } else {

      /* The binding has created a name in some filesystem namespace, *
       * represented by its own inode.  Make `p_other_inode' point to *
       * this inode and set its level appropriately.                  */
      p_name_dentry = socket_name( p_socket_dentry );
      if( !p_name_dentry ) {
	/* we expect no failures since sys_socketcall() succeeded. */
	PANIC( "LOMAC: can't find socket peer in handle_bind.\n" );
      }

      monitor_unix_socket_bind( current, p_socket_dentry, p_name_dentry );

    } /* if/else abstract/filesystem namespace */

  } /* if we need to monitor bind */

  return( ret_val );

} /* handle_bind() */


/* handle_connect()
 *
 *
 */

int handle_connect( int call, unsigned long *args ) {

  unsigned long arg;               /* kernel-space copy of args[ 0 ] */
  int client_socket_fd;            /* client socket file descriptor */
  struct dentry *p_client_socket;  /* `current' wants to connect on this */
  struct dentry *p_session_socket; /* session socket for connection */
  int ret_val;                     /* value returned by function calls */

  /* The client socket's fd number is in args[ 0 ].  Use it to get *
   * dentry for client socket.                                     */
  copy_from_user( &arg, args, sizeof( unsigned long ) );
  client_socket_fd = (int)arg;
  p_client_socket = fd_to_dentry( client_socket_fd );

  /* We'll proceed without checking for errors - if there were errors, *
   * the following call to the real sys_socketcall will catch them.    */
  
  /* Call original sys_socketcall().  It should return fd of session socket */
  ret_val = ((int (*)(int, unsigned long *))orig_socketcall)( call, args );

  /* We're concerned only with successful connects on local domain sockets. */
  if( ( !ret_val ) && IS_LOCAL_SOCKET( p_client_socket ) ) {

      /* sys_connect() links the client's socket to the socket      *
       * created by sys_accept() in the server.  The implementation *
       * of the link has changed a bit from the 2.0 Linux kernels.  *
       * We must set the level of the client's socket to match the  *
       * level of the server's session socket that was created by   *
       * sys_accept().                                              */
      p_session_socket = socket_peer( p_client_socket );
      if( !p_session_socket ) {

	/* this indicates a connect to a socket bound to an abstract  *
	 * name, rather than a name in the filesystem namespace.  We  *
	 * (mis)handle these connects differently from normal         *
	 * connects.  It would probably be better to detect this case *
	 * by examining the address parameter.                        */
	monitor_unix_socket_abstract( current, p_client_socket );

      } else {

	monitor_unix_socket_accept_connect( current, p_session_socket,
					    p_client_socket );
      }

  } /* if we must monitor connect */

  return( ret_val );

} /* handle_connect() */


/* handle_recv()
 *
 *
 */

int handle_recv( int call, unsigned long *args ) {

  unsigned long arg;            /* kernel-space copy of args[ 0 ] */
  int socket_fd;                /* socket file descriptor */
  struct dentry *p_socket;      /* `current' wants to recv on this socket */
  int ret_val;                  /* value returned by function calls */

  /* The client socket's fd number is in args[ 0 ].  Use it to get *
   * dentry for socket.                                            */
  copy_from_user( &arg, args, sizeof( unsigned long ) );
  socket_fd = (int)arg;
  p_socket = fd_to_dentry( socket_fd );

  /* We'll proceed without checking for errors - if there were errors, *
   * the following call to the real sys_socketcall will catch them.    */
  
  /* Call original sys_socketcall().  It should return fd of session socket */
  ret_val = ((int (*)(int, unsigned long *))orig_socketcall)( call, args );

  /* We're concerned only with successful recv's on local domain sockets. */
  if( ( ret_val >= 0 ) && IS_LOCAL_SOCKET( p_socket ) ) {
      handle_monitor_read( p_socket );
  } /* if we must monitor recv */

  return( ret_val );

} /* handle_recv() */


/* handle_socketpair()
 *
 * This code sets all UNIX-domain sockets created with socketpair to
 * LOMAC_LOWEST_LEVEL.  This is a poor solution, since it prevents
 * high-level processes from having high-level socketpairs.
 *
 *
 */

int handle_socketpair( int call, unsigned long *args ) {

  unsigned long arg_array[ 4 ];         /* kernel-space copy of args */
  int *sock_fds;                        /* fds for sockets in pair */
  struct dentry *socket_dentry[ 2 ];    /* sockets in pair */
  int ret_val;                          /* value returned by function calls */

  /* Call original sys_socketcall(). */
  ret_val = ((int (*)(int, unsigned long *))orig_socketcall)( call, args );

  /* Return immediately if the kernel failed to create a socket pair */
  if( ret_val ) {
    return( ret_val );
  }

  /* The arguments for SYS_SOCKETCALL are as follows:
   * args[ 0 ] family
   * args[ 1 ] type
   * args[ 2 ] protocol
   * args[ 3 ] int * to the first of two ints.  The kernel's original
   * SYS_SOCKETPAIR code writes the fds for the new pair of sockets
   * to this location.
   */

  copy_from_user( arg_array, args, ( sizeof( unsigned long ) * 4 ) );
  
  /* we need to monitor only UNIX (local) domain sockets.  Return *
   * if this is not a UNIX (local) domain socketpair.             */
  if( (int)(arg_array[ 0 ]) != AF_UNIX ) {
    return( ret_val );
  }

  sock_fds = (int *)(arg_array[ 3 ]);
  socket_dentry[ 0 ] = fd_to_dentry( sock_fds[ 0 ] );
  socket_dentry[ 1 ] = fd_to_dentry( sock_fds[ 1 ] );
#ifdef PARANOID
  if( !( socket_dentry[ 0 ] && socket_dentry[ 1 ] ) ) {
    PANIC( "LOMAC/K: null socket dentries in handle_socketpair.\n" );
  }
#endif

  monitor_unix_socketpair( current, socket_dentry[ 0 ], socket_dentry[ 1 ] );  

  return( ret_val );

} /* handle_socketpair() */

