/***************************************************************************
 *
 * kernel_util.c
 *
 * LOMAC - Low Water-Mark Mandatory Access Control for Linux 
 * Copyright (C) 1999, 2000 NAI Labs
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU General Public
 * License as published by the Free Software Foundation.  This program
 * is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.  You should have received a copy of the
 * GNU General Public License along with this program; if not, write
 * to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 *
 * This file contains some kernel-specific utility functions used by 
 * routines in other kernel_*.c files.  These utility functions are
 * kernel-specific - they should not contains any lomac-specific
 * types or function calls.
 *
 ***************************************************************************/

#include "kernel_util.h"

#include <linux/file.h>    /* for fcheck() */
#include <linux/net.h>     /* to support net/sock.h */
#include <net/sock.h>      /* for struct sock, struct unix_opt */
#include <asm/uaccess.h>   /* for copy_from_user() */

/* pid_to_task()
 *
 * in:     pid - pid of the task to find
 * out:    nothing
 * return: pointer to task struct corresponding to `pid' or NULL if not found
 *
 * This is a module-friendly replacement of the find_task_by_pid macro
 * in linux/sched.h .
 *
 */

struct task_struct *pid_to_task( int pid ) {

  struct task_struct *p = current;

  do {
    if( p->pid == pid ) {
      return( p );
    }
    p = p->next_task;
  } while( p != current );

  return( NULL );

} /* pid_to_task() */


/* pgrp_to_task()
 *
 * in:     pgrp - pgrp of the task to find
 * out:    nothing
 * return: pointer to task struct corresponding to `pgrp' or NULL if not found
 *
 * Similar to pid_to_task(), but finds first occurrence of task with the
 * specified `pgrp' instead of pid.
 *
 */

struct task_struct *pgrp_to_task( int pgrp ) {

  struct task_struct *p = current;

  do {
    if( p->pgrp == pgrp ) {
      return( p );
    }
    p = p->next_task;
  } while( p != current );

  return( NULL );

} /* pgrp_to_task() */
    

/* fd_to_dentry()
 *
 * in:     fd
 * out:    nothing
 * return: pointer to inode corresponding to `fd' in the `current' process
 *         or NULL if there is no such inode.
 *
 */

struct dentry *fd_to_dentry( unsigned long fd ) {

  struct file *p_file;    

  /* Make `p_file' point to the file corresponding to `fd'.  Like *
   * sys_fstat(), we're not going to use fget().  We just want a  *
   * brief atomic peek at the dentry, so we're not going to fool  *
   * with all that reference-counting fget() and fput() jazz.     */

  /* fcheck() doesn't check to see if `current's `files' pointer is  *
   * NULL.  Some kernel procs have a NULL pointer there.  So, we use *
   * fcheck_task(), which does the check, even though it requires    *
   * more parameters.                                                */
  if( ( p_file = fcheck_task( current, (unsigned int)fd ) ) ) {
    return( p_file->f_dentry );
  }

  return( NULL );

} /* fd_to_dentry() */


/* socket_peer()
 *
 * in:     p_socket - one of a pair of connected UNIX (local) domain sockets.
 * out:    nothing
 * return: the other socket in the pair, or NULL if it cannot be found.
 *
 * Given one UNIX (local) domain socket in a connected pair, this function
 * will find the other socket.  Note that the sockets are represented
 * by their dentries.
 *
 */

struct dentry *socket_peer( struct dentry *p_socket_dentry ) {

  struct socket *p_socket;        /* points to `p_socket_dentry's socket */
  struct sock   *p_sock;          /* points to `p_socket's sock struct   */

#ifdef PARANOID
  if( !p_socket_dentry ) {
    panic( "LOMAC: socket_peer called on null socket.\n" );
  }
  if( !( IS_LOCAL_SOCKET( p_socket_dentry ) ) ) {
    panic( "LOMAC: socket_peer called on non-UNIX-domain socket.\n" );
  }
#endif

  if( !( p_socket_dentry->d_inode ) ) {
    return( NULL );
  }
  p_socket = &(p_socket_dentry->d_inode->u.socket_i);

  if( !p_socket ) {
    return( NULL );
  }
  p_sock = (struct sock *)p_socket->sk;

  if( !( p_sock ) ) {
    return( NULL );
  }
  if( !( p_sock->pair ) ) {
    return( NULL );
  }
  return( p_sock->pair->protinfo.af_unix.dentry );

} /* socket_peer() */


/* socket_name()
 *
 * in:     p_socket - a UNIX (local) domain socket that is bound to a name.
 * out:    nothing
 * return: the dentry of the name of the socket in the filesystem, or
 *         NULL if this cannot be determined.
 *
 * Given a UNIX (local) domain socket that is bound to a name in the
 * filesystem, this function will find the dentry corresponding to the
 * name.
 *
 */

struct dentry *socket_name( struct dentry *p_socket_dentry ) {

  struct socket *p_socket;        /* points to `p_socket_dentry's socket */
  struct sock   *p_sock;          /* points to `p_socket's sock struct   */

#ifdef PARANOID
  if( !p_socket_dentry ) {
    panic( "LOMAC: socket_peer called on null socket.\n" );
  }
  if( !( IS_LOCAL_SOCKET( p_socket_dentry ) ) ) {
    panic( "LOMAC: socket_peer called on non-UNIX-domain socket.\n" );
  }
#endif

  if( !( p_socket_dentry->d_inode ) ) {
    return( NULL );
  }
  p_socket = &(p_socket_dentry->d_inode->u.socket_i);

  if( !p_socket ) {
    return( NULL );
  }
  p_sock = (struct sock *)p_socket->sk;

  if( !( p_sock ) ) {
    return( NULL );
  }
  return( p_sock->protinfo.af_unix.dentry );

} /* socket_name() */


/* copy_sockadd_to_kernel()
 *
 * in:     p_user_sockaddr   - pointer to sockaddr in user-space
 *         length            - length of sockaddr
 * out:    p_kernel_sockaddr - `*p_user_sockaddr' copied here
 * return: value      condition
 *         -----      ---------
 *           0        success
 *         -EINVAL    `length' > MAX_SOCK_ADDR
 *         -EFAULT    copyin failed
 *
 * This function is a copy of move_addr_to_kernel() from
 * linux/net/socket.c - a useful function that is unfortunately
 * not exported to modules.
 */

int copy_sockaddr_to_kernel( void *p_user_sockaddr, int length, 
			     void *p_kernel_sockaddr ) {

  if( ( length < 0 ) || ( length > MAX_SOCK_ADDR ) ) {
    return( -EINVAL );
  }
  if( length == 0 ) {
    return( 0 );
  }
  if( copy_from_user( p_kernel_sockaddr, p_user_sockaddr, length ) ) {
    return( -EFAULT );
  }
  return( 0 );

} /* copy_sockaddr_to_kernel() */


