/***************************************************************************
 *
 * kernel_interface.c
 *
 * LOMAC - Low Water-Mark Mandatory Access Control for Linux 
 * Copyright (C) 1999, 2000 NAI Labs
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU General Public
 * License as published by the Free Software Foundation.  This program
 * is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.  You should have received a copy of the
 * GNU General Public License along with this program; if not, write
 * to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 *
 * This file contains routines that provide LOMAC's high-level
 * routines with a kernel-independent interface to several important
 * kinds of kernel objects.  This level of abstraction is of practical
 * importance to LOMAC, because ongoing kernel developments change the
 * way some kernel objects are accessed over time.  When such changes
 * occur, this file can encapsulate the change, leaving the
 * higher-level routines as they were.
 *
 ***************************************************************************/

#include "kernel_interface.h"
#include "kernel_util.h"
#include "lomac_level.h"
#include "lomac_log.h"

/* local utility functions */
struct dentry *kernel_namei( const char *path_s );


/***********************************************************************
 * Types and routines to map the LOMAC notion of "subject" onto the
 * kernel notion of process groups ("jobs").
 ***********************************************************************/

/* The following constants are meant to coexist with the PF_* constants *
 * defined in sched.h.  All these flags are meant for task_struct's     *
 * `flags' field.                                                       */
#define SUBJECT_LEVEL_1    0x00010000
#define SUBJECT_LEVEL_2    0x00020000
#define SUBJECT_LEVEL_BITS ( SUBJECT_LEVEL_1 | SUBJECT_LEVEL_2 )

/* get_subject_id()
 * 
 * in:     p_subject - pointer to subject whose ID we wish to know
 *         id_s      - pointer to a character buffer to hold ID string.
 *         bufflen   - number of chars available for non-terminator
 *                     chars in `id_s'.
 * out:    id_s      - buffer will contain null-terminated ID string
 * return: the number of non-terminator characters written to `id_s'.
 *
 *     This function writes a string describing the identity of
 * the subject indicated by `p_subject' to `id_s'.  This string
 * is guaranteed to be no more than `bufflen' non-terminator characters
 * long.
 *
 * Currently the ID string is a concatenation of the following process
 * attributes: pid, pgrp, uid, and the name of the executable the 
 * process is running.  This string is for logging purposes only; 
 * higher-level LOMAC functions do not interpret the contents of this
 * string.
 *
 */

int get_subject_id( lomac_subject_t *p_subject, char *id_s, int bufflen ) {

  char ascii[ MAX_DIGITS ];  /* holds the ASCII representation of ints */
  int component_length;      /* length of component parts of ID string */
  int space_left;            /* portion of `bufflen' used so far... */

#ifdef PARANOID
  if( !p_subject ) {
    PANIC( "LOMAC/K: null subject pointer passed to get_subject_id()\n" );
  }
  if( !id_s ) {
    PANIC( "LOMAC/K: null id_s pointer passed to get_subject_id()\n" );
  }
  if( bufflen <= 0 ) {
    PANIC( "LOMAC/K: non-positive buffer length "
	   "passed to get_subject_id()\n" );
  }
#endif


  space_left = bufflen;

  /* Append pid. */
  if( space_left < 1 ) {
    goto out;
  }
  STRNCAT( id_s, "p", 1 );
  space_left--;
  TOASCII( p_subject->pid, ascii );
  component_length = STRNLEN( ascii, MAX_DIGITS );
  if( component_length > space_left ) {
    goto out;
  }
  STRNCAT( id_s, ascii, component_length );
  space_left -= component_length;

  /* Append pgid. */
  if( space_left < 1 ) {
    goto out;
  }
  STRNCAT( id_s, "g", 1 );
  space_left--;
  TOASCII( p_subject->pgrp, ascii );
  component_length = STRNLEN( ascii, MAX_DIGITS );
  if( component_length > space_left ) {
    goto out;
  }
  STRNCAT( id_s, ascii, component_length );
  space_left -= component_length;

  /* Append uid. */
  if( space_left < 1 ) {
    goto out;
  }
  STRNCAT( id_s, "u", 1 );
  space_left--;
  TOASCII( p_subject->uid, ascii );
  component_length = STRNLEN( ascii, MAX_DIGITS );
  if( component_length > space_left ) {
    goto out;
  }
  STRNCAT( id_s, ascii, component_length );
  space_left -= component_length;

  /* Append program name.  We use the comm field of task_struct to get *
   * the program name.  This gives us only a limited number of         *
   * characters.  The PROC_PID_EXE case in linux/fs/proc/link.c shows  *
   * how to get the actual dentry for the exe file.  Perhaps it would  *
   * be better to use this method in the future.                       */
  if( space_left < 1 ) {
    goto out;
  }
  STRNCAT( id_s, ":", 1 );
  space_left--;
  component_length = STRNLEN( p_subject->comm, space_left );
  STRNCAT( id_s, p_subject->comm, component_length );
  space_left -= component_length;

 out:

#ifdef PARANOID

  /* We should have used up to `bufflen' chars, but no more. */
  if( ( bufflen - space_left ) < 0 ) {
    PANIC( "LOMAC/K: id too long in get_subject_id()\n" );
  }

  /* All chars from `id_s[ 0 ]' to `id_s[ bufflen - space_left - 1 ]' *
   * should be non-null.  `id_s[ bufflen - space_left ]' should be    *
   * null.  We're re-using component_length here as an index offset   *
   * counter just to avoid declaring another variable.                */
  for( component_length = 0; component_length < ( bufflen - space_left );
       component_length++ ) {
    if( id_s[ component_length ] == '\0' ) {
      PANIC( "LOMAC/K: prematurely terminated id in get_subject_id()\n" );
    }
  }
  if( id_s[ bufflen - space_left ] != '\0' ) {
      PANIC( "LOMAC/K: improperly terminated id in get_subject_id()\n" );
  }

#endif

  return( bufflen - space_left );

} /* get_subject_id() */


/* set_subject_level()
 * 
 * in:     p_subject - pointer to subject whose level we want to set
 *         level     - level information we desire to set
 * out:    see description of side-effects, below
 * return: nothing
 *
 * This routine sets the level of `p_subject' to `level'.  The
 * high-level LOMAC routines use `p_subject' to refer to "a subject" -
 * a conceptually singular entity equivalent to the kernel's notion of
 * process group.  Here, where kernel-specific details are visible, we
 * know that `p_subject' is really a pointer to a single process in
 * the process group.  Consequently, in order to set the level of
 * "the" subject, we will actually set the levels of all the processes
 * in `p_subject's process group.  Once done, we can "get" the level
 * of the process group by examining any individual process in that
 * group.
 *
 * This version of LOMAC stores subject level information by shoe-horning
 * its level bits into an unused part of the struct task_struct's
 * `flags' field.  When you call this routine, it will change the `flags'
 * fields of all the processes in `p_subject's process group to reflect
 * the new level.
 */

void set_subject_level( lomac_subject_t *p_subject, level_t level ) {

  struct task_struct *p_proc;   /* iterates through procs in pgrp */

#ifdef PARANOID
  level_t paranoid_level;       /* holds levels during paranoid assertions */

  if( !p_subject ) {
    PANIC( "LOMAC/K: null subject pointer passed to set_subject_level()\n" );
  }
#endif

  /* Set the levels of each process in `p_subject's pgrp */
  p_proc = p_subject;   /* start with the `p_subject' process */
  do {

    /* We're iterating through the entire process table, but we're   *
     * modifying only those processes that are in `p_subject's pgrp. */
    if( p_proc->pgrp == p_subject->pgrp ) {

      switch( level ) {
      case LOMAC_HIGHEST_LEVEL:
	p_proc->flags &= ~SUBJECT_LEVEL_BITS;
	p_proc->flags |= SUBJECT_LEVEL_2;
	break;
      case LOMAC_LOWEST_LEVEL:
	p_proc->flags &= ~SUBJECT_LEVEL_BITS;
	p_proc->flags |= SUBJECT_LEVEL_1;
	break;
      default:
#ifdef PARANOID
	PANIC( "LOMAC/K: attempted to set invalid level for p%ug%u", 
		p_proc->pid, p_proc->pgrp );
#endif /* PARANOID */
      }

    } /* if same pgrp */

    /* move to next process in task list */
    p_proc = p_proc->next_task;

  } while( p_proc != p_subject );  /* for all processes */

#ifdef PARANOID
  /* All processes in `p_subject's process group should be at level `level'. */
  p_proc = p_subject;
  do {
    if( p_proc->pgrp == p_subject->pgrp ) {

      get_subject_level( p_proc, &paranoid_level );
      if( !( LEVEL_LEQ( level, paranoid_level ) &&
	     LEVEL_LEQ( paranoid_level, level ) ) ) {
	PANIC( "LOMAC/K: set_subject_level not consistent across pgrp\n" );
      }

    }
    p_proc = p_proc->next_task;
  } while( p_proc != p_subject );
#endif

} /* set_subject_level() */


/* get_subject_level()
 *
 * in:     p_subject - pointer to subject whose level we wish to know
 * out:    p_level   - level of `p_subject' will be written here
 * return: nothing
 *
 */

void get_subject_level( lomac_subject_t *p_subject, level_t *p_level ) {

#ifdef PARANOID
  if( !p_subject ) {
    PANIC( "LOMAC/K: null subject pointer passed to get_subject_level()\n" );
  }
  if( !p_subject ) {
    PANIC( "LOMAC/K: null level pointer passed to get_subject_level()\n" );
  }
#endif

  switch( p_subject->flags & SUBJECT_LEVEL_BITS ) {
  case SUBJECT_LEVEL_2:
    *p_level = LOMAC_HIGHEST_LEVEL;
    break;
  case SUBJECT_LEVEL_1:
    *p_level = LOMAC_LOWEST_LEVEL;
    break;
  default:
    *p_level = LOMAC_INVALID_LEVEL;
#ifdef PARANOID
    printk( "LOMAC/K: found invalid level for p%ug%u\n", 
	    p_subject->pid, p_subject->pgrp );
#endif
  }

} /* get_subject_level() */



/***********************************************************************
 * Types and routines to map the LOMAC notion of "object" to the
 * kernel notion of dcache entries.
 ***********************************************************************/

/* The following constants are meant to coexist with the DCACHE flags  *
 * defined in dcache.h.  These flags are all supposed to reside in the *
 * d_flags fields of the dentry structure.                             */

#define OBJECT_LEVEL_1    0x00010000
#define OBJECT_LEVEL_2    0x00020000
#define OBJECT_LEVEL_BITS ( OBJECT_LEVEL_1 | OBJECT_LEVEL_2 )

int get_abscan_path( struct dentry *p_leaf_dentry, char *path_s );


/* get_object_id()
 *
 * in:     p_object - pointer to object whose identity we wish to know
 *         id_s     - buffer to contain path corresponding to `p_object'.
 *         bufflen  - length of `id_s' buffer.
 * out:    id_s     - will contain absolute canonical path for `p_object',
 *                    or a string constructed from its device and inode
 *                    numbers if there is no path corresponding to 
 *                    `p_object'. 
 * return: number of non-terminator characters written to `id_s'.
 *
 *     This function appends a string identifying `p_object' to
 * `id_s'.  The string will either be an absolute canonical path, or
 * for those objects that have no such path, it will be a
 * concatenation of device major number, device minor number, and
 * inode number.  This string is for logging purposes only;
 * higher-level LOMAC functions do not interpret the contents of this
 * string.
 *
 */

int get_object_id( lomac_object_t *p_object, char *id_s, int bufflen ) {

#ifdef PARANOID
  if( !p_object ) {
    PANIC( "LOMAC/K: null object pointer passed to get_object_id()\n" );
  }
  if( !id_s ) {
    PANIC( "LOMAC/K: null id_s pointer passed to get_object_id()\n" );
  }
  if( bufflen <= 0 ) {
    PANIC( "LOMAC/K: non-positive buffer length passed to get_object_id()\n" );
  }
#endif


  /* If the dcache contains a path, then copy it to `id_s', otherwise *
   * construct an id string from the corresponding inode's device     *
   * and inode numbers.                                               *
   * We shouldn't need the IS_PIPE thing, but it seems that linux     *
   * pipes have paths of "/" in the dcache.  Go figure.               *
   * Same goes for IS_SOCKET.                                         */

  if( IS_PIPE( p_object ) || IS_SOCKET( p_object ) ||
      get_object_canabspath( p_object, id_s, bufflen ) ) {

    /* If there isn't enough room for a maximally long ID, just give up. */
    if( bufflen < MAX_DIGITS * 3 ) {
      return( 0 );
    }

    /* No path for this object, make up an id based on numbers... */
    sprintf( id_s, "D%ud%ui%lu", 
	     MAJOR(p_object->d_inode->i_dev), 
	     MINOR(p_object->d_inode->i_dev),
	     p_object->d_inode->i_ino );

  }

#ifdef PARANOID
  /* We should have used no more than `bufflen-1' non-null characters. */
  if( STRNLEN( id_s, bufflen ) >= bufflen ) {
    PANIC( "LOMAC/K: object id too long in get_object_id()\n" );
  }
#endif

  return( STRNLEN( id_s, bufflen ) );

} /* get_object_id() */


/* get_object_canabspath()
 *
 * in:     p_object - pointer to object whose canabspath we want.
 *         path_s   - buffer to contain path corresponding to `p_object'.
 *         bufflen  - length of `path_s' buffer.
 * out:    path_s   - will contain absolute canonical path for `p_object'
 *                    if such a path exists.  If there is no such path,
 *                    `path_s' will not be modified.
 * return: value    condition
 *         -----    ---------
 *           0      `path_s' contains the canabspath of `p_object'.
 *           1      there is no canabspath for `p_object'.
 *
 *  This function is used to get the canonical absolute paths of
 *  objects, in order to facilitate lookups in the file system object
 *  -> level mapping.  This function works only on objects which
 *  exist.  Use predict_object_canabspath() for non-existent objects
 *  which are candidates for creation.
 */

int get_object_canabspath( lomac_object_t *p_object, char *path_s,
			   int bufflen ) {

#ifdef PARANOID
  if( !p_object ) {
    PANIC( "LOMAC/K: null object pointer "
	   "passed to get_object_canabspath()\n" );
  }
  if( !path_s ) {
    PANIC( "LOMAC/K: null path_s pointer "
	   "passed to get_object_canabspath()\n" );
  }
  if( bufflen <= 0 ) {
    PANIC( "LOMAC/K: non-positive buffer "
	   "length passed to get_object_canabspath()\n" );
  }
#endif

  if( p_object->d_name.name && p_object->d_name.name[ 0 ] != '\0' ) {

    strncpy( path_s, d_path( p_object, path_s, bufflen ), bufflen );
    return( 0 );

  } else {

    return( 1 );

  }

} /* get_object_canabspath() */


/* predict_object_canabspath()
 *
 * in:     path_s       - path to object that may or may not yet exist;
 *                        path may or may not be canonical or absolute;
 *                        path may or may not be valid.
 *         len          - length of `canabspath_s' buffer.
 * out:    canabspath_s - absolute path in canonical form corresponding
 *                        to `path_s'.
 * return: 0 on success, else kernel namei()'s error codes.
 *
 * This function is used to predict the absolute, canonical-form
 * pathnames of objects that are about to be opened, based on the
 * not-necessarily absolute, canonical-form, or valid paths supplied
 * by the calling process.  Due to the nature of the operating system,
 * given a particular `path_s', there are three cases:
 *
 * 1)  `path_s' describes an existing object in the file system.
 *     In this case, we convert `path_s' into absolute, canonical form,
 *     and output this new canonical-form path out in `canabspath_s'.
 * 2)  `path_s' does not describe an existing object in the file system,
 *     but `path_s' with its last component removed describes a directory.
 *     This case may occur when the calling process wants to create a
 *     new filesystem object in that directory.  In this case, we
 *     convert the directory's path into absolute canonical form, append
 *     the last component of `path_s', and output this in `canabspath_s'.
 * 3)  `path_s' does not describe any object in the filesystem, and when
 *     its last component is removed, it still does not describe a
 *     directory in the filesystem.  In this case, we return an error
 *     code, following the normal error behavior of the open (etc.) 
 *     system calls.
 *
 */

int predict_object_canabspath( const char *path_s, char *canabspath_s, 
			       int len ) {

  char *scratch_path_s;    /* dynamically allocated page for path-mangling */
  struct dentry *p_dentry;            /* points to dcache entries (inodes) */
  int null_index;                   /* index of '\0' that terminates paths */
  int ret_val;                                 /* holds error return codes */

  /* `first_char_index' is the index of the first non-"/" char of the    *
   * last component in `path_s'.  `delimiter_index' is the index of      *
   * the '/' or '\0' character that terminates `path_s's last component. *
   * Example:                                                            *
   * for path_s = "/a///ok//                                             *
   *               012345678                                             *
   * first_char_index = 5 and delimiter_index = 7.                       */
  int first_char_index;
  int delimiter_index;

  
#ifdef PARANOID
  if( !path_s ) {
    PANIC( "LOMAC/K: predict_object_canabspath() given null path_s.\n" );
  }
  if( !canabspath_s ) {
    PANIC( "LOMAC/K: predict_object_canabspath() given null canabspath_s.\n" );
  }
#endif

  /* Get a dynamically allocated page of scratch kernel memory from *
   * the kernel and copyin `path_s' into it.                        */
  scratch_path_s = getname( path_s );
  if( IS_ERR( scratch_path_s ) ) {
    return( PTR_ERR( scratch_path_s ) );
  }
  
  /* Check for case 1: `path_s' refers to a valid filesystem object. */
  p_dentry = kernel_namei( scratch_path_s );
  ret_val = PTR_ERR( p_dentry );
  if( !IS_ERR( p_dentry ) ) {
    
    /* `path_s' does indeed refer to a valid object.  Place the         *
     * absolute canonical form version of `path_s' into `canabspath_s'. */
    strncpy( canabspath_s, d_path( p_dentry, canabspath_s, MAX_PATH_LEN ),
	     MAX_PATH_LEN ); 
    dput( p_dentry );  /* we're done with `p_dentry', thank you. */
    ret_val = 0;       /* OK! */
    goto out;

  }

  /* Sometimes we arrive here because the file described by `path_s'   *
   * exists, but the `current' process does not have (UNIX) permission *
   * to examine it.  In this case, return namei's eperm error code.    */
  if( ret_val == -EPERM ) {
    goto out;
  }


  /* If we reach here, we do not have a case 1 situation.  Check for  *
   * case 2 by removing last component of `scratch_path_s' and trying *
   * namei() again.                                                   */

  null_index = STRNLEN( scratch_path_s, MAX_PATH_LEN );
  first_char_index = delimiter_index = null_index;

  /* First, remove any trailing slashes.  As a special case, we don't  *
   * want to remove a slash at index 0, since this represents the root *
   * directory.                                                        */
  for( null_index--; /* start with char to the left of null terminator */
       ( null_index > 0 && scratch_path_s[ null_index ] == '/' ); 
       null_index-- ) {
    scratch_path_s[ null_index ] = '\0';
    delimiter_index = null_index;
  }

  /* Next, remove any non-"/" chars until we find another "/".  It's OK   *
   * to remove every remaining character in the string - and empty        *
   * string represents the `current' process's current working directory. */
  for( ;               /* start with the same char previous loop ended on */
       ( null_index >= 0 && scratch_path_s[ null_index ] != '/' ); 
       null_index-- ) {
    scratch_path_s[ null_index ] = '\0';
    first_char_index = null_index;  /* move `first_char_index' left as we go */
  }

  /* Finally, remove any slashes that separated the last component     *
   * from the second-to-last component.  As a special case, we don't   *
   * want to remove a slash at index 0, since this represents the root *
   * directory.                                                        */
  for( ;            /* start with the same char previous loop ended on */
       ( null_index > 0 && scratch_path_s[ null_index ] == '/' ); 
       null_index-- ) {
    scratch_path_s[ null_index ] = '\0';
  }

  /* Last component removed.  Now try kernel_namei() again. */
  p_dentry = kernel_namei( scratch_path_s );
  ret_val = PTR_ERR( p_dentry );
  if( !IS_ERR( p_dentry ) ) {
  
    /* The path in the `scratch_path_s' scratch space does indeed refer *
     * to a valid filesystem object.  Is it a directory?                */
    if( S_ISDIR( p_dentry->d_inode->i_mode ) ) {

      /* Yes, the path does refer to a directory.  Convert it into   *
       * absolute, canonical form, append the last component of      *
       * `path_s' to it, and copy it into `canabspath_s' for output. */

      /* Start by putting canabspath for `p_dentry' into scratch_path_s. */
      strncpy( scratch_path_s, 
	       d_path( p_dentry, scratch_path_s, MAX_PATH_LEN ),
	       MAX_PATH_LEN ); 

      /* Put a "/" between `p_dentry's path and the last component *
       * from path_s that we'll be adding in the next step.        *
       * Exception: we don't need to do this if the path in        *
       * `scratch_path_s' is the root directory ("/").             */
      if( !( scratch_path_s[ 0 ] == '/' &&
	     scratch_path_s[ 1 ] == '\0' ) ) {
	STRNCAT( scratch_path_s, "/", 1 );
      }
 
      /* Append last component of `path_s'. */
      STRNCAT( scratch_path_s, &(path_s[ first_char_index ]),
	       (delimiter_index - first_char_index ) );

      /* Copy result from scratch space to `canabspath_s' for output */
      strncpy( canabspath_s, scratch_path_s, len );
      ret_val = 0;   /* OK! */

    } else {

      /* Although the shortened path in `scratch_path_s' is valid, it *
       * does not describe a directory.  Return the appropriate error *
       * code.                                                        */
      ret_val = -ENOTDIR;

    } /* if/else `p_dentry' is a directory */

    dput( p_dentry );                          /* all done, thanks! */
    goto out;

  } /* if the shortened path in `scratch_path_s' refers to a valid FS object */

  /* If we reach here, we have a case 3 situation.  Return error code. */

 out:
  putname( scratch_path_s );        /* give back our scratch page */
  return( ret_val );

} /* predict_object_canabspath() */



/* set_object_level()
 *
 * in:     p_object - object whose level we want to set.
 *         level    - new level for p_object.
 * out:    p_object - bits will be twiddled to reflect new level.
 * return: nothing
 * 
 * This version of LOMAC stores object levels by making use of some
 * extra bits in the `d_flags' field of struct dentry.
 *
 */

void set_object_level( lomac_object_t *p_object, level_t level ) {

#ifdef PARANOID
  if( !p_object ) {
    PANIC( "LOMAC/K: null object pointer passed to set_object_level()\n" );
  }
#endif

  switch( level ) {
  case LOMAC_HIGHEST_LEVEL:
    p_object->d_flags &= ~OBJECT_LEVEL_BITS;
    p_object->d_flags |= OBJECT_LEVEL_2;
    break;
  case LOMAC_LOWEST_LEVEL:
    p_object->d_flags &= ~OBJECT_LEVEL_BITS;
    p_object->d_flags |= OBJECT_LEVEL_1;
    break;
  default:
#ifdef PARANOID
    printk( "LOMAC/K: attempted to set invalid level for D%ud%ui%lu", 
	    MAJOR(p_object->d_inode->i_dev),
	    MINOR(p_object->d_inode->i_dev),
	    p_object->d_inode->i_ino );
#endif /* PARANOID */
  }

} /* set_object_level() */


/* get_object_level()
 *
 * in:     p_object - the object whose level we wish to know.
 * out:    p_level  - `p_object's level will go here.
 * return: nothing
 *
 */

void get_object_level( lomac_object_t *p_object, level_t *p_level ) {

#ifdef PARANOID
  if( !p_object ) {
    PANIC( "LOMAC/K: null object pointer passed to get_object_level()\n" );
  }
  if( !p_level ) {
    PANIC( "LOMAC/K: null level pointer passed to get_object_level()\n" );
  }
#endif

  switch( p_object->d_flags & OBJECT_LEVEL_BITS ) {
  case OBJECT_LEVEL_2:
    *p_level = LOMAC_HIGHEST_LEVEL;
    break;
  case OBJECT_LEVEL_1:
    *p_level = LOMAC_LOWEST_LEVEL;
    break;
  default:
    *p_level = LOMAC_INVALID_LEVEL;
#ifdef PARANOID
    printk( "LOMAC/K: found invalid level for D%ud%ui%lu\n", 
	    MAJOR(p_object->d_inode->i_dev),
	    MINOR(p_object->d_inode->i_dev),
	    p_object->d_inode->i_ino );
#endif
  }

} /* get_object_level() */


/* get_subject_program_object()
 *
 * in:     p_subject - subject whose program object we want
 * out:    nothing
 * return: object `p_subject' is using for program text, or NULL on error.
 *
 * This function effectively returns the dentry * of the file the process
 * is using for program text.  It is based on the code in the PROC_PID_EXE
 * case of proc_follow_link() in linux/fs/proc/link.c .
 *
 * When this function returns an object (dentry), it will increment
 * that object's reference count.  The caller must eventually call
 * object_done() on that object.
 *
 */

lomac_object_t *get_subject_program_object( lomac_subject_t *p_subject ) {

  struct vm_area_struct *p_vma; /* used to iterate through list of vm areas */

  if( !p_subject->mm ) {
    return( NULL );
  }

  /* Iterate through `p_subject's list of vm areas looking *
   * for the one that describes `p_subject's program text. */
  for( p_vma = p_subject->mm->mmap; p_vma; p_vma = p_vma->vm_next ) {
    if( ( p_vma->vm_flags & VM_EXECUTABLE ) && p_vma->vm_file ) {
      return( dget( p_vma->vm_file->f_dentry ) );
    }
  }

  return( NULL );

} /* get_subject_program_object() */


/* path_to_object()
 *
 * in:     path_s - path whose object we wish to find (a kernel-space string)
 * out:    nothing
 * return: pointer to the object corresponding to `path_s', or NULL if
 *         no such object exists. 
 *
 * This function is an interface to kernel_namei(), which is an
 * interface to the kernel's namei() function that takes its argument
 * from kernel-space rather than user-space.
 *
 * This function has simplified return value behavior - it returns
 * NULL if no object could be found, for any reason.
 *
 * If the object corresponding to `path_s' is found, namei() will
 * increment its reference count.  So, the caller must eventually
 * call object_done() on that object.
 *
 */

lomac_object_t *path_to_object( const char *path_s ) {

  lomac_object_t *p_object;   /* the object, if found */

  if( IS_ERR( p_object = kernel_namei( path_s ) ) ) {
    return( NULL );
  }

  return( p_object );

} /* path_to_object() */


/* object_done()
 *
 * in:     p_object - object we are done manipulating
 * out:    p_object - will have reference count decreased
 * return: nothing
 *
 * This is an interface to dput()
 * */

void object_done( lomac_object_t *p_object ) {
  dput( p_object );
}



/**************************************************************
 * local utility functions
 **************************************************************/

/* kernel_namei()
 *
 * in:     path_s - path to lookup
 * out:    nothing
 * return: as the kernel's namei() function.  Use IS_ERR() and ERR_PTR().
 *
 *     This function provides an alternative to the kernel's namei()
 * which (unlike the kernel version) takes its path argument from 
 * kernel-space.
 * 
 *     When this function returns a valid dentry, it will increment
 * the dentry's reference count.  The caller must eventually call
 * pdut() on that dentry.
 *
 */

struct dentry *kernel_namei( const char *path_s ) {

  struct dentry *p_dentry;   /* points to `path_s's dir cache entry. */

  p_dentry = lookup_dentry( path_s, NULL, 1 );  /* 1 means follow links */
  if( !IS_ERR( p_dentry ) ) {
    if( ! ( p_dentry->d_inode ) ) {
      dput( p_dentry );
      return( ERR_PTR( -ENOENT ) );
    }
  }
  return( p_dentry );

} /* kernel_namei() */
