/*
 * FILE:
 * signal.c
 *
 * FUNCTION:
 * Handle a gtk signal
 *
 * HISTORY:
 * Linas Vepstas March 2002
 */

#include <string.h>
#include <glib.h>
#include <gtk/gtk.h>

#include "perr.h"

#include "readfile.h"
#include "signal.h"
#include "window.h"

struct DuiSignal_s
{
	DuiWindow *window;

	char * signalname;
	char * widgetname;
	char * actionname;
	GtkObject *obj;     /* when this object is activated, the action will run */
	guint  signal_id;   /* the gtk_signal id */

	GtkSignalFunc sigfunc;
	gpointer user_data;
};


/* ============================================================ */

DuiSignal * 
dui_signal_new (DuiWindow *win, const char * wig, const char * signame, const char *act)
{
	DuiSignal *sig;

	if (!win || !signame || !wig || !act) return NULL;

	sig = g_new (DuiSignal, 1);
	sig->window = win;
	sig->widgetname = g_strdup (wig);
	sig->signalname = g_strdup (signame);
	sig->actionname = g_strdup (act);
	sig->obj = NULL;
	sig->signal_id = -1;
	sig->sigfunc = NULL;
	sig->user_data = NULL;

	dui_window_add_signal (win, sig);

	return sig;
}

/* ============================================================ */

void
dui_signal_destroy (DuiSignal *sig)
{
	if (!sig) return;
 	if (sig->obj && GTK_IS_OBJECT(sig->obj) )
	{
		gtk_signal_disconnect (sig->obj, sig->signal_id);
	}
	g_free (sig->widgetname);
	g_free (sig->signalname);
	g_free (sig->actionname);

	sig->widgetname = NULL;
	sig->signalname = NULL;
	sig->actionname = NULL;

	sig->obj = NULL;
	sig->signal_id = -1;
	sig->sigfunc = NULL;
	sig->user_data = NULL;

	g_free (sig);
}

/* ============================================================ */

const char *
dui_signal_get_widget (DuiSignal *sig)
{
	if (!sig) return NULL;
	return sig->widgetname;
}

/* ============================================================ */
/* xxx FIXME hack alert -- should use GtkSignalQuery to determine
 * the proper shim type.
 */

static void
sig_shim_vi (GtkObject *obj, gint x, DuiSignal *sig)
{
	(sig->sigfunc) (obj, sig->user_data);
}

static void
sig_shim_viip (GtkObject *obj, gint x, gint y, gpointer z, DuiSignal *sig)
{
	(sig->sigfunc) (obj, sig->user_data);
}

void
dui_signal_connect (DuiSignal *sig, GtkObject *obj, 
                    GtkSignalFunc sigfunc, gpointer user_data)
{
	if (!sig || !obj || !sigfunc) return;

	sig->obj = obj;
	sig->sigfunc = sigfunc;
	sig->user_data = user_data;
	
	/* shim because user_data is not second arg */
	if ((GTK_IS_CLIST(obj) || GTK_IS_CTREE(obj)) &&
	      ((0 == strcmp (sig->signalname, "select_row"))
	    || (0 == strcmp (sig->signalname, "unselect_row"))))
	{
		sig->signal_id = gtk_signal_connect_after(obj, 
                   sig->signalname, GTK_SIGNAL_FUNC(sig_shim_viip), sig);
	}
	else
	if ((GTK_IS_CLIST(obj) || GTK_IS_CTREE(obj)) &&
	      ((0 == strcmp (sig->signalname, "fake_unselect_all"))))
	{
		sig->signal_id = gtk_signal_connect_after(obj, 
                   sig->signalname, GTK_SIGNAL_FUNC(sig_shim_vi), sig);
	}
	else
	{
		sig->signal_id = gtk_signal_connect_after(obj, 
                    sig->signalname, sigfunc, user_data);
	}
}

/* ============================================================ */

static void
close_window (GtkWidget *w, DuiWindow *win)
{
	gtk_widget_hide (dui_window_get_widget(win, NULL));
}

static void
main_quit (GtkWidget *w, DuiWindow *win)
{
	gtk_main_quit();
}

/* ============================================================ */

void
dui_signal_do_realize (DuiSignal *sig)
{
	if (!sig) return;
	ENTER ("(sig=%p)", sig);

	if (!strcmp (sig->actionname, "close_window"))
	{
		sig->obj = GTK_OBJECT (dui_window_get_widget(sig->window, sig->widgetname));
		sig->sigfunc = close_window;
		sig->user_data = sig->window;
	
		sig->signal_id = gtk_signal_connect(sig->obj, sig->signalname, 
			close_window, sig->window);
	}
	else if (!strcmp (sig->actionname, "main_quit"))
	{
		sig->obj = GTK_OBJECT (dui_window_get_widget(sig->window, sig->widgetname));
		sig->sigfunc = main_quit;
		sig->user_data = sig->window;
	
		sig->signal_id = gtk_signal_connect(sig->obj, sig->signalname, 
			main_quit, sig->window);
	}
	LEAVE ("(sig=%p)", sig);
}

/* ============================== END OF FILE =================== */
