/*
 * FILE:
 * report.c
 *
 * FUNCTION:
 * Fill in widget values based on results of database query
 *
 * HISTORY:
 * Linas Vepstas March 2002
 */

#include <stdlib.h>
#include <string.h>
#include <glib.h>
#include <gtk/gtk.h>
#include <gnome.h>

#include "perr.h"
#include "util.h"

#include "action.h"
#include "dui-initdb.h"
#include "filter.h"
#include "report.h"
#include "window.h"

typedef struct DuiColumn_s DuiColumn;
typedef struct DuiRow_s DuiRow;

struct DuiReport_s
{
	DuiWindow *window;       /* parent window */
	char * name;

	GList *rows;             /* list of nested/tree row iterators */
	DuiRow *curr_row;

	void (*pre)  (DuiReport *, DuiRow *);
	void (*iter) (DuiReport *, DuiRow *, const char *);
	void (*set)  (DuiReport *, DuiRow *, int, int, const char *);
	void (*set_data)  (DuiReport *, DuiRow *, int, const char *, const char *);
	void (*post) (DuiReport *, DuiRow *);

	/* list of actions that last drew into this window */
	GList *last_queries;

	/* misc scratch space used by clist iterator */
	char **blank_row;
};

struct DuiColumn_s
{
	char * col_widgetname;
	GtkWidget *col_widget;
	int col_num;         /* widget - column in which to display the result */
	char *datakey;       /* gtk_object key in which to store result */
	char *arg;           /* GtkArg to set */
	char *db_field;
	char *key;           /* global hash table key */
	char *value;
	DuiFilter *filter;
};

struct DuiRow_s
{
	char * row_name;
	char * row_widgetname;
	GtkWidget *row_iterator; /* get next row */

	GList *columns;          /* list of columns */

	int nest_level;
	int match_column;
	char * match_field;
	char * match_key;
	char * match_value;

	/* used only by the tree iterator */
	GtkCTreeNode *matching_ctn;
};

void object_browse (GtkObject *obj);
guint dui_string_hash_func (const char * key);
gint dui_string_compare_func (const char *a, const char * b);

/* ============================================================ */

DuiReport * 
dui_report_new (DuiWindow *win, const char * name)
{
	DuiReport * rpt;

	if (!win) return NULL;

	rpt = g_new (DuiReport, 1);

	rpt->window = win;
	rpt->name = g_strdup (name);
	rpt->rows = NULL;
	rpt->curr_row = NULL;
	rpt->last_queries = NULL;

	rpt->pre = NULL;
	rpt->iter = NULL;
	rpt->post = NULL;
	rpt->set = NULL;
	rpt->blank_row = NULL;
	
	dui_window_add_report (win, rpt);
	return rpt;
}

void 
dui_report_destroy (DuiReport *rpt)
{
	GList *rnode, *cnode;
	if (NULL == rpt) return;

	rpt->window = NULL;
	g_free(rpt->name);
	g_list_free (rpt->last_queries);

	rpt->pre = NULL;
	rpt->iter = NULL;
	rpt->post = NULL;
	rpt->set = NULL;
	g_free (rpt->blank_row);
	rpt->blank_row = NULL;

	for (rnode=rpt->rows; rnode; rnode=rnode->next)
	{
		DuiRow *row = rnode->data;
		g_free (row->row_name);
		g_free (row->row_widgetname);
		g_free (row->match_field);
		g_free (row->match_key);
		g_free (row->match_value);

		for (cnode=row->columns; cnode; cnode=cnode->next)
		{
			DuiColumn *col = cnode->data;
			g_free (col->col_widgetname);
			g_free (col->datakey);
			g_free (col->arg);
			g_free (col->db_field);
			g_free (col->key);
			g_free (col->value);
			g_free (col);
		}
		g_list_free (row->columns);

		g_free (row);
	}
	g_list_free (rpt->rows);
	rpt->rows = NULL;

	g_free (rpt);
}

/* ============================================================ */

static DuiRow *
find_row_by_name (DuiReport *rpt, const char * row_name)
{
	GList *node;
	
	/* locate the target row.  */
	for (node=rpt->rows; node; node=node->next)
	{
		DuiRow * row = node->data;

		if (NULL==row_name && NULL==row->row_name) return row;
		if (NULL==row_name) continue;
		if (NULL==row->row_name) continue;
		if (!strcmp (row->row_name, row_name)) return row;
	}

	return NULL;
}

void
dui_report_set_last_action (DuiReport *rpt, DuiAction *act, 
                            const char * row_name)
{
	DuiRow *row;
	if (!rpt) return;

	row = find_row_by_name (rpt, row_name);
		
	if (!row || 0 == row->nest_level)
	{
		g_list_free (rpt->last_queries);
		rpt->last_queries = NULL;
	}

	rpt->last_queries = g_list_append (rpt->last_queries,  act);
}

/* ============================================================ */

void
dui_report_add_row (DuiReport *rpt, const char * name,
                    const char *widname, int nest,
                    int match_col, const char * match_field,
                    const char * match_key, const char * match_value)
{
	DuiRow *row;
	if (!rpt) return;
	
	row = g_new (DuiRow, 1);
	row->row_name = NULL;
	row->row_widgetname = NULL;
	row->columns = NULL;

	row->match_field = NULL;
	row->match_key = NULL;
	row->match_value = NULL;

	row->row_iterator = NULL;
	row->nest_level = nest;
	row->match_column = match_col;

	if (name) row->row_name = g_strdup (name);
	if (widname) row->row_widgetname = g_strdup (widname);
	if (match_field) row->match_field = g_strdup (match_field);
	if (match_key) row->match_key = g_strdup (match_key);
	if (match_value) row->match_value = g_strdup (match_value);

	row->matching_ctn = NULL;

	rpt->rows = g_list_append (rpt->rows, row);
	rpt->curr_row = row;
}

/* ============================================================ */

void
dui_report_add_column (DuiReport *rpt, const char * widname, int num, 
                        const char * datakey, const char * arg,
                        const char * field, const char * key, 
                        const char * value, const char * filter)
{
	DuiColumn *col;

	if (!rpt) return;
	ENTER ("(rpt=%p, wid=%s, col=%d, field=%s, key=%s, val=%s)",
		rpt, widname, num, field, key, value);

	/* Its valid to have reports without a row iterator. However, 
	 * column definitions are stored with a row iterator, so ...
	 */
	if (NULL == rpt->curr_row)
	{
		dui_report_add_row (rpt, NULL, NULL, 0, -1, NULL, NULL, NULL);
	}

	col = g_new (DuiColumn, 1);

	col->col_widgetname = g_strdup (widname);
	col->col_widget = NULL;
	col->col_num  = num;
	col->datakey  = g_strdup (datakey);
	col->arg      = g_strdup (arg);
	col->db_field = g_strdup (field);
	col->key      = g_strdup (key);
	col->value    = g_strdup (value);
	col->filter   = dui_filter_find_by_name (filter);
	
	rpt->curr_row->columns = g_list_append (rpt->curr_row->columns, col);

}

/* ============================================================ */
/* ============================================================ */

static void
ctree_pre (DuiReport *rpt, DuiRow *row)
{
	int i, columns;
	// GtkCTree *ctree = GTK_CTREE (row->row_iterator);
	GtkCList *clist = GTK_CLIST (row->row_iterator);
	
	gtk_clist_freeze (clist);

	/* don't clear if we're focusing on subtrees */
	if (0 == row->nest_level)
	{
		gtk_clist_clear (clist);
	}

	/* create a blank row -- hack for adding rows */
	columns = clist->columns;
	if (rpt->blank_row) g_free (rpt->blank_row);
	rpt->blank_row = g_new (char *, columns);
	for (i=0; i<columns; i++) rpt->blank_row[i] = "";
}
	
static void
ctree_post (DuiReport *rpt, DuiRow *row)
{
	// GtkCTree *ctree = GTK_CTREE (row->row_iterator);
	GtkCList *clist = GTK_CLIST (row->row_iterator);
	
	gtk_clist_thaw(clist);
	g_free (rpt->blank_row);
	rpt->blank_row = NULL;
}

static GtkCTreeNode *
ctree_recur (GtkCTree *ctree, GtkCTreeNode *start_ctn, 
                 int match_col, const char * match_value)
{
	GtkCTreeRow *ctr;
	GtkCTreeNode *ctn;

	ctn = start_ctn;

	/* breadth first search */
	while (ctn)
	{
		char * cval;

		gtk_ctree_node_get_text (ctree, ctn, match_col, &cval);
		if (!strcmp (cval, match_value))
		{
			return ctn;
		}

		ctr = ctn->list.data;
		if (!ctr) break;
		ctn = ctr->sibling;
	}

	ctn = start_ctn;
	while (ctn)
	{
		GtkCTreeNode *child_ctn, *match_ctn;
		ctr = ctn->list.data;
		if (!ctr) break;
		child_ctn = ctr->children;

		match_ctn = ctree_recur (ctree, child_ctn, match_col, match_value);

		if (match_ctn) return match_ctn;

		ctn = ctr->sibling;
	}

	return NULL;
}
	
static void
ctree_iter (DuiReport *rpt, DuiRow *row, const char * match_value)
{
	GtkCTree *ctree = GTK_CTREE (row->row_iterator);
	GtkCTreeNode *ctn;
	int match_col;

	row->matching_ctn = NULL;
	if (NULL == match_value)
	{
		row->matching_ctn = gtk_ctree_insert_node (ctree, 
			   NULL, NULL, rpt->blank_row, 0, 
				NULL, NULL, NULL, NULL, FALSE, FALSE);
		return;
	} 

	match_col = row->match_column;

	ctn = gtk_ctree_node_nth (ctree, 0);
	ctn = ctree_recur (ctree, ctn, match_col, match_value);
	if (ctn)
	{
		row->matching_ctn = gtk_ctree_insert_node (ctree, ctn,
				NULL, rpt->blank_row, 0, 
				NULL, NULL, NULL, NULL, FALSE, FALSE);
		return;
	}

	SYNTAX ("could not locate a parent row with value %s", match_value);
}
	
static void
ctree_set (DuiReport *rpt, DuiRow *row, 
           int row_num, int col_num, const char * text)
{
	GtkCTree *ctree = GTK_CTREE (row->row_iterator);

	if (NULL == row->matching_ctn) return;
	PINFO ("(row=%d (%p), col=%d, text=\'%s\')", row_num, row->matching_ctn, col_num, text);
	gtk_ctree_node_set_text (ctree, row->matching_ctn, col_num, text);
}
	
static void
ctree_set_data (DuiReport *rpt, DuiRow *row, 
           int row_num, const char * datakey, const char * text)
{
	GHashTable *tbl;
	GtkCTree *ctree = GTK_CTREE (row->row_iterator);

	PINFO ("(row=%d, datakey=\'%s\', text=\'%s\')", row_num, datakey, text);
   if (NULL == row->matching_ctn) return;

	tbl = gtk_ctree_node_get_row_data (ctree, row->matching_ctn);
	if (!tbl)
	{
		/* hack alert -- we need to create destructors as well,
		 * otherwise a memory leak */
		tbl = g_hash_table_new ((GHashFunc) dui_string_hash_func, 
							         (GCompareFunc) dui_string_compare_func);
		/*	gtk_ctree_node_set_row_data_full() */
	   gtk_ctree_node_set_row_data (ctree, row->matching_ctn, tbl);
	}
	g_hash_table_insert (tbl, g_strdup(datakey), g_strdup(text));
}
	
/* ============================================================ */

static void
clist_pre (DuiReport *rpt, DuiRow *row)
{
	int i, columns;
	GtkCList *clist = GTK_CLIST (row->row_iterator);
	
	gtk_clist_freeze (clist);
	gtk_clist_clear (clist);

	/* create a blank row -- hack for adding rows */
	columns = clist->columns;
	if (rpt->blank_row) g_free (rpt->blank_row);
	rpt->blank_row = g_new (char *, columns);
	for (i=0; i<columns; i++) rpt->blank_row[i] = "";
}
	
static void
clist_post (DuiReport *rpt, DuiRow *row)
{
	GtkCList *clist = GTK_CLIST (row->row_iterator);
	
	gtk_clist_thaw(clist);
	g_free (rpt->blank_row);
	rpt->blank_row = NULL;
}
	
static void
clist_iter (DuiReport *rpt, DuiRow *row, const char * match_value)
{
	GtkCList *clist = GTK_CLIST (row->row_iterator);
	gtk_clist_append (clist, rpt->blank_row);
}
	
static void
clist_set (DuiReport *rpt, DuiRow *row, 
           int row_num, int col_num, const char * text)
{
	GtkCList *clist = GTK_CLIST (row->row_iterator);
	PINFO ("(row=%d, col=%d, text=%s)", row_num, col_num, text);
	gtk_clist_set_text (clist, row_num, col_num, text);
}
	
static void
clist_set_data (DuiReport *rpt, DuiRow *row, 
           int row_num, const char * datakey, const char * text)
{
	GHashTable *tbl;
	GtkCList *clist = GTK_CLIST (row->row_iterator);

	PINFO ("(row=%d, datakey=%s, text=%s)", row_num, datakey, text);
	tbl = gtk_clist_get_row_data (clist, row_num);
	if (!tbl)
	{
		/* hack alert -- we need to create destructors as well,
		 * otherwise a memory leak */
		tbl = g_hash_table_new ((GHashFunc) dui_string_hash_func, 
							         (GCompareFunc) dui_string_compare_func);
		/*	gtk_clist_set_row_data_full() */
		gtk_clist_set_row_data (clist, row_num, tbl);
	}
	g_hash_table_insert (tbl, g_strdup(datakey), g_strdup(text));
}
	
static void 
clist_sort_column (GtkCList *clist, gint column, DuiReport *rpt)
{
	static GtkSortType direction = GTK_SORT_DESCENDING;

	if (GTK_SORT_DESCENDING == direction) direction = GTK_SORT_ASCENDING;
	else direction = GTK_SORT_DESCENDING;

	gtk_clist_freeze (clist);
	gtk_clist_set_sort_type (clist, direction);
	gtk_clist_set_sort_column (clist, column);
	gtk_clist_sort (clist);

	gtk_clist_thaw (clist);

}

/* ============================================================ */
/* fetch value from recordset, or hash table or hard-coded value */

static const char *
get_value (DuiReport *rpt, DuiDBRecordSet *recs, const char *db_field,
           const char *key, const char * value)
{
	const char * val = NULL;
	
	if (recs && db_field) 
	{
		val = dui_recordset_get_value (recs, db_field);
	}
	else
	if (key)
	{
		DuiInterface *dui = dui_window_get_interface (rpt->window);
		val = dui_interface_kvp_lookup (dui, key);
	}
	else
	val = value;
   PINFO ("for dbfield=\'%s\' key=\'%s\' found val=\'%s\'\n", db_field, key, val);
	
	if (!val) val = "";
	return val;
}
				
/* ============================================================ */

static void
set_widget_arg (GtkObject *obj, const char * argname, const char * val)
{
	GtkArgInfo *arginfo;
	GtkType obj_type, arg_type;
	int ival = 0;
	double fval = 0.0;
	
	ENTER ("(obj=%p, arg=\'%s\', val=\'%s\')", obj, argname, val);
	obj_type = GTK_OBJECT_TYPE(obj);
	gtk_object_arg_get_info (obj_type, argname, &arginfo);
	arg_type = arginfo->type;
	PINFO ("arg is of type (%d) \'%s\'\n", arg_type, gtk_type_name (arg_type));

	switch (arg_type)
	{
		case GTK_TYPE_INVALID:
		case GTK_TYPE_NONE:
			break;
		case GTK_TYPE_CHAR:
		case GTK_TYPE_UCHAR:
			gtk_object_set (obj, argname, val[0], NULL);
			break;
		case GTK_TYPE_BOOL:
			if (val) ival = atoi(val);
			gtk_object_set (obj, argname, (gboolean) ival, NULL);
			break;
		case GTK_TYPE_INT:
		case GTK_TYPE_UINT:
			if (val) ival = atoi(val);
			gtk_object_set (obj, argname, (int) ival, NULL);
			break;
		case GTK_TYPE_LONG:
		case GTK_TYPE_ULONG:
			if (val) ival = atoi(val);
			gtk_object_set (obj, argname, (long) ival, NULL);
			break;
		case GTK_TYPE_FLOAT:
			if (val) fval = atof(val);
			gtk_object_set (obj, argname, (float) fval, NULL);
			break;
		case GTK_TYPE_DOUBLE:
			if (val) fval = atof(val);
			gtk_object_set (obj, argname, fval, NULL);
			break;
		case GTK_TYPE_STRING:
			gtk_object_set (obj, argname, val, NULL);
			break;
		default:
			break;
	}
}

/* ============================================================ */

static void
set_widget_value (GtkObject *obj, const char * val)
{
	/* The order in which the classes appear here is IMPORTANT!
	 * Child classes must appear before parents, else inheritance
	 * will hose the results!
	 */
	
	if (GTK_IS_SPIN_BUTTON (obj))
	{
		gfloat flt;
		flt = atof (val);
		gtk_spin_button_set_value (GTK_SPIN_BUTTON(obj), flt);
	}
	if (GTK_IS_ENTRY (obj))
	{
		gtk_entry_set_text (GTK_ENTRY(obj), val);
	}
	else
	if (GTK_IS_LABEL (obj))
	{
		gtk_label_set_text (GTK_LABEL(obj), val);
	}
	else
	if (GTK_IS_TEXT (obj))
	{
		xxxgtk_text_set_text (GTK_TEXT(obj), val);
	}
	else
	if (GTK_IS_COMBO (obj))
	{
		gtk_entry_set_text (GTK_ENTRY(GTK_COMBO (obj)->entry), val);
	}
	else
	if (GTK_IS_RANGE (obj))
	{
		gfloat flt = atof (val);
		GtkAdjustment *adj = gtk_range_get_adjustment (GTK_RANGE(obj));
		gtk_adjustment_set_value (adj, flt);
	}
	else
	if (GTK_IS_MENU_SHELL (obj))
	{
		/* handles menubar, menu, and option menu */
		int i=0;
		int ival = 0;
		GList *node; 
		GtkMenuShell *menu = GTK_MENU_SHELL(obj);
		if (val) ival = atoi (val);
		for (node = menu->children; node; node=node->next)
		{
			if (i == ival)
			{
				gtk_menu_shell_select_item (menu, node->data);
				gtk_menu_shell_activate_item (menu, node->data, 1);
				break;
			}
			i++;
		}
	}
	else
	if (GTK_IS_OPTION_MENU (obj))
	{
		GtkObject *menu = GTK_OBJECT(GTK_OPTION_MENU(obj)->menu);
		set_widget_value (menu, val);
	}
	else
	if (GTK_IS_RADIO_BUTTON (obj))
	{
		GSList *node;
		const char *data = gtk_object_get_data (obj, "/system/data");
							 
		/* if a value hasn't been set, set it now */
		if (!data)
		{
			gtk_object_set_data (obj, "/system/data", (gpointer) val);
			gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(obj), 1);
		}
		else
		{
			/* find the button with the matching data an push it */
			node = gtk_radio_button_group (GTK_RADIO_BUTTON (obj));
			for ( ; node; node=node->next)
			{
				data = gtk_object_get_data (GTK_OBJECT(node->data), 
			                 "/system/data");
				if (data && !strcmp (data, val)) break;
			}
			if (node)
			{
				gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(node->data), 1);
			}
		}
	}
	else
	if (GTK_IS_TOGGLE_BUTTON (obj))
	{
		int ival = 0;
		if (val) ival = atoi (val);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(obj), ival);
	}
	else
	if (GNOME_IS_FILE_ENTRY (obj))
	{
		GtkWidget * entry = gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY(obj));
		gtk_entry_set_text (GTK_ENTRY (entry), val);
	}
	else
	if (GTK_IS_FILE_SELECTION (obj))
	{
		gtk_entry_set_text (GTK_ENTRY (GTK_FILE_SELECTION 
		                 (obj)->selection_entry), val);
	}
	else
	if (GTK_IS_FRAME(obj))
	{
		gtk_frame_set_label (GTK_FRAME(obj), val);
	}
	else
	if (GTK_IS_WINDOW(obj))
	{
		gtk_window_set_title (GTK_WINDOW(obj), val);
	}
	else
	if (GTK_IS_BIN (obj))  /* must appear after all other bins */
	{
		/* catch-all for GtkButton, others */
object_browse (obj);
		set_widget_value (GTK_OBJECT(GTK_BIN(obj)->child), val);
	}
	else
	if (GNOME_IS_DATE_EDIT (obj))
	{
		/* hack alert -- we assume the database returns iso8601
		 * formatted date strings ... true for postgres, but what
		 * about others ???
		 */
		time_t thyme = gnc_iso8601_to_secs_gmt (val);
		gnome_date_edit_set_time (GNOME_DATE_EDIT(obj), thyme);
	}
	else
	if (GNOME_IS_ABOUT (obj))
	{
		/* no-op */
	}
	else
	{
		SYNTAX ("unsupported report widget \"%s\"\n", 
	     	gtk_type_name(GTK_OBJECT_TYPE(obj)));
	}
}

/* ============================================================ */

int
dui_report_show_data (DuiReport *rpt, DuiDBRecordSet *recs, 
                      const char * row_name)
{
	int rc = 0;
	GList *cnode;
	DuiRow *row = NULL;
	int row_num = 0;
	int more_rows = 1;
	const char * match_val = NULL;
	
	if (!rpt) return 0;

	ENTER ("(rpt=%p \'%s\', recs=%p, row=%s)", rpt, dui_report_get_name (rpt), recs, row_name);

	/* locate the target row.  */
	row = find_row_by_name (rpt, row_name);

	if (NULL == row)
	{
		PERR ("could not find the target row %s", row_name);
		return 0;
	}
	
	/* Check to see if there's any data to display at all.
	 * If there's not, then we don't realize the report widget, 
	 * we don't show anytihing, we bail from here, and let the
	 * next action in the chain take over. */
	if (recs) more_rows = dui_recordset_fetch_row (recs);
	while (more_rows)
	{
		for (cnode=row->columns; cnode; cnode=cnode->next)
		{
			DuiColumn *col = cnode->data;
			const char * val;

			val = get_value (rpt, recs, col->db_field, col->key, col->value);
			if (col->filter) val = dui_filter_apply (col->filter, val);
			if (val) { rc = 1; break; }
		}
		if (rc) break; 
				
		row_num ++;
		more_rows = 0;
		if (recs) more_rows = dui_recordset_fetch_row (recs);
	}
	if (0 == rc) return 0;
	
	/* ------------------------------------------------- */
	/* make widget show up on screen & other intialization */
	dui_window_realize (rpt->window);

	/* currently, the only supported multi-rowed widgets are
	 * ctree and clist */
	if (GTK_IS_CTREE(row->row_iterator))
	{
		rpt->pre = ctree_pre;
		rpt->post = ctree_post;
		rpt->iter = ctree_iter;
		rpt->set = ctree_set;
		rpt->set_data = ctree_set_data;

		gtk_signal_connect (GTK_OBJECT(row->row_iterator), 
			"click_column", GTK_SIGNAL_FUNC(clist_sort_column), rpt);
	}
	else
	if (GTK_IS_CLIST(row->row_iterator))
	{
		rpt->pre = clist_pre;
		rpt->post = clist_post;
		rpt->iter = clist_iter;
		rpt->set = clist_set;
		rpt->set_data = clist_set_data;

		gtk_signal_connect (GTK_OBJECT(row->row_iterator), 
			"click_column", GTK_SIGNAL_FUNC(clist_sort_column), rpt);
	}
	else
	{
		rpt->pre = NULL;
		rpt->post = NULL;
		rpt->iter = NULL;
		rpt->set = NULL;
		rpt->set_data = NULL;
	}
	
	/* match val may be a row-indep constant; fetch it now */
	if (row_name && (row->match_key || row->match_value)) 
	{
		match_val = get_value (rpt, NULL, NULL, 
		                       row->match_key, row->match_value);
	}

	if (rpt->pre) (rpt->pre) (rpt, row);
				
	row_num = 0;
	
	while (more_rows)
	{
		/* fetch match val if it comes from the database */
		if (row_name && row->match_field)
		{
			match_val = get_value (rpt, recs, row->match_field, 
			                       row->match_key, row->match_value);
		}
		
		if (rpt->iter) (rpt->iter) (rpt, row, match_val);

		for (cnode=row->columns; cnode; cnode=cnode->next)
		{
			DuiColumn *col = cnode->data;
			const char * val;

			val = get_value (rpt, recs, col->db_field, col->key, col->value);
			if (col->filter) val = dui_filter_apply (col->filter, val);
			

			if (!col->col_widget || col->col_widget == row->row_iterator)
			{
				if (col->datakey && rpt->set_data)
				{
					(rpt->set_data) (rpt, row, row_num, col->datakey, val);
				} else
				if (rpt->set) (rpt->set) (rpt, row, row_num, col->col_num, val);
			}
			else
			if (col->datakey)
			{
				gtk_object_set_data (GTK_OBJECT (col->col_widget), 
				            col->datakey, g_strdup(val));
			}
			else
			if (col->arg)
			{
				set_widget_arg (GTK_OBJECT(col->col_widget), col->arg, val);
			}
			else
			set_widget_value (GTK_OBJECT(col->col_widget), val);
		}

		row_num ++;
		more_rows = 0;
		if (recs) more_rows = dui_recordset_fetch_row (recs);
	}

	if (rpt->post) (rpt->post) (rpt, row);

	LEAVE ("(rpt=%p \'%s\', recs=%p, row=%s) rc=%d", rpt, dui_report_get_name (rpt), recs, row_name, rc);
	
	return rc;
}

/* ============================================================ */

const char * 
dui_report_get_name (DuiReport *rpt)
{
	if (!rpt) return NULL;
	if (rpt->name) return (rpt->name);
	return dui_window_get_name (rpt->window);
}

/* ============================================================ */

void
dui_report_do_realize (DuiReport *rpt)
{
	GList *rnode, *cnode;

	if (!rpt) return;

	ENTER ("(rpt=%p \'%s\')", rpt, dui_report_get_name (rpt));

	/* convert widget names into pointers to actual widgets */
	for (rnode=rpt->rows; rnode; rnode=rnode->next)
	{
		DuiRow *row = rnode->data;

		if (row->row_widgetname)
		{
			row->row_iterator = dui_window_get_widget (rpt->window, 
				                           row->row_widgetname);
		}
	
		for (cnode=row->columns; cnode; cnode=cnode->next)
		{
			DuiColumn *col = cnode->data;
	
			if (col->col_widgetname)
			{
				col->col_widget = dui_window_get_widget (rpt->window, 
				                               col->col_widgetname);
			}
		}
	}

	LEAVE ("(rpt=%p \'%s\')", rpt, dui_report_get_name (rpt));
}

/* ============================================================ */

void
dui_report_refresh (DuiReport *rpt)
{
	GList *node;
	if (!rpt) return;

	ENTER ("(rpt=%p \'%s\')", rpt, dui_report_get_name (rpt));
	/* no refresh needed if the main window widget doesn't exist */
	if (0 == dui_window_is_realized (rpt->window)) return;

	for (node=rpt->last_queries; node; node=node->next)
	{
		DuiAction *last_action = node->data;
		dui_action_rerun_last_query (last_action);
	}
	LEAVE ("(rpt=%p \'%s\')", rpt, dui_report_get_name (rpt));
}

/* ============================== END OF FILE =================== */
