/*
 * FILE:
 * readfile.c
 *
 * FUNCTION:
 * parse an XML file that specifies the glade-to-database coupling
 *
 * HISTORY:
 * Linas Vepstas March 2002
 */

#include <string.h>
#include <stdlib.h>

#include <glib.h>

#include <parser.h>
#include <gnome.h>

#include "perr.h"
#include "dui-initdb.h"

#include "action.h"
#include "database.h"
#include "filter.h"
#include "readfile.h"
#include "report.h"
#include "signal.h"
#include "util.h"
#include "window.h"

struct DuiInterface_s 
{
	/* application name */
	char * app_name;

	/* XML file parser */
	xmlSAXHandler sax;
	char * xml_filename;

	/* current database connection that is being used */
	DuiDatabase *database;
	
	/* current glade window that is being processed */
	DuiWindow *window;

	/* current action that is being processed */
	DuiAction  *action;

	/* current report that is being processed */
	DuiReport *report;

	/* current filter definition */
	DuiFilter *filter;
	
	/* collection of handlers */
	GList *root_windows;
	GList *action_handlers;
	GList *report_handlers;

	/* set of key-value pairs */ 
	GHashTable *kvp; 
};


/* =================================================================== */
/* a really cheap and dirty thing .. */

static const char * 
de_escape (const char *val)
{
	if (!val) return NULL;

	if (!strcmp (val, "&gt;")) return ">";
	if (!strcmp (val, "&lt;")) return "<";

	return val;
}

/* =================================================================== */

#define CASE_STRDUP(STR,MEMB)                          \
	if (!strcmp (attrs[i], STR)) {                 \
		MEMB = g_strdup(attrs[i+1]);           \
	} else

#define CASE(STR,MEMB)                                 \
	if (!strcmp (attrs[i], STR)) {                 \
		MEMB = attrs[i+1];                     \
	} else

#define CASE_UNKNOWN                                 \
	{                                                 \
		SYNTAX ("unknown %s attribute %s", eltname, attrs[i]); \
	}   

#define GENERIC_CHECKS                               \
	if (!attrs)                                       \
	{                                                 \
		SYNTAX ("%s must have attributes!", eltname);  \
		return;                                        \
	}

#define CHECK_NEST(fld,prnt)                         \
	if (!dui->fld)                                    \
	{                                                 \
		SYNTAX ("%s can only occur within a " prnt "!", eltname);\
		return;                                        \
	}

#define REQUIRE(var,what)                            \
	if (NULL == var)                                  \
	{                                                 \
		SYNTAX ("%s must specify " what, eltname);     \
		return;                                        \
	}

#define REQUIRE2(vara,varb,what)                     \
	if ((NULL == vara) && (NULL == varb))             \
	{                                                 \
		SYNTAX ("%s must specify " what, eltname);     \
		return;                                        \
	}

#define REQUIRE3(vara,varb,varc,what)                \
	if ((NULL == vara) && (NULL == varb) && (NULL == varc))   \
	{                                                 \
		SYNTAX ("%s must specify " what, eltname);     \
		return;                                        \
	}

#define REQUIRE4(vara,varb,varc,vard,what)           \
	if ((NULL==vara) && (NULL==varb) && (NULL==varc) && (NULL==vard))   \
	{                                                 \
		SYNTAX ("%s must specify " what, eltname);     \
		return;                                        \
	}


/* =================================================================== */

static void 
handle_report (DuiInterface *dui, const xmlChar **attrs)
{
	DuiReport *rpt;
	static const char * eltname = "<report>";
	const char * name = NULL;
	int i;

	if (!dui->window) 
	{
		SYNTAX ("must specify a glade file first!");
		return;
	}
	CHECK_NEST (window, "<window>");

	if (attrs)
	{
		i = 0;
		while (attrs[i])
		{
			CASE ("name",  name)
			CASE_UNKNOWN ;
			i += 2;
		}
	}

	/* ------------------------------------------------------------------ */
	/* Set up the widget so that when its clicked or whatever, 
	 * we run our sql statement.
	 */
	rpt = dui_report_new (dui->window, name);
	dui->report = rpt;
	dui->report_handlers = g_list_append (dui->report_handlers, rpt);
}

static void 
handle_report_end (DuiInterface *dui)
{
	if (!dui->report) return;

	dui->report = NULL;
}

/* =================================================================== */

static void 
handle_row (DuiInterface *dui, const xmlChar **attrs)
{
	static const char * eltname = "<row>";
	int i;
	const char * name = NULL;
	const char * widgetname = NULL;
	const char * nest = NULL;
	const char * match_col = NULL;
	const char * match_field = NULL;
	const char * match_key = NULL;
	const char * match_value = NULL;
	int nest_level = 0;
	int mcol = -1;

	GENERIC_CHECKS ;
	CHECK_NEST (window, "<window>");
	CHECK_NEST (report, "<report>");

	i = 0;
	while (attrs[i])
	{
		CASE ("name",         name)
		CASE ("widget",       widgetname)
		CASE ("nest",         nest)
		CASE ("match_column", match_col)
		CASE ("match_field",  match_field)
		CASE ("match_key",    match_key)
		CASE ("match_value",  match_value)
		CASE_UNKNOWN ;
		i += 2;
	}

	REQUIRE (widgetname, "a widget!");

	if (nest) nest_level = atoi (nest);
	if (match_col) mcol = atoi (match_col);
	
	dui_report_add_row (dui->report, name, widgetname, nest_level, 
	                    mcol, match_field, match_key, match_value);

}

/* =================================================================== */

static void 
handle_entry (DuiInterface *dui, const xmlChar **attrs)
{
	static const char * eltname = "<entry>";
	int i;
	const char * name = NULL;
	const char * widgetname = NULL;
	const char * fieldname = NULL;
	const char * column = NULL;
	const char * datakey = NULL;
	const char * arg = NULL;
	const char * key = NULL;
	const char * value = NULL;
	const char * filtername = NULL;
	int col = -1;

	GENERIC_CHECKS ;
	CHECK_NEST (window, "<window>");
	CHECK_NEST (report, "<report>");

	i = 0;
	while (attrs[i])
	{
		CASE ("name",   name)
		CASE ("widget", widgetname)
		CASE ("column", column)
		CASE ("data",   datakey)
		CASE ("arg",    arg)
		CASE ("key",    key)
		CASE ("value",  value)
		CASE ("filter", filtername)
		CASE ("field",  fieldname)
		CASE_UNKNOWN ;
		i += 2;
	}

	// hack alert -- fixme -- 
	// a widget is not required if we are nested inside a row;
	// otherwise a widget is mandatory
	// REQUIRE (widgetname, "a widget!");
	// 
	REQUIRE3 (fieldname, value, key, "a field or a key or a value!");

	if (column) col = atoi (column);

	dui_report_add_column (dui->report, widgetname, col, datakey, arg,
	                       fieldname, key, value, filtername);
}


/* =================================================================== */
/* ********************* form handlers below ************************* */
/* =================================================================== */

/* =================================================================== */

static void 
handle_form (DuiInterface *dui, const xmlChar **attrs)
{
	static const char * eltname = "<form>";
	int i;
	const char * formname = NULL;
	const char * reportname = NULL;
	const char * sql_querytype = NULL;
	const char * sql_tables = NULL;
	const char * key_to_sql_tables = NULL;
	const char * row_name = NULL;
	DuiAction *act;

	if (attrs)
	{
		i = 0;
		while (attrs[i])
		{
			CASE ("name",       formname)
			CASE ("type",       sql_querytype)
			CASE ("tables",     sql_tables)
			CASE ("tables_key", key_to_sql_tables)
			CASE ("report",     reportname)
			CASE ("row",        row_name)
			CASE_UNKNOWN ;
			i += 2;
		}
	}

	/* ------------------------------------------------------------------ */
	/* Set up the widget so that when its clicked or whatever, 
	 * we run our sql statement.
	 */
	act = dui_action_new (dui->window, formname);
	dui->action = act;
	dui->action_handlers = g_list_append (dui->action_handlers, act);

	dui_action_set_table (act, sql_tables, key_to_sql_tables, sql_querytype);
	dui_action_set_report (act, reportname, row_name);
	dui_action_set_database (act, dui->database);
}

static void 
handle_form_end (DuiInterface *dui)
{
	if (!dui->action) return;

	dui->action = NULL;
}

/* =================================================================== */

static void 
handle_submit (DuiInterface *dui, const xmlChar **attrs)
{
	static const char * eltname = "<submit>";
	int i;
	const char * widgetname = NULL;
	const char * signalname = NULL;

	GENERIC_CHECKS ;
	CHECK_NEST (window, "<window>");
	CHECK_NEST (action, "<form>");

	i = 0;
	while (attrs[i])
	{
		CASE ("widget", widgetname)
		CASE ("signal", signalname)
		CASE_UNKNOWN ;
		i += 2;
	}

	REQUIRE (widgetname, "a widget!");
	REQUIRE (signalname, "a signal!");

	dui_action_add_signal (dui->action, widgetname, signalname);
}

/* =================================================================== */

static void 
handle_refresh (DuiInterface *dui, const xmlChar **attrs)
{
	static const char * eltname = "<refresh>";
	int i;
	const char * windowname = NULL;

	GENERIC_CHECKS ;
	CHECK_NEST (window, "<window>");
	CHECK_NEST (action, "<form>");

	i = 0;
	while (attrs[i])
	{
		CASE ("window", windowname)
		CASE_UNKNOWN ;
		i += 2;
	}

	REQUIRE (windowname, "a window name!");
	dui_action_add_refresh (dui->action, windowname);
}

/* =================================================================== */

static void 
handle_select (DuiInterface *dui, const xmlChar **attrs)
{
	static const char * eltname = "<select>";
	int i;
	const char * fieldname = NULL;

	GENERIC_CHECKS ;
	CHECK_NEST (window, "<window>");
	CHECK_NEST (action, "<form>");

	i = 0;
	while (attrs[i])
	{
		CASE ("field", fieldname)
		CASE_UNKNOWN ;
		i += 2;
	}

	REQUIRE (fieldname, "a data field!");
	dui_action_add_select_term (dui->action, fieldname);
}

/* =================================================================== */

static void 
handle_insert (DuiInterface *dui, const xmlChar **attrs)
{
	static const char * eltname = "<insert>";
	int i;
	const char * widgetname = NULL;
	const char * column = NULL;
	const char * datakey = NULL;
	const char * key = NULL;
	const char * value = NULL;
	const char * filtername = NULL;
	const char * fieldname = NULL;
	int colnum = -1;

	GENERIC_CHECKS ;
	CHECK_NEST (window, "<window>");
	CHECK_NEST (action, "<form>");

	i = 0;
	while (attrs[i])
	{
		CASE ("widget", widgetname)
		CASE ("column", column)
		CASE ("data",   datakey)
		CASE ("key",    key)
		CASE ("value",  value)
		CASE ("filter", filtername)
		CASE ("field",  fieldname)
		CASE_UNKNOWN ;
		i += 2;
	}

	REQUIRE3 (widgetname, key, value, "a widget or a key or a value!");
	REQUIRE (fieldname, "a data field!");
	
	if (column) colnum = atoi (column);

	/* set up the action so that we fetch values from it */
	dui_action_add_update_term(dui->action, widgetname, 
                   colnum, datakey, key, value, filtername, fieldname);
}

/* =================================================================== */

static void 
handle_where (DuiInterface *dui, const xmlChar **attrs)
{
	static const char * eltname = "<where>";
	int i;
	const char * widgetname = NULL;
	const char * column = NULL;
	const char * datakey = NULL;
	const char * fieldname = NULL;
	const char * compareop = NULL;
	const char * key = NULL;
	const char * value = NULL;
	const char * filtername = NULL;
	int colnum = -1;

	GENERIC_CHECKS ;
	CHECK_NEST (window, "<window>");
	CHECK_NEST (action, "<form>");

	i = 0;
	while (attrs[i])
	{
		CASE ("widget", widgetname)
		CASE ("column", column)
		CASE ("data",   datakey)
		CASE ("key",    key)
		CASE ("value",  value)
		CASE ("filter", filtername)
		CASE ("field",  fieldname)
		CASE ("op",     compareop)
		CASE_UNKNOWN ;
		i += 2;
	}

	/* require either value or a key or a widget ... */
	REQUIRE3 (widgetname, key, value, "a widget or a key or a value!");
	REQUIRE (fieldname, "a data field!");

	if (column) colnum = atoi (column);

	/* set up the action so that we fetch values from it */
	compareop = de_escape (compareop);
	dui_action_add_where_term(dui->action, widgetname, colnum, 
		         datakey, key, value, filtername, fieldname, compareop);

}

/* =================================================================== */

static void 
handle_chain (DuiInterface *dui, const xmlChar **attrs)
{
	static const char * eltname = "<chain>";
	int i;
	const char * actionname = NULL;

	GENERIC_CHECKS ;
	CHECK_NEST (window, "<window>");
	CHECK_NEST (action, "<form>");

	i = 0;
	while (attrs[i])
	{
		CASE ("form", actionname)
		CASE_UNKNOWN ;
		i += 2;
	}

	/* require an action name ... */
	REQUIRE (actionname, "a form!");

	/* set up the action so that check values */
	dui_action_add_chain (dui->action, actionname);
}

/* =================================================================== */

static void 
handle_wtokey (DuiInterface *dui, const xmlChar **attrs)
{
	static const char * eltname = "<wtokey>";
	int i;
	const char * widgetname = NULL;
	const char * column = NULL;
	const char * datakey = NULL;
	const char * key = NULL;
	const char * value = NULL;
	const char * filtername = NULL;
	int colnum = -1;

	GENERIC_CHECKS ;
	CHECK_NEST (window, "<window>");
	CHECK_NEST (action, "<form>");

	i = 0;
	while (attrs[i])
	{
		CASE ("widget", widgetname)
		CASE ("column", column)
		CASE ("data",   datakey)
		CASE ("key",    key)
		CASE ("value",  value)
		CASE ("filter", filtername)
		CASE_UNKNOWN ;
		i += 2;
	}

	/* require either value or widget ... */
	REQUIRE2 (widgetname, value, "a widget or a value!");
	REQUIRE (key, "a key!");

	if (column) colnum = atoi (column);

	/* set up the action so that we fetch values from it */
	dui_action_add_kvp_term(dui->action, widgetname, colnum, 
	                         datakey, value, filtername, key);

}

/* =================================================================== */

static void 
handle_sigaction (DuiInterface *dui, const xmlChar **attrs)
{
	static const char * eltname = "<sigaction>";
	int i;
	const char * widgetname = NULL;
	const char * signalname = NULL;
	const char * sigaction = NULL;

	GENERIC_CHECKS ;
	CHECK_NEST (window, "<window>");

	i = 0;
	while (attrs[i])
	{
		CASE ("widget", widgetname)
		CASE ("signal", signalname)
		CASE ("action", sigaction)
		CASE_UNKNOWN ;
		i += 2;
	}

	REQUIRE (widgetname, "a widget!");
	REQUIRE (signalname, "a valid signal name on the widget!");
	REQUIRE (sigaction,  "an action to perform!");

	dui_signal_new (dui->window, widgetname, signalname, sigaction);
}

/* =================================================================== */

static void 
handle_window (DuiInterface *dui, const xmlChar **attrs)
{
	static const char * eltname = "<window>";
	int i;
	const char * provider = NULL;
	const char * winname = NULL;
	const char * filepath = NULL;
	const char * root_widget = NULL;
	const char * mainattr = NULL;
	int is_main = 0;

	GENERIC_CHECKS ;

	i = 0;
	while (attrs[i])
	{
		CASE ("provider",    provider)
		CASE ("name",        winname)
		CASE ("filename",    filepath)
		CASE ("root_widget", root_widget)
		CASE ("main",        mainattr)
		CASE_UNKNOWN ;
		i += 2;
	}

	if (!winname)
	{
		REQUIRE (filepath, "a file path!");
		REQUIRE (root_widget, "a root widget!");
	}

	/* if there's already a window by this name, append to it;
	 * else create a new window */
	dui->window = dui_interface_find_window_by_name (dui, winname);
	if (NULL == dui->window)
	{
		REQUIRE (filepath, "a file path!");
		REQUIRE (root_widget, "a root widget!");

		if (mainattr && !strcasecmp (mainattr, "yes")) is_main = 1;
	
		dui->window = dui_window_new (winname, dui);
		dui_window_set_glade (dui->window, filepath, root_widget, is_main);
		dui->root_windows = g_list_append (dui->root_windows, dui->window);
	}
}

static void 
handle_window_end (DuiInterface *dui)
{
	dui->window = NULL;
}

/* =================================================================== */

static void 
handle_database (DuiInterface *dui, const xmlChar **attrs)
{
	static const char * eltname = "<database>";
	int i;
	const char * provider = NULL;
	const char * dbname = NULL;
	const char * hostname = NULL;
	const char * username = NULL;

	GENERIC_CHECKS ;

	i = 0;
	while (attrs[i])
	{
		CASE_STRDUP ("provider", provider)
		CASE_STRDUP ("dbname",   dbname)
		CASE_STRDUP ("host",     hostname)
		CASE_STRDUP ("username", username)
		CASE_UNKNOWN ;
		i += 2;
	}

	REQUIRE (provider, "a database provider!");
	REQUIRE (dbname,   "a database name!");
	REQUIRE (username, "a user name!");

	/* now actually open the database */
	dui->database = dui_database_new (provider, dbname, hostname, username);
}

/* =================================================================== */

static void 
handle_lookup (DuiInterface *dui, const xmlChar **attrs)
{
	static const char * eltname = "<lookup>";
	const char * key = NULL; 
	const char * val = NULL; 
	int i;

	GENERIC_CHECKS ;
	CHECK_NEST (filter, "<filter>");

	i = 0;
	while (attrs[i])
	{
		CASE_STRDUP ("key",    key)
		CASE_STRDUP ("val",    val)
		CASE_UNKNOWN ;
		i += 2;
	}

	REQUIRE (key, "a lookup key!");
	REQUIRE (val, "a value that is looked up!");
	dui_filter_add_lookup (dui->filter, key, val);
}

/* =================================================================== */

static void 
handle_filter (DuiInterface *dui, const xmlChar **attrs)
{
	static const char * eltname = "<filter>";
	const char * filtername = NULL; 
	const char * reversed_name = NULL; 
	int i;

	GENERIC_CHECKS ;

	i = 0;
	while (attrs[i])
	{
		CASE_STRDUP ("name",    filtername)
		CASE_STRDUP ("revname", reversed_name)
		CASE_UNKNOWN ;
		i += 2;
	}

	REQUIRE (filtername, "a name for the new filter!");
	dui->filter = dui_filter_new (filtername, reversed_name);
}

static void 
handle_filter_end (DuiInterface *dui)
{
	dui->filter = NULL;
}

/* =================================================================== */

static void 
handle_dwi (DuiInterface *dui, const xmlChar **attrs)
{
	static const char * eltname = "<dwi>";
	int i;

	if (attrs)
	{
		i = 0;
		while (attrs[i])
		{
			CASE_STRDUP ("name", dui->app_name)
			CASE_UNKNOWN ;
			i += 2;
		}
	}
}

/* =================================================================== */

static void 
fatal_err (void *user_data, const char *msg, ...)
{
	/* hack laert -- handle the elipses in the warning message */
	// DuiInterface *dui = (DuiInterface *) user_data;
	PERR ("fatal parsing error: %s\n", msg);
}

static void 
err (void *user_data, const char *msg, ...)
{
	// DuiInterface *dui = (DuiInterface *) user_data;
	PERR ("parsing error: %s\n", msg);
}

static void 
warn (void *user_data, const char *msg, ...)
{
	// DuiInterface *dui = (DuiInterface *) user_data;
	PERR ("parse warning: %s\n", msg);
}

static void 
start_doc (void *user_data)
{
	// DuiInterface *dui = (DuiInterface *) user_data;
	PINFO ("document start ptr=%p", user_data);
}

static void 
end_doc (void *user_data)
{
	// DuiInterface *dui = (DuiInterface *) user_data;
	PINFO ("document end");
}

/* =================================================================== */

#define DISPATCH(tok,handler)                  \
	if (!strcmp (name, tok)) {             \
		handler (dui, attrs);          \
	} else

static void 
start_elt (void *user_data, const xmlChar *name, const xmlChar **attrs)
{
	DuiInterface *dui = (DuiInterface *) user_data;


	DISPATCH ("dwi",       handle_dwi)
	DISPATCH ("filter",    handle_filter)
	DISPATCH ("lookup",    handle_lookup)
	DISPATCH ("database",  handle_database)
	DISPATCH ("window",    handle_window)

	DISPATCH ("report",    handle_report)
	DISPATCH ("row",       handle_row)
	DISPATCH ("entry",     handle_entry)
	
	DISPATCH ("form",      handle_form)
	DISPATCH ("chain",     handle_chain)
	DISPATCH ("select",    handle_select)
	DISPATCH ("insert",    handle_insert)
	DISPATCH ("where",     handle_where)

	DISPATCH ("submit",    handle_submit)
	DISPATCH ("refresh",   handle_refresh)
	DISPATCH ("wtokey",    handle_wtokey)

	DISPATCH ("sigaction", handle_sigaction)
	{
		int i;
		PERR ("unknown element %s\n", name);
		i = 0;
		while (attrs && attrs[i])
		{
			printf ("\tattr=%s\n", attrs[i]);
			i++;
		}
	}
}


static void 
end_elt (void *user_data, const xmlChar *name)
{
	DuiInterface *dui = (DuiInterface *) user_data;
	PINFO ("</%s>", name);

	if (!strcmp (name, "form"))
	{
		handle_form_end (dui);
	}
	else
	if (!strcmp (name, "report"))
	{
		handle_report_end (dui);
	}
	else
	if (!strcmp (name, "window"))
	{
		handle_window_end (dui);
	}
	else
	if (!strcmp (name, "filter"))
	{
		handle_filter_end (dui);
	}
}


static void
cdata_cb (void *user_data, const xmlChar *value, int len)
{
#if 0
	DuiInterface *dui = (DuiInterface *) user_data;
	char *p;
	p = g_strndup (value, len);
	printf ("cdata duude len=%d %s\n", len, p);
	g_free (p);
#endif

}

/* =================================================================== */

guint 
dui_string_hash_func (const char * key)
{
	guint hash = 0;
	int i=0;
	if (!key) return 0;

	while (key[i])
	{
		hash += key[i] << 5 * (i%6);
		i++;
	}
	return hash;
}

gint 
dui_string_compare_func (const char *a, const char * b)
{
	return (0 == strcmp (a,b));
}


/* =================================================================== */

DuiInterface * 
dui_interface_new (void)
{
	DuiInterface * dui;

	dui = g_new (DuiInterface,1);

	dui->app_name = NULL;

	/* initialize the SAX parser */
	bzero (&dui->sax, sizeof (xmlSAXHandler));
	dui->sax.startDocument = start_doc;
	dui->sax.endDocument = end_doc;
	dui->sax.fatalError = fatal_err;
	dui->sax.error = err;
	dui->sax.warning = warn;
	dui->sax.startElement = start_elt;
	dui->sax.endElement = end_elt;
	dui->sax.characters = cdata_cb;

	dui->xml_filename = NULL;


	/* current state */
	dui->database = NULL;
	dui->window = NULL;
	dui->action = NULL;
	dui->report = NULL;
	dui->filter = NULL;

	dui->action_handlers = NULL;
	dui->report_handlers = NULL;
	dui->root_windows = NULL;

	dui->kvp = g_hash_table_new ((GHashFunc) dui_string_hash_func, 
	                   (GCompareFunc) dui_string_compare_func);
	return dui;
}


/* =================================================================== */

#define FREE_STR(str)  if(dui->str) g_free (dui->str); dui->str = NULL;

void 
dui_interface_destroy (DuiInterface *dui)
{
	GList *node;

	if (!dui) return;

	FREE_STR (app_name);
	FREE_STR (xml_filename);

	/* current state */
	dui->database = NULL;
	dui->window = NULL;
	dui->action = NULL;
	dui->report = NULL;
	dui->filter = NULL;

	/* the action and report handlers are deleted when thier 'parent'
	 * window is deleted; we don't need to delete them here. */
	g_list_free (dui->action_handlers);
	g_list_free (dui->report_handlers);

	for (node=dui->root_windows; node; node=node->next)
	{
		DuiWindow *win = node->data;
		dui_window_destroy (win);
	}
	g_list_free (dui->root_windows);

	/* XXX fixme hack alert need to for-each free the keys and values */
	g_hash_table_destroy (dui->kvp);

	/* XXX fixme need to dui_filter_destroy the filter list */

	/* XXX fixme need to dui_database_destroy() */
}

/* =================================================================== */

void
dui_interface_kvp_insert (DuiInterface *dui, const char *key, 
                                             const char *value)
{
	if (!dui || !key || !value) return;

	/* refuse insertion of new system keys */
	if (!strncmp (key, "/system/", 8)) return;

	PINFO ("(key=\'%s\' value=\'%s\')", key, value);
	g_hash_table_insert (dui->kvp, g_strdup(key), g_strdup(value));
}

const char *
dui_interface_kvp_lookup (DuiInterface *dui, const char *key)
{
	if (!dui) return NULL;

	/* intercept pre-defined system keys */
	if (!strncmp (key, "/system/", 8)) 
	{
		key += 8;
		if (!strcmp (key, "datetime"))
		{
			static char date_string [100];
			time_t now = time (0);
			gnc_secs_to_iso8601_buff (now, date_string);
			return date_string;
		}
		return NULL;
	}
	return g_hash_table_lookup (dui->kvp, key);
}

/* =================================================================== */

void 
dui_interface_parse (DuiInterface *dui, const char * filename)
{
	if (!dui || !filename) return;

	if (0 == g_file_exists (filename))
	{
		FATAL ("Can't find DWI interface description file %s\n", filename);
		return;
	}
	dui->xml_filename = g_strdup (filename);
	xmlSAXUserParseFile (&dui->sax, dui, filename);
}

/* =================================================================== */

DuiWindow *
dui_interface_find_window_by_name (DuiInterface *dui, const char * name)
{
	GList *node;

	if (!dui || !name) return NULL;

	for (node=dui->root_windows; node; node=node->next)
	{
		DuiWindow *win = node->data;
		const char *winname = dui_window_get_name (win);
		if (winname && !strcmp(winname, name))
		{
			return win;
		}
	}
	return NULL;
}

/* =================================================================== */

DuiReport *
dui_interface_find_report_by_name (DuiInterface *dui, const char * name)
{
	GList *node;

	if (!dui || !name) return NULL;

	for (node=dui->report_handlers; node; node=node->next)
	{
		DuiReport *rpt = node->data;
		const char *rptname = dui_report_get_name (rpt);
		if (rptname && !strcmp(rptname, name))
		{
			return rpt;
		}
	}
	return NULL;
}

/* =================================================================== */

DuiAction *
dui_interface_find_action_by_name (DuiInterface *dui, const char * name)
{
	GList *node;

	if (!dui || !name) return NULL;

	for (node=dui->action_handlers; node; node=node->next)
	{
		DuiAction *act = node->data;
		const char *actname = dui_action_get_name (act);
		if (actname && !strcmp(actname, name))
		{
			return act;
		}
	}
	return NULL;
}

/* =================================================================== */

void 
dui_interface_realize (DuiInterface *dui, int fatal_if_no_main)
{
	GList *node;
	int found_app_window = 0;

	if (!dui) return;

	for (node=dui->root_windows; node; node=node->next)
	{
		DuiWindow *win = node->data;
		int is_app_window = dui_window_is_app_main_window(win);
		if (is_app_window) 
		{
			dui_window_realize (win);
			found_app_window = 1;
		}
	}

	/* Without this test, its just to easy to write a dwi file 
	 * that just hangs there, doing nothing, and you scratch your 
	 * head wondering why ...
	 */
	if (fatal_if_no_main && 0 == found_app_window)
	{
		FATAL ("could not find an application main window!");
	}

}

/* ========================== END OF FILE ============================ */
