/*
 * FILE:
 * action.c
 *
 * FUNCTION:
 * Handler for a button click.
 *
 * HISTORY:
 * Linas Vepstas March 2002
 */

#include <string.h>

#include <glib.h>
#include <gnome.h>
#include <gtk/gtk.h>

#include "action.h"
#include "builder.h"
#include "dui-initdb.h"
#include "filter.h"
#include "perr.h"
#include "report.h"
#include "signal.h"
#include "util.h"
#include "window.h"

typedef struct DuiCheck_s DuiCheck;
typedef struct DuiRefresh_s DuiRefresh;

typedef enum {
	QUERY_NONE,
	QUERY_SIMPLE,
	QUERY_TABLES,
	QUERY_FIELDS
} MetaQueryType;

struct DuiAction_s
{
	DuiWindow *window;  /* the window in which we live */
	char * name;        /* our name */
	gboolean is_finalized;

	/* special DB queries that don't use normal SQL channels */
	MetaQueryType meta_qtype;
	
	/* table(s) to query */
	char * table;
	char * key_to_table;

	/* the query builder */
	SqlBuilder *sql_builder;

	/* a portion of the sql query */
	SqlBuilderQType qtype;
	GList *update_terms;
	GList *where_terms;

	/* cache of most recent sql query string; used for rerunning the query. */
	char * query_string;

	/* database connection on which to send the query */
	DuiDatabase *database;
	DuiDBConnection *db_conn;
	
	/* list of widgets whose values we put into kvp */
	GList *keys;
	
	/* the report that will report the results of the query */
	char * report_name;
	DuiReport * report;
	char * row_name;

	/* list of gtk signals that can trigger this action. */
	GList *signals;

	/* list of checks that must be performed first (and pass)
	 * before the form is allowed to proceed.
	 */
	GList *check_list;

	/* list of windows that need to be refreshed by this action */
	GList *refresh_list;

};

struct DuiRefresh_s
{
	char * reportname;
	DuiReport *report;
};

struct DuiCheck_s
{
	char * actionname;
	DuiAction *action;
};

static void dui_action_signal_handler (GtkObject *obj, DuiAction *act);

/* =================================================================== */
/* the following struct defines how widget fields (read as input) are
 * mapped to database fields (used to formulate the query)
 */

typedef struct DuiQueryFieldMap_s DuiQueryFieldMap;

struct DuiQueryFieldMap_s 
{
	DuiAction *act;
	
	/* the next four members specify the widget from which
	 * we will be reading data. */
	char * widgetname;
	GtkWidget * widget; 
	int column;
	int row;
	char * datakey;  /* key side of gtk_object_data */
	char * key;   /* the key side of a kvp pair */
	char * value; /* a hard-coded value to use, if the widget 
                  * is not present */
	char * filtername;
	DuiFilter *filter;

	/* the next three members specify how that widget's
	 * data is mapped to a database field. */
	char * fieldname;
	char * compareop;

	/* effing ctree not directly addressible by row number,
	 * so cache the node instead */
	GtkCTreeNode *ctn;
};


/* =================================================================== */

static void
terms_destroy (GList *list)
{
	GList *node;
	
	/* remove the insert/update terms */
	for (node=list; node; node=node->next)
	{
		DuiQueryFieldMap *qfm = node->data;

		g_free (qfm->widgetname);
		g_free (qfm->datakey);
		g_free (qfm->key);
		g_free (qfm->value);
		g_free (qfm->filtername);
		g_free (qfm->fieldname);
		g_free (qfm->compareop);
		g_free (qfm);
	}
	g_list_free (list);
}

/* =================================================================== */

DuiAction * 
dui_action_new (DuiWindow *win, const char * name)
{
	DuiAction *act;

	if (!win) return NULL;

	act = g_new (DuiAction, 1);
	act->window = win;
	act->name = g_strdup (name);
	act->is_finalized = FALSE;

	act->signals = NULL;

	act->meta_qtype = QUERY_NONE;
	act->table = NULL;
	act->key_to_table = NULL;
	act->sql_builder = NULL;

	act->qtype = 0;
	act->update_terms = NULL;
	act->where_terms = NULL;
	act->query_string = NULL;
	act->database = NULL;
	act->db_conn = NULL;

	act->keys = NULL;

	act->report_name = NULL;
	act->report = NULL;
	act->row_name = NULL;
	act->check_list = NULL;
	act->refresh_list = NULL;

	dui_window_add_action (win, act);
	return act;
}

void 
dui_action_destroy (DuiAction * act)
{
	GList *node;
	if (!act) return;

	act->window = NULL;
	g_free (act->name);
	act->name = NULL;

	act->meta_qtype = QUERY_NONE;
	g_free (act->table);
	act->table = NULL;
	g_free (act->key_to_table);
	act->key_to_table = NULL;
	sql_builder_destroy(act->sql_builder);
	act->sql_builder = NULL;
	act->qtype = 0;

	/* remove the insert/update terms */
	terms_destroy (act->update_terms);
	terms_destroy (act->where_terms);
	terms_destroy (act->keys);
	act->update_terms = NULL;
	act->where_terms = NULL;
	act->keys = NULL;

	g_free(act->report_name);
	g_free(act->row_name);
	act->report_name = NULL;
	act->report = NULL;
	act->row_name = NULL;

	/* the signals themselves will be reaped by window */
	g_list_free (act->signals);
	act->signals = NULL;

	act->database = NULL;
	act->db_conn = NULL;

	for (node=act->check_list; node; node=node->next)
	{
		DuiCheck *chk = node->data;
		g_free (chk->actionname);
		g_free (chk);
	}
	for (node=act->refresh_list; node; node=node->next)
	{
		DuiRefresh *rsh = node->data;
		g_free (rsh->reportname);
		g_free (rsh);
	}
	g_list_free (act->refresh_list);
	g_free (act);
}

/* =================================================================== */

const char * 
dui_action_get_name (DuiAction *act)
{
	if (!act) return NULL;
	if (act->name) return (act->name);
	return dui_window_get_name (act->window);
}

/* =================================================================== */

void 
dui_action_set_report (DuiAction * act, const char * dpyname, 
                       const char * row)
{
	if (!act) return;
	if (act->report_name) g_free (act->report_name);
	act->report_name = g_strdup (dpyname);
	act->row_name = g_strdup (row);
}
	
/* =================================================================== */

void 
dui_action_add_chain (DuiAction *act, const char * actname)
{
	DuiCheck *chk;
	if (!act) return;
	
	chk = g_new (DuiCheck, 1);
	chk->actionname = g_strdup (actname);
	chk->action = NULL;

	act->check_list = g_list_prepend (act->check_list, chk);
}

/* =================================================================== */

void 
dui_action_add_refresh (DuiAction *act, const char * name)
{
	DuiRefresh *rsh;
	if (!act) return;
	
	rsh = g_new (DuiRefresh, 1);
	rsh->reportname = g_strdup (name);
	rsh->report = NULL;

	act->refresh_list = g_list_prepend (act->refresh_list, rsh);
}

/* =================================================================== */

void 
dui_action_set_database (DuiAction * act, DuiDatabase *db)
{
	if (!act) return;
	act->database = db;
}
	
DuiDatabase *
dui_action_get_database (DuiAction * act)
{
	if (!act) return NULL;
	return act->database;
}

/* =================================================================== */

void 
dui_action_add_signal (DuiAction *act, const char *objectname, const char *signalname)
{
	DuiSignal *sig;

	if (!act || !objectname || !signalname) return;
	
	sig = dui_signal_new (act->window, objectname, signalname, "submit_form");
	act->signals = g_list_append (act->signals, sig);
}
			
/* =================================================================== */

void 
dui_action_set_table (DuiAction *act, const char * tables, 
                                      const char * key_to_tables,
                                      const char * sql_querytype)
{
	ENTER ("(act=%p, tables=%s, key=%s, qry=%s)", act, tables, key_to_tables, sql_querytype);
	if (!act || !sql_querytype) return;
	if (!strcasecmp (sql_querytype, "tables"))
	{
		act->meta_qtype = QUERY_TABLES;
		g_free (act->table);
		act->table = NULL;
		return;
	}

	if (!strcasecmp (sql_querytype, "fields"))
	{
		act->meta_qtype = QUERY_FIELDS;
		act->table = g_strdup (tables);
		act->key_to_table = g_strdup (key_to_tables);
		return;
	}

	/* Set up the sql statement builder so that the next set of
	 * of xml stanzas set up a valid sql statement. At this point,
	 * a table must be specified.
	 */
	if (!tables) return;
	act->meta_qtype = QUERY_SIMPLE;
	if (!strcasecmp (sql_querytype, "select")) act->qtype = SQL_SELECT;
	else if (!strcasecmp (sql_querytype, "insert")) act->qtype = SQL_INSERT;
	else if (!strcasecmp (sql_querytype, "update")) act->qtype = SQL_UPDATE;
	else if (!strcasecmp (sql_querytype, "delete")) act->qtype = SQL_DELETE;
	else 
	{
		SYNTAX ("unknown query type \'%s\'", sql_querytype);
		return;
	}

	act->sql_builder = sql_builder_new ();
	sql_builder_table (act->sql_builder, tables, act->qtype);

}

/* =================================================================== */

void 
dui_action_add_select_term (DuiAction * act, 
                           const char *field_name)
{
	if (!act || !field_name) return;
	sql_builder_set_str (act->sql_builder, field_name, NULL);
}


/* =================================================================== */

void 
dui_action_add_update_term (DuiAction * act, 
                           const char * widname,
                           int colnum,
                           const char *datakey,
                           const char *key,
                           const char *value,
                           const char *filter_name,
                           const char *field_name)
{
	DuiQueryFieldMap *qfm;

	if (!act || !field_name) return;

	/* save them up for when we need them */
	qfm = g_new (DuiQueryFieldMap, 1);
	qfm->act = act;
	qfm->widgetname = g_strdup (widname),
	qfm->widget = NULL,
	qfm->column = colnum;
	qfm->row = -1;
	qfm->datakey = g_strdup (datakey);
	qfm->key = g_strdup (key);
	qfm->value = g_strdup (value);
	qfm->filtername = g_strdup (filter_name);
	qfm->filter = NULL;
	qfm->fieldname = g_strdup (field_name);
	qfm->compareop = NULL;
	qfm->ctn = NULL;
	
	act->update_terms = g_list_append (act->update_terms, qfm);
}

/* =================================================================== */

void 
dui_action_add_where_term (DuiAction * act, 
                           const char * widname,
                           int colnum,
                           const char *datakey,
                           const char *key,
                           const char *value,
                           const char *filter_name,
                           const char *field_name,
                           const char *compare_op)
{
	DuiQueryFieldMap *qfm;

	if (!act || !field_name) return;

	/* save them up for when we need them */
	qfm = g_new (DuiQueryFieldMap, 1);
	qfm->act = act;
	qfm->widgetname = g_strdup (widname);
	qfm->widget = NULL;
	qfm->column = colnum;
	qfm->row = -1;
	qfm->datakey = g_strdup (datakey);
	qfm->key = g_strdup (key);
	qfm->value = g_strdup (value);
	qfm->filtername = g_strdup (filter_name);
	qfm->filter = NULL;
	qfm->fieldname = g_strdup (field_name);
	qfm->compareop = g_strdup (compare_op);
	qfm->ctn = NULL;

	act->where_terms = g_list_append (act->where_terms, qfm);
}

/* =================================================================== */

void 
dui_action_add_kvp_term (DuiAction * act, 
                           const char * widname,
                           int colnum,
                           const char *datakey,
                           const char *value,
                           const char *filter_name,
                           const char *key)
{
	DuiQueryFieldMap *qfm;

	if (!act) return;

	/* save them up for when we need them */
	qfm = g_new (DuiQueryFieldMap, 1);
	qfm->act = act;
	qfm->widgetname = g_strdup (widname);
	qfm->widget = NULL;
	qfm->column = colnum;
	qfm->row = -1;
	qfm->datakey = g_strdup (datakey);
	qfm->key = g_strdup (key);
	qfm->value = g_strdup (value);
	qfm->filtername = g_strdup (filter_name);
	qfm->filter = NULL;
	qfm->fieldname = NULL;
	qfm->compareop = NULL;
	qfm->ctn = NULL;

	act->keys = g_list_append (act->keys, qfm);
}

/* =================================================================== */
/* "final_setup" resolves window names, filter names, etc, without
 * actually touching the GUI in any way.
 */

static void
add_filters (DuiAction *act, GList *terms)
{
	GList *node;

	for (node=terms; node; node=node->next)
	{
		DuiQueryFieldMap *qfm = node->data;
		if (qfm->filtername)
		{
			qfm->filter =  dui_filter_find_by_name (qfm->filtername);
		}
	}
}

static void
final_setup (DuiAction *act)
{
	GList *node;
	DuiInterface *dui;

	if (act->is_finalized) return;
	act->is_finalized = TRUE;
	ENTER ("(act=%p)", act);
	
	/* locate the reports the action will use, and plug them in. */
	dui = dui_window_get_interface (act->window);
	act->report = dui_interface_find_report_by_name (dui, act->report_name);
	if (act->report_name && !act->report) 
	{
		const char * tmp = dui_action_get_name(act);
		SYNTAX ("In form \'%s\', can't find the report \'%s\'", 
		                tmp, act->report_name);
	}

	/* do the same thing for the check list */
	for (node=act->check_list; node; node=node->next)
	{
		DuiCheck *chk = node->data;
		chk->action = dui_interface_find_action_by_name (dui, chk->actionname);

		if (chk->actionname && !chk->action) 
		{
			const char * tmp = dui_action_get_name(act);
			SYNTAX ("In form \'%s\', can't find the chain form \'%s\'", 
			                tmp, chk->actionname);
		}
	}
	
	/* do the same thing for the refresh list */
	for (node=act->refresh_list; node; node=node->next)
	{
		DuiRefresh *rsh = node->data;
		rsh->report = dui_interface_find_report_by_name (dui, rsh->reportname);

		if (rsh->reportname && !rsh->report) 
		{
			const char * tmp = dui_action_get_name(act);
			SYNTAX ("In form \'%s\', can't find the refresh report \'%s\'", 
			                tmp, rsh->reportname);
		}
	}
	
	add_filters (act, act->where_terms);
	add_filters (act, act->update_terms);
	add_filters (act, act->keys);

	LEAVE ("(act=%p)", act);
}

/* =================================================================== */

static void
get_clist_row (GtkCList *clist,
               gint            row,
               gint            column,
               GdkEvent       *event,
               DuiQueryFieldMap *qfm)
{
	qfm->row = row;
}

static void
unget_clist_row (GtkCList *clist,
               gint            row,
               gint            column,
               GdkEvent       *event,
               DuiQueryFieldMap *qfm)
{
	qfm->row = -1;
}

static void
get_ctree_row (GtkCList *clist,
               GtkCTreeNode *ctn,
               gint            column,
               DuiQueryFieldMap *qfm)
{
	qfm->ctn = ctn;
}

static void
unget_ctree_row (GtkCList *clist,
               GtkCTreeNode *ctn,
               gint            column,
               DuiQueryFieldMap *qfm)
{
	qfm->ctn = NULL;
}

static void
add_watchers (DuiAction *act, GList *terms)
{
	GList *node;

	for (node=terms; node; node=node->next)
	{
		DuiQueryFieldMap *qfm = node->data;
		if (qfm->widgetname)
		{
			qfm->widget =  dui_window_get_widget (act->window, qfm->widgetname);

			/* There's no way to ask the clist what row has been selected,
			 * so we have to listen for events instead. */
			if (GTK_IS_CTREE(qfm->widget))
			{
				gtk_signal_connect(GTK_OBJECT(qfm->widget), "tree_select_row",
		   	           GTK_SIGNAL_FUNC(get_ctree_row), qfm);
				gtk_signal_connect(GTK_OBJECT(qfm->widget), "tree_unselect_row",
		   	           GTK_SIGNAL_FUNC(unget_ctree_row), qfm);
			}
			else
			if (GTK_IS_CLIST (qfm->widget))
			{
				gtk_signal_connect(GTK_OBJECT(qfm->widget), "select_row",
		   	           GTK_SIGNAL_FUNC(get_clist_row), qfm);
				gtk_signal_connect(GTK_OBJECT(qfm->widget), "unselect_row",
		   	           GTK_SIGNAL_FUNC(unget_clist_row), qfm);
			}
		}
	}
}

void
dui_action_do_realize (DuiAction *act)
{
	GList *node;
	if (!act) return;

	ENTER ("(act=%p)", act);
	/* Don't bother if the parent window doesn't exist. */
	/* Technically, this is an error, it shouldn't happen ... */
	if (0 == dui_window_is_realized(act->window)) return;

	final_setup (act);

	for (node=act->signals; node; node=node->next)
	{
		GtkWidget *w;
		DuiSignal *sig = node->data;
		w = dui_window_get_widget (act->window, dui_signal_get_widget(sig));
		/* might not be a widget if the root window doesn't exist */
		if (!w) continue;  

		dui_signal_connect (sig, GTK_OBJECT(w),
		         GTK_SIGNAL_FUNC(dui_action_signal_handler), act);
	}

	add_watchers (act, act->where_terms);
	add_watchers (act, act->update_terms);
	add_watchers (act, act->keys);

	/* now get the database too */
	act->db_conn = dui_database_do_realize (act->database);

	LEAVE ("(act=%p)", act);
}

/* =================================================================== */

int
dui_action_is_realized (DuiAction *act)
{
	if (!act) return 0;

	return dui_window_is_realized(act->window);
}

/* =================================================================== */

static const char * qfm_get_value (DuiQueryFieldMap *qfm);

static const char *
widget_get_value (DuiQueryFieldMap *qfm)
{
	GtkWidget *widget = qfm->widget;
	const char * val = NULL;

	if (!widget) return NULL;   /* this shouldn't happen, really */

	/* supported widgets, in pseudo-alpha order;
	 * gotta ctree first */
	if (GTK_IS_CTREE(widget))
	{
		if (NULL == qfm->ctn) return NULL;
		gtk_ctree_node_get_text (GTK_CTREE(widget), qfm->ctn, qfm->column, (gchar **) &val);
	}
	else
	if (GTK_IS_CLIST(widget))
	{
		if (0 > qfm->row) return NULL;
		gtk_clist_get_text (GTK_CLIST(widget), qfm->row, qfm->column, (gchar **) &val);
	}
	else
	if (GTK_IS_MENU(widget))
	{
		static char menuval[18];  /* hack alert not thread safe */
		GList *node; 
		int i=0;
		GtkWidget *w = gtk_menu_get_active (GTK_MENU(widget));
		for (node=GTK_MENU_SHELL(widget)->children; node; node=node->next)
		{
			if (w == node->data)
			{
				snprintf (menuval, 18, "%d", i);
				return menuval;		  
			}
			i++;
		}
	}
	else
	if (GTK_IS_OPTION_MENU(widget))
	{
		DuiQueryFieldMap dqfm;
		dqfm = *qfm;
		dqfm.widget = GTK_OPTION_MENU(widget)->menu;
		dqfm.filter = NULL;

		/* pass-through on menu, let the child report */
		val = qfm_get_value (&dqfm);
	}
	else
	if (GTK_IS_RADIO_BUTTON(widget))
	{
		GSList *node;

		/* loop over buttons in group, get data for the one that was pushed */
		node = gtk_radio_button_group (GTK_RADIO_BUTTON (widget));
		for ( ; node; node=node->next)
		{
			gboolean pushed;
			pushed = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(node->data));
			if (pushed) break;
		}
		if (!node) return NULL;
		val = gtk_object_get_data (GTK_OBJECT(node->data), "/system/data");
	}
	else
	if (GTK_IS_TOGGLE_BUTTON(widget))
	{
		static char togbuff[2];  /* hack alert not thread safe */
		gboolean state;
	  	state = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(widget));
		if (state) togbuff[0] = '1'; else togbuff[0] = '0';
		togbuff[1] = 0;  
		val = togbuff;
	}
	else
	if (GTK_IS_SPIN_BUTTON(widget))
	{
		static char spinbuff[40];    /* hack alert not thread safe */
	  	gfloat flt = gtk_spin_button_get_value_as_float (GTK_SPIN_BUTTON(widget));
		snprintf (spinbuff, 40, "%24.18g", (double) flt);
		return spinbuff;
	}
	else
	if (GTK_IS_ENTRY(widget))
	{
	  	val = gtk_entry_get_text (GTK_ENTRY(widget));
	}
	else
	if (GTK_IS_LABEL(widget))
	{
	  	gtk_label_get (GTK_LABEL(widget), (gchar **) &val);
	}
	else
	if (GTK_IS_TEXT(widget))
	{
	  	val = xxxgtk_text_get_text (GTK_TEXT(widget));
	}
	else
	if (GTK_IS_COMBO(widget))
	{
	  	val = gtk_entry_get_text (GTK_ENTRY(GTK_COMBO(widget)->entry));
	}
	else
	if (GTK_IS_RANGE(widget))
	{
		GtkAdjustment *adj = gtk_range_get_adjustment (GTK_RANGE(widget));
		double flt = adj->value;
		static char rangebuff[40];
		snprintf (rangebuff, 40, "%24.18g", flt);
	  	val = rangebuff;
	}
	else
	if (GNOME_IS_FILE_ENTRY (widget))
	{
		GtkWidget * entry = gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY(widget));
		val = gtk_entry_get_text (GTK_ENTRY (entry));
	}
	else
	if (GTK_IS_FILE_SELECTION (widget))
	{
		val = gtk_entry_get_text (GTK_ENTRY (GTK_FILE_SELECTION
		                   (widget)->selection_entry));
	}
	else
	if (GNOME_IS_DATE_EDIT(widget))
	{
		static char datebuff[50];
		time_t thyme = gnome_date_edit_get_date (GNOME_DATE_EDIT(widget));
		gnc_secs_to_iso8601_buff (thyme, datebuff);
		return datebuff;
	}
	else
	if (GTK_IS_BIN(widget))
	{
		/* catch-all handles both GtkButton and GtkOptionMenu */
		DuiQueryFieldMap dqfm;
		dqfm = *qfm;
		dqfm.widget = GTK_BIN(widget)->child;
		dqfm.filter = NULL;

		/* pass-through on bin, let the child report */
		val = qfm_get_value (&dqfm);
	}
	else
	{
	  	PERR ("unsupported <where> or <update> widget type %s",
			gtk_type_name(GTK_OBJECT_TYPE(widget)));
	}
	return val;
}

/* =================================================================== */

static const char *
widget_get_data (DuiQueryFieldMap *qfm)
{
	GtkWidget *widget = qfm->widget;
	const char * val = NULL;

	if (!widget) return NULL;   /* this shouldn't happen, really */

	/* supported widgets, in alpha order */
	if (GTK_IS_CTREE(widget))
	{
		GHashTable *tbl;
		if (NULL == qfm->ctn) return NULL;
		tbl = gtk_ctree_node_get_row_data (GTK_CTREE(widget), qfm->ctn);
		if (tbl) val = g_hash_table_lookup (tbl, qfm->datakey);
	}
	else
	if (GTK_IS_CLIST(widget))
	{
		GHashTable *tbl;
		if (0 > qfm->row) return NULL;
		tbl = gtk_clist_get_row_data (GTK_CLIST(widget), qfm->row);
		if (tbl) val = g_hash_table_lookup (tbl, qfm->datakey);
	}
	else
	if (GTK_IS_RADIO_BUTTON(widget))
	{
		GSList *node;

		/* loop over buttons in group, get data for the one that was pushed */
		node = gtk_radio_button_group (GTK_RADIO_BUTTON (widget));
		for ( ; node; node=node->next)
		{
			gboolean pushed;
			pushed = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(node->data));
			if (pushed) break;
		}
		if (!node) return NULL;
		val = gtk_object_get_data (GTK_OBJECT(node->data), qfm->datakey);
	}
	else
	{
		val = gtk_object_get_data (GTK_OBJECT(widget), qfm->datakey);
	}
	return val;
}

/* =================================================================== */
/* get the value, from the widget a kvp or hardcoded. Then filter it */

static const char *
qfm_get_value (DuiQueryFieldMap *qfm)
{
	const char * fieldval;
	if (qfm->widget)
	{
		if (qfm->datakey)
		{
			fieldval = widget_get_data (qfm);
		}
		else 
		{
			fieldval = widget_get_value (qfm);
		}
	}
	else
	if (qfm->key)
	{
		DuiInterface *dui = dui_window_get_interface (qfm->act->window);
		fieldval = dui_interface_kvp_lookup (dui, qfm->key);
	}
	else
	{
		fieldval = qfm->value;
	}

	if (qfm->filter) fieldval = dui_filter_apply (qfm->filter, fieldval);
	return fieldval;
}

/* =================================================================== */

static int
dui_action_run (DuiAction *act)
{
	DuiInterface *dui;
	const char * stmt;
	DuiDBRecordSet *recs = NULL;
	GList *node;
	int rc = 0;
	
	if (!act) return 0;
	ENTER ("(act=%p \'%s\')", act, dui_action_get_name (act));
	final_setup (act);

	/* loop over chained actions; quit if one of them handled things */
	for (node=act->check_list; node; node=node->next)
	{
		DuiCheck *chk = node->data;
		rc = dui_action_run (chk->action);
		if (rc) return rc;
	}

	/* loop over keys, fetch values and stuff them into kvp */
	dui = dui_window_get_interface (act->window);
	for (node = act->keys; node; node=node->next)
	{
		DuiQueryFieldMap *qfm = node->data;
		const char * fieldval;
		if (qfm->widget)
		{
			if (qfm->datakey)
			{
				fieldval = widget_get_data (qfm);
			}
			else
			{
				fieldval = widget_get_value (qfm);
			}
		}
		else
		{
			fieldval = qfm->value;
		}
		
		dui_interface_kvp_insert (dui, qfm->key, fieldval);
	}
	
	switch (act->meta_qtype)
	{
		case QUERY_NONE: break;
		case QUERY_SIMPLE: break;
		case QUERY_TABLES:
			if (act->db_conn)
			{
				recs = dui_connection_tables(act->db_conn);
			}
			break;
		case QUERY_FIELDS:
			if (act->db_conn)
			{
				const char * table;
				table = act->table;
				if (act->key_to_table)
				{
					table = dui_interface_kvp_lookup (dui, act->key_to_table);
					PINFO ("lookup key to table=\'%s\', find table=\'%s\'",
					              act->key_to_table, table);
				}
				PINFO ("perform fields query with table=\'%s\'", table);
				recs = dui_connection_table_columns(act->db_conn, table);
			}
			break;
	}

	/* If we have data we need to query, then do the query.
	 * (This is the usual case.  Sometimes we have no data, e.g.
	 * when setting a form to a constant value e.g. to initialize it)
	 */
	if (act->sql_builder)
	{
		GList *node;
		SqlBuilder *dupe;
		short have_where_terms = 0;

		g_free (act->query_string);
		act->query_string = NULL;

		/* make a copy, add the appropriate where clauses */
		dupe = sql_builder_copy (act->sql_builder);
	
		for (node = act->update_terms; node; node=node->next)
		{
			DuiQueryFieldMap *qfm = node->data;
			const char * fieldval = qfm_get_value (qfm);
			
			sql_builder_set_str (dupe, qfm->fieldname, fieldval);
		}
	
		for (node = act->where_terms; node; node=node->next)
		{
			DuiQueryFieldMap *qfm = node->data;
			const char * fieldval = qfm_get_value (qfm);
	
			/* fields left blank don't match */
			fieldval = whitespace_filter (fieldval);
			if (fieldval)
			{
				sql_builder_where_str (dupe, qfm->fieldname, 
					fieldval, qfm->compareop);
				have_where_terms = 1;
			}
		}
	
		/* Perform the query only if its not a wild-card
		 * update or delete.  Wildcard updates/deletes are baaad. */
		if (have_where_terms || ((SQL_UPDATE != act->qtype) && 
		                         (SQL_DELETE != act->qtype)))
		{
			/* get the actual sql statement */
			stmt = sql_builder_query (dupe);
			act->query_string = g_strdup (stmt);
	
			/* now run the query */
			if (act->db_conn)
			{
				recs = dui_connection_exec(act->db_conn, stmt);
			}
		}	
		sql_builder_destroy (dupe);
	}

	/* now report the results */
	dui_report_set_last_action (act->report, act, act->row_name);
	rc = dui_report_show_data (act->report, recs, act->row_name);

	/* don't need it no more */
	dui_recordset_free (recs);

	/* refresh all windows that show data modified by this query */
	for (node=act->refresh_list; node; node=node->next)
	{
		DuiRefresh *rsh = node->data;
		dui_report_refresh (rsh->report);
	}
	LEAVE ("(act=%p \'%s\') rc=%d", act, dui_action_get_name (act), rc);
	return rc;
}

static void
dui_action_signal_handler (GtkObject *obj, DuiAction *act)
{
	dui_action_run (act);
}

/* =================================================================== */

void
dui_action_rerun_last_query (DuiAction *act)
{
	DuiDBRecordSet *recs = NULL;

	if (!act) return;

	ENTER ("(act=%p \'%s\')", act, dui_action_get_name (act));

	if (!act->db_conn || !act->query_string) 
	{
		dui_report_show_data (act->report, NULL, act->row_name);
	}
	else
	{
		/* display the results of the query */
		recs = dui_connection_exec(act->db_conn, act->query_string);
		dui_report_show_data (act->report, recs, act->row_name);
		dui_recordset_free (recs);
	}
	LEAVE ("(act=%p \'%s\')", act, dui_action_get_name (act));
}

/* ========================== END OF FILE ============================ */
