
/* 
 * FILE:
 * dui-initdb.c
 *
 * FUNCTION:
 * Wrapper for different database drivers.
 *
 * HISTORY:
 * created by Linas Vepstas  March 2002
 */

#include <string.h>

#include <glib.h>

#include "dui-initdb.h"
#include "dui-odbc.h"

typedef enum
{
	ODBC = 1,
} DuiDBProvider;

struct DuiDBConnection_s
{
	DuiDBProvider     provider;
	DuiODBCConnection *odbc;
};

struct DuiDBRecordSet_s
{
	DuiDBConnection *conn;
	DuiODBCRecordSet *rs_odbc;
};

/* ========================================================= */

DuiDBConnection * 
dui_connection_new (const char * provider,
                    const char * dbname, 
                    const char * username)
{
	DuiDBConnection *dbc = NULL;

	if (!provider) return NULL;

	if (!strcmp (provider, "odbc"))
	{
		DuiODBCConnection *odbc;
		odbc = dui_odbc_connection_new (dbname, username);
		if (!odbc) return NULL;
		
		dbc = g_new (DuiDBConnection, 1);
		dbc->provider = ODBC;
		dbc->odbc = odbc;
		return dbc;
	}

	return NULL;
}

/* ========================================================= */

void 
dui_connection_free (DuiDBConnection *conn)
{
	if (!conn) return;
	
	switch (conn->provider)
	{
		case ODBC:
			dui_odbc_connection_free (conn->odbc);
			break;
	}
	conn->provider = 0;
	g_free (conn);
}

/* ========================================================= */

DuiDBRecordSet * 
dui_connection_exec (DuiDBConnection *conn, const char * buff)
{
	DuiDBRecordSet *rs;
	
	if (!conn) return NULL;
	
	switch (conn->provider)
	{
		case ODBC:
			rs = g_new (DuiDBRecordSet,1);
			rs->conn = conn;
			rs->rs_odbc = dui_odbc_connection_exec (conn->odbc, buff);
			return rs;
			break;
	}
	return NULL;
}

/* ========================================================= */

DuiDBRecordSet * 
dui_connection_tables (DuiDBConnection *conn)
{
	DuiDBRecordSet *rs;
	
	if (!conn) return NULL;
	
	switch (conn->provider)
	{
		case ODBC:
			rs = g_new (DuiDBRecordSet,1);
			rs->conn = conn;
			rs->rs_odbc = dui_odbc_connection_tables (conn->odbc);
			return rs;
			break;
	}
	return NULL;
}

/* ========================================================= */

DuiDBRecordSet * 
dui_connection_table_columns (DuiDBConnection *conn, const char * tablename)
{
	DuiDBRecordSet *rs;
	
	if (!conn) return NULL;
	
	switch (conn->provider)
	{
		case ODBC:
			rs = g_new (DuiDBRecordSet,1);
			rs->conn = conn;
			rs->rs_odbc = dui_odbc_connection_table_columns (conn->odbc, tablename);
			return rs;
			break;
	}
	return NULL;
}

/* ========================================================= */

void 
dui_recordset_free (DuiDBRecordSet *rs)
{
	if (!rs) return;
	
	switch (rs->conn->provider)
	{
		case ODBC:
			dui_odbc_recordset_release (rs->rs_odbc);
			break;
	}

	g_free (rs);
}

/* ========================================================= */

int 
dui_recordset_fetch_row (DuiDBRecordSet *rs)
{
	if (!rs) return -1;
	
	switch (rs->conn->provider)
	{
		case ODBC:
			return dui_odbc_recordset_fetch_row (rs->rs_odbc);
			break;
	}

	return -1;
}


/* ========================================================= */

const char * 
dui_recordset_get_value (DuiDBRecordSet *rs, const char * fieldname)
{
	if (!rs) return NULL;
	
	switch (rs->conn->provider)
	{
		case ODBC:
			return dui_odbc_recordset_get_value (rs->rs_odbc, fieldname);
			break;
	}

	return NULL;
}


/* ==================== END OF FILE ======================== */
