/*****
*
* This file is part of the OMS program.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by 
* the Free Software Foundation; either version 2, or (at your option)
* any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; see the file COPYING.  If not, write to
* the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*
*****/

/*  Open Media System - oms
 * 
 *  configuration-management, -file i/o
 *
 * Copyright (C) 1999  Thomas Mirlacher, Guenter Bartsch
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config_store.h"
#include <fcntl.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <sys/stat.h>

#include <oms/log.h>

GData *pConfigData;


/**
 * get string from config-file
 **/

gchar *cfg_get_val_string (const gchar *strKey_) 
{
	return (gchar*) g_datalist_get_data (&pConfigData, strKey_);
}


/**
 * get string from config-file, return default if value is not set yet
 **/

gchar *cfg_get_defval_string (const gchar *strKey_, gchar *strDefault_)
{
	gchar *str = cfg_get_val_string (strKey_);

	if (str)
		return str;
	else {
		cfg_set_val_string (strKey_, strDefault_);
		return strDefault_;
	}
} 


/**
 * get int value from config file
 **/

int cfg_get_val_int (const gchar *strKey_, int nDefaultValue_) 
{
	gpointer pData ;
	gchar    *pDataString;
	int      nValue;

	pData = g_datalist_get_data (&pConfigData, strKey_);

	if (!pData) {
		LOG (LOG_DEBUG, "config key %s not found", strKey_);

		cfg_set_val_int (strKey_, nDefaultValue_);
      
		return nDefaultValue_;
	} else {
		pDataString = (gchar*) pData;
      
		sscanf (pDataString, "%d", &nValue);
      
		LOG (LOG_DEBUG, "GET config key %s is %s int => %d", strKey_, pDataString, nValue);

		return nValue;
	}

}


/**
 * get string value from config file
 **/

void cfg_set_val_string (const gchar *strKey_, gchar *strValue_) 
{
	gpointer pData;

// free old data
	if ((pData = g_datalist_get_data (&pConfigData, strKey_))) {
		g_datalist_remove_data (&pConfigData, strKey_);
	}

	LOG (LOG_DEBUG,  "CONFIG key STRING %s => %s", strKey_, strValue_);

	g_datalist_set_data (&pConfigData, strKey_, g_strdup(strValue_)) ;

	if (pData)
		g_free (pData);
}


/**
 * put integer value int config file
 **/

void cfg_set_val_int (const gchar *strKey_, int nValue_) 
{
	gchar str[256];

	sprintf (str, "%d", nValue_);

	cfg_set_val_string (strKey_, str);
}


/**
 * init config file handling
 **/

void cfg_init () 
{
	LOG (LOG_DEBUG, " ");
	g_datalist_init (&pConfigData);

	cfg_read ();
}


/**
 * read config file
 **/

static void _cfg_read_file (FILE *pFile_) 
{
	gchar line[256];
	gchar *pEqual, *pValue;
	gint nLength;

	while (fgets (&line[0], 256, pFile_)) {
		nLength = strlen(line);
		line [nLength-1] = 0; // delete \n

		pEqual = strchr (line, '=');

		if (pEqual) {
			pValue = ((gchar *)pEqual+1);
			*pEqual = 0;

			LOG (LOG_DEBUG, "Value : %s", pValue);

			cfg_set_val_string (g_strdup(&line[0]), pValue);
		}
	}
}


/**
 * read config
 **/

void cfg_read () 
{
	FILE *pFConfig;
	GString *strFilename;

	g_datalist_clear (&pConfigData);

// read global configuration

// read user specific configuration
	strFilename = g_string_new (getenv("HOME"));

	g_string_append (strFilename, "/.oms/config");

	LOG (LOG_DEBUG, "private config file %s", strFilename->str);

	pFConfig = fopen (strFilename->str, "r");

	if (pFConfig != NULL) {
		_cfg_read_file (pFConfig);
		fclose (pFConfig);
	}

	g_string_free (strFilename, FALSE);
}


/**
 * write a line to the config file
 **/

static void _cfg_write_line  (GQuark key_id,
			  gpointer data,
			  gpointer user_data) 
{
	FILE *pFConfig = (FILE *) user_data;
	gchar *strKey = g_quark_to_string (key_id);

	//LOG (LOG_DEBUG, "Writing %s = %s\n", strKey, data);
	LOG (LOG_DEBUG,"%s=%s",strKey,(gchar*) data);
	fprintf (pFConfig,"%s=%s\n",strKey,(gchar *) data);
}


/**
 * save config
 **/

void cfg_save (void) 
{
	FILE *pFConfig;
	GString *strFilename;
	struct stat stat_buffer;

	LOG (LOG_DEBUG, " ");

	strFilename = g_string_new (getenv ("HOME"));
	g_string_append (strFilename, "/.oms");

	if (stat (strFilename->str, &stat_buffer) != 0)
		mkdir(strFilename->str, S_IRUSR | S_IWUSR | S_IXUSR);

	g_string_append (strFilename, "/config");

	if ((pFConfig = fopen (strFilename->str, "w"))) {
		LOG (LOG_DEBUG, "CONFIG Writing %s", strFilename->str);

		g_datalist_foreach (&pConfigData,
			_cfg_write_line,
			pFConfig);

		fclose (pFConfig);
	}

	g_string_free (strFilename, FALSE);
}
