//PLUGIN_INFO(INFO_NAME, "WAV file audio driver output");
//PLUGIN_INFO(INFO_AUTHOR, "Aaron Holtzman <aholtzma@ess.engr.uvic.ca>");

/*
 *
 *  audio_out_wav.c
 *    
 *	Copyright (C) Aaron Holtzman - May 1999
 *
 *  This file is part of ac3dec, a free Dolby AC-3 stream decoder.
 *	
 *  ac3dec is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  ac3dec is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 *  This file is based on output_linux.c by Aaron Holtzman.
 *  All .wav modifications were done by Jorgen Lundman <lundman@lundman.net>
 *  Any .wav bugs and errors should be reported to him.
 *
 *
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>

#include <oms/oms.h>
#include <oms/plugin/output_audio.h>

#define WAVE_FORMAT_PCM  0x0001
#define FORMAT_MULAW     0x0101
#define IBM_FORMAT_ALAW  0x0102
#define IBM_FORMAT_ADPCM 0x0103

struct riff_struct {
	u_char id[4];		/*
				 * RIFF 
				 */
	u_long len;
	u_char wave_id[4];	/*
				 * WAVE 
				 */
};

struct chunk_struct {
	u_char id[4];
	u_long len;
};

struct common_struct {
	u_short wFormatTag;
	u_short wChannels;
	u_long dwSamplesPerSec;
	u_long dwAvgBytesPerSec;
	u_short wBlockAlign;
	u_short wBitsPerSample;	/*
				 * Only for PCM 
				 */
};

struct wave_header {
	struct riff_struct riff;
	struct chunk_struct format;
	struct common_struct common;
	struct chunk_struct data;

	struct riff_struct riffdata;
	struct chunk_struct dataformat;
};

static int fd;

static struct wave_header wave;
static void (*old_sig) (int);

static int _audio_wav_open (void *plugin, void *name);
static int _audio_wav_close (void *plugin);
static int _audio_wav_setup (plugin_output_audio_attr_t * attr);
static int _audio_wav_write (const void *buf, size_t num_bytes);
static int _audio_wav_check (void);

static plugin_output_audio_t audio_wav = {
	open:_audio_wav_open,
	close:_audio_wav_close,
	setup:_audio_wav_setup,
	write:_audio_wav_write,
	check:_audio_wav_check,
	config:NULL,
};

static void signal_handler (int sig)
{
// FIXME
	_audio_wav_close (NULL);
	signal (sig, old_sig);
	raise (sig);
}

static int _audio_wav_open (void *plugin, void *name)
{
	if ((fd = open ((char *) name, O_WRONLY | O_TRUNC | O_CREAT, 0644))
	    < 0) {
		LOG (LOG_ERROR, "%s: Opening audio file %s",
		     strerror (errno), (char *) name);
		return -1;
	}

	return 0;
}

static int _audio_wav_close (void *plugin)
{
	off_t size;

// Find how long our file is in total, including header
	size = lseek (fd, 0, SEEK_CUR);

	/*
	 * If it's a pipe, end gracefully! 
	 */
	if (size < 0) {
		if (errno == ESPIPE) {
			LOG (LOG_ERROR,
			     "lseek failed - working on a pipe");
		} else {
			LOG (LOG_ERROR,
			     "lseek failed - wav-header is corrupt");
		}
		goto error;
	}
// Rewind file
	if (lseek (fd, 0, SEEK_SET) < 0) {
		LOG (LOG_ERROR, "rewind failed - wav-header is corrupt");
		goto error;
	}
// Fill some information into the wav-header
	size -= 8;

	wave.riff.len = size;
	size -= 20 + sizeof (struct common_struct);

	wave.data.len = size;

	if (write (fd, &wave, sizeof (wave)) < sizeof (wave)) {
		LOG (LOG_ERROR,
		     "wav-header write failed -- file is corrupt");
		goto error;
	}

      error:
	return close (fd);
}

static int _audio_wav_setup (plugin_output_audio_attr_t * attr)
{
	/*
	 * Write out a ZEROD wave header first 
	 */
	memset (&wave, 0, sizeof (wave));

	strncpy (wave.riff.id, "RIFF", 4);
	strncpy (wave.riff.wave_id, "WAVE", 4);
	strncpy (wave.format.id, "fmt ", 4);
	wave.format.len = sizeof (struct common_struct);

	/*
	 * Store information 
	 */
	wave.common.wFormatTag = WAVE_FORMAT_PCM;

	/*
	 * Store information 
	 */
	wave.common.wChannels = attr->channels;
	wave.common.dwSamplesPerSec = attr->speed;
	wave.common.dwAvgBytesPerSec = attr->channels *
	    attr->format * attr->speed / 8;
	wave.common.wBlockAlign = attr->channels * attr->format;
	wave.common.wBitsPerSample = attr->format;

	strncpy (wave.data.id, "data", 4);
	wave.data.len = 0xffffffff;

	if (write (fd, &wave, sizeof (wave)) != sizeof (wave)) {
		LOG (LOG_ERROR, "failed to write wav-header: %s",
		     strerror (errno));
		goto error;
	}
// FIXME: enable atexit again
	//atexit (_wav_close);

	//install our handler to properly write the riff header
	old_sig = signal (SIGINT, signal_handler);

	return 1;

      error:
	if (fd >= 0)
		close (fd);

	return 0;
}

static int _audio_wav_write (const void *buf, size_t num_bytes)
{
	return write (fd, buf, num_bytes);
}

static int _audio_wav_check (void)
{
	return 0;
}

int PLUGIN_INIT (audio_wav) (char *whoami) {
	pluginRegister (whoami,
			PLUGIN_ID_OUTPUT_AUDIO,
			NULL, "wav", NULL, &audio_wav);

	return 0;
}

void PLUGIN_EXIT (audio_wav) (void) {
}
