/*****
*
* This file is part of the OMS program.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by 
* the Free Software Foundation; either version 2, or (at your option)
* any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; see the file COPYING.  If not, write to
* the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*
*****/

#if HAVE_CONFIG_H
# include <config.h>
#endif

#include <oms/oms.h>
#include "linux_raw.h"

#ifndef HAVE_LINUX_RAW_H
int raw_open (const char *dev) {
	return -1;
}
int raw_close (int fd) {
	return -1;
}
#else

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <inttypes.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>

#include <linux/raw.h>

static struct rawdev_name {
	char const *raw_ctl;
	char const *raw_nr;
} rawdev_names[] = {
	{ "/dev/raw",    "/dev/raw%d" },     /* devices.txt from >2.3.99 kernels */
	{ "/dev/rawctl", "/dev/raw/raw%d" }, /* found on redhat 6.2 */
	{ NULL }
};


/**
 *
 **/

int raw_open (const char *dev)
{
	struct stat info;
	struct rawdev_name *rd_name;
	int fd_rawctl = 0, fd_raw = 0;
	struct raw_config_request raw_conf;
	char devname[256];
   
/* Step 1: find raw control device */
	for(rd_name=rawdev_names;rd_name->raw_ctl!=NULL;++rd_name) {
		LOG (LOG_DEBUG, "RAW: opening raw control `%s'",rd_name->raw_ctl);

		if ((fd_rawctl = open (rd_name->raw_ctl,O_RDWR))!=-1)
/* control device found */
			break;
/* control device not found */
		if(errno==ENOENT || errno==EISDIR)
			continue;

		LOG (LOG_DEBUG, "while opening raw control device `%s': %s", rd_name->raw_ctl,strerror(errno));
		return -1;
	}

	if (!rd_name->raw_ctl) {
		LOG (LOG_DEBUG, "no raw control device found");
		return -1;
	}

/* Step 2: examine device to open */
	if(stat(dev,&info)==-1) {
		LOG (LOG_DEBUG, "stat(`%s'): %s", dev, strerror (errno));
		return -1;
	}

	raw_conf.block_major = major (info.st_rdev);
	raw_conf.block_minor = minor (info.st_rdev);

	for(raw_conf.raw_minor=1; raw_conf.raw_minor <= LINUX_RAWDEV_MAX; raw_conf.raw_minor++) {
		LOG (LOG_DEBUG, "RAW: binding to device %d", raw_conf.raw_minor);

		if (ioctl (fd_rawctl, RAW_SETBIND, &raw_conf)) {
			if(errno==EBUSY)
				continue;
			LOG (LOG_DEBUG, "ioctl(`%s',RAW_SETBIND,...): %s", rd_name->raw_ctl, strerror (errno));
			if (close (fd_rawctl))
				LOG (LOG_DEBUG, "close(`%s'): %s", rd_name->raw_ctl, strerror (errno));
			return -1;
		}
		break;
	}
   
	if (close (fd_rawctl))
		LOG (LOG_DEBUG, "close(`%s'): %s", rd_name->raw_ctl, strerror (errno));
   
	if (raw_conf.raw_minor==LINUX_RAWDEV_MAX) {
		LOG (LOG_WARNING, "no free raw device found");
		return -1;
	}

	snprintf(devname,sizeof(devname),rd_name->raw_nr,raw_conf.raw_minor);
	LOG (LOG_DEBUG, "RAW: opening device `%s'", devname);

	if ((fd_raw = open (devname, O_RDONLY)) == -1) {
		LOG (LOG_DEBUG, "open(`%s',O_RDONLY): %s", devname, strerror (errno));
		return -1;
	}

	LOG (LOG_DEBUG, "RAW: everything went ok");

	return fd_raw;
}


/**
 *
 **/

int raw_close (int raw_fd)
{
	return close (raw_fd);
}
#endif

