/*****
*
* This file is part of the OMS program.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by 
* the Free Software Foundation; either version 2, or (at your option)
* any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; see the file COPYING.  If not, write to
* the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*
*****/

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/ioctl.h>

#include <oms/log.h>
#include <oms/plugin/input.h>

#include "input_vcd.h"

typedef struct {
	uint8_t sync		[12];
	uint8_t header		[4];
	uint8_t subheader	[8];
	uint8_t data		[2324];
	uint8_t spare		[4];
} cdsector;

static vcd_priv_t priv;

static int _vcd_open		(void *self, void *name);
static int _vcd_close		(void *self);
static ssize_t _vcd_read	(void *self, void *buf, size_t count);
static int _vcd_seek		(void *self, off_t pos);

//static int set_position	(int min, int sec);
static int _vcd_read_toc	();
static int _vcd_get_track_number(uint8_t min, uint8_t sec, uint8_t frame);
static int _vcd_msf2lba		(uint8_t min, uint8_t sec, uint8_t frame);
static void _vcd_next_sector	();

static plugin_input_t input = {
	priv:		&priv,
	open:		_vcd_open,
	close:		_vcd_close,
	read:		_vcd_read,
	seek:		_vcd_seek,
	blocksize:	CD_FRAMESIZE_RAW,	
        config:         NULL,
};


static int _vcd_open (void *self, void *device)
{
	if ((priv.fd = open (device, O_RDONLY)) == -1) {
		LOG (LOG_ERROR, "%s device:%s",
				strerror (errno),
				(char*)device);
		return -1;
	}

	if (_vcd_read_toc ())
		return -1;

	if (_vcd_seek (self, 0))
		return -1;

#if defined (__sun__)
	if (ioctl(priv.fd, CDROMSBLKMODE, CDROM_BLK_2352))
		return -1;
#endif

	return 0;
}


static int _vcd_close (void *self)
{
#if defined (__sun__)
	if (ioctl(priv.fd, CDROMSBLKMODE, CDROM_BLK_2352))
		return -1;
#endif

	close (priv.fd);

	return 0;
}


static int _vcd_seek (void *self, off_t pos)
{
	priv.time.minute	= priv.tocent[pos].cdte_addr.msf.minute;
	priv.time.second	= priv.tocent[pos].cdte_addr.msf.second;
	priv.time.frame		= priv.tocent[pos].cdte_addr.msf.frame;
	priv.current_track	= pos;
	priv.counter		= 0;

	return 0;
}


static ssize_t _vcd_read (void *self, void *buf, size_t count)
{
#if defined (__sun__)
	struct cdrom_cdxa cdxa;
#else
	struct cdrom_msf *msf = (struct cdrom_msf *) buf;
#endif
	int track;

	if (count != CD_FRAMESIZE_RAW)
		return -1;

#if defined (__sun__)
	cdxa.cdxa_addr = _vcd_msf2lba (priv.time.minute, 
		priv.time.second, 
		priv.time.frame);
	cdxa.cdxa_length = CD_FRAMESIZE_RAW;
	cdxa.cdxa_data = buf;
	cdxa.cdxa_format = CDROM_XA_SECTOR_DATA;
	if (ioctl(priv.fd, CDROMCDXA, &cdxa)) {
#else
	msf->cdmsf_min0		= priv.time.minute;
	msf->cdmsf_sec0		= priv.time.second;
	msf->cdmsf_frame0	= priv.time.frame;

//CDROMREADMODE2
	if (ioctl (priv.fd, CDROMREADRAW, buf)) {
#endif
		int track = _vcd_get_track_number (priv.time.minute,
			priv.time.second,
			priv.time.frame);
		LOG (LOG_DEBUG, "ioctl %s", strerror (errno));
		if (track == priv.tochdr.cdth_trk1)
			return 0;

// FIXME
		_vcd_seek (NULL, track);
		return _vcd_read (self, buf, count); // try again with next track
	}
	_vcd_next_sector ();

	if (priv.callb.chapter) {
		priv.counter++;
		if (priv.counter % 64 == 0) {
			track=_vcd_get_track_number (priv.time.minute,
				priv.time.second,
				priv.time.frame)-1;
			if (track != priv.current_track) {
				priv.callb.chapter (priv.callb.chapter_data, track);
				priv.current_track=track;
			}
		}
	}

	return count;
}


static int _vcd_msf2lba (uint8_t min, uint8_t sec, uint8_t frame)
{
	return (((min * CD_SECS) + sec) * CD_FRAMES + frame) - CD_MSF_OFFSET;
}


static int _vcd_get_track_number (uint8_t min, uint8_t sec, uint8_t frame)
{
	int i;
	int lba = _vcd_msf2lba (min, sec, frame);

	for (i=priv.tochdr.cdth_trk0; i <= priv.tochdr.cdth_trk1; i++) {
		int lba2 = _vcd_msf2lba (priv.tocent[i-1].cdte_addr.msf.minute,priv.tocent[i-1].cdte_addr.msf.second,priv.tocent[i-1].cdte_addr.msf.frame);
		if (lba2>lba)
			return i-1;
	}

	return -1;
}


static void _vcd_next_sector ()
{
	priv.time.frame++;

	if (priv.time.frame>=CD_FRAMES) {
		priv.time.frame = 0;
		priv.time.second++;

		if (priv.time.second>=CD_SECS) {
			priv.time.second = 0;
			priv.time.minute++;
		}
	}
}

  
static int _vcd_read_toc ()
{
	int i;

// read TOC header
	if ( ioctl(priv.fd, CDROMREADTOCHDR, &priv.tochdr) == -1 ) {
		LOG (LOG_DEBUG, "readvcd: ioctl cdromreadtochdr\n");
		return -1;
	}

// read individual tracks
	for (i=priv.tochdr.cdth_trk0; i<=priv.tochdr.cdth_trk1; i++) {
		priv.tocent[i-1].cdte_track = i;
		priv.tocent[i-1].cdte_format = CDROM_MSF;
		if ( ioctl(priv.fd, CDROMREADTOCENTRY, &priv.tocent[i-1]) == -1 ) {
			LOG (LOG_DEBUG, "readvcd: ioctl cdromreadtocentry\n");
			return -1;
		}
	}

// read the lead-out track
	priv.tocent[priv.tochdr.cdth_trk1].cdte_track = CDROM_LEADOUT;
	priv.tocent[priv.tochdr.cdth_trk1].cdte_format = CDROM_MSF;

	if (ioctl(priv.fd, CDROMREADTOCENTRY, &priv.tocent[priv.tochdr.cdth_trk1]) == -1 ) {
		LOG (LOG_DEBUG, "readvcd: ioctl cdromreadtocentry\n");
		return -1;
	}

	priv.total_tracks = priv.tochdr.cdth_trk1;

	return 0;
}


#if 0
static int set_position (int min,int sec)
{
	if (min > priv.tocent[priv.tochdr.cdth_trk1].cdte_addr.msf.minute)
		return -1;

	if (min < 0 || sec < 0)
		return -1;

	priv.time.minute=min;
	priv.time.second=sec;
	priv.time.frame=0;

	return 0;
}


int VCDSetTime(int time)
{
	return set_position(0,time/60,time%60);
}
  
/* TODO: fix track_end */
int VCDSetTrack(int track, int track_end)
{
	return set_position(track+1,-1,-1);
}

int VCDGetTime()
{
	return m*60+s;
}
      
int VCDGetTotalTime()
{
	return  priv.tocent[priv.tochdr.cdth_trk1].cdte_addr.msf.minute*60+priv.tocent[priv.tochdr.cdth_trk1].cdte_addr.msf.second;
}
      
int VCDGetTrack()
{
	return _vcd_get_track_number(m,s,f)-1;
}
      
int VCDGetTotalTracks()
{
	return priv.tochdr.cdth_trk1-1;
}
#endif


int PLUGIN_INIT (input_vcd) (uint8_t *whoami)
{
	pluginRegister (whoami,
		PLUGIN_ID_INPUT,
		"vcd",
		NULL,
		NULL,
		&input);

	return 0;
}


void PLUGIN_EXIT (input_vcd) (void)
{
}
