
/*
 * TRANSPORT STREAM DECAPSULATION
 *
 * Copyright (C) 1999  Thomas Mirlacher
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * The author may be reached as <dent@linuxvideo.org>
 *
 *------------------------------------------------------------
 *
 */

//#define DEBUG

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <inttypes.h>
#include <sys/stat.h>
#include <netinet/in.h>

#include <stdlib.h>		// alloc
#include <string.h>		// memcpy

#include <dvb/ts.h>
#include <dvb/pes.h>
#include <oms/oms.h>
#include <oms/plugin/decaps.h>

static int _ts_open (void *self, void *name);
static int _ts_close (void *self);
static int _ts_decaps (buf_t * buf);
static int _ts_ctrl (void *self, uint ctrl_id, ...);

static buf_t *buf;
static u_char ts_file_magic[] = { 0x47 };

static plugin_decaps_magic_t _ts_magic = {
	"ts",
	1,
	ts_file_magic
};

#define PCR	0x03
#define TS_FILTER_MAX	0x04
#define PES_MAX_BLOCK_LEN 0x10000

static struct ts_priv_struct {
	int filter_pid[TS_FILTER_MAX];
	buf_entry_t ts_buf[_BUF_MAX];
	buf_entry_t *buf_entry;
	buf_t *buf;
	u_char cont_count[TS_FILTER_MAX];
} _ts_priv;

static plugin_decaps_t decaps = {
	priv:&_ts_priv,
	open:_ts_open,
	close:_ts_close,
	decaps:_ts_decaps,
	ctrl:_ts_ctrl,
	magic:&_ts_magic,
	blocksize:2256,		// blocksize (inserted: this is a default)
	// need 11.13 ts cells to contain the same inforamtion
	// as a DVD block (2018/184), so use 12 packets
	// (12 * 188) = 2256
	config:NULL,
};

static int _ts_open (void *self, void *name)
{
	struct ts_priv_struct *priv = decaps.priv;

	priv->filter_pid[BUF_VIDEO] = 0xb0;	//b0;
	priv->filter_pid[BUF_AUDIO] = 0xa1;	//0xa1;

	return 0;
}

static int _ts_close (void *self)
{
	return 0;
}

static inline char *_get_next_block (buf_t * _buf)
{
	buf = _buf;

	LOG (LOG_DEBUG, " ");
	if (!_ts_priv.buf_entry)
		if (!(_ts_priv.buf_entry = oms_buf_alloc (buf, decaps.blocksize)))
			return NULL;

	if (
	    (_ts_priv.buf_entry->mem_len =
	     decaps.read (NULL, _ts_priv.buf_entry->mem,
			  decaps.blocksize)) <= 0)
		return NULL;

	_ts_priv.buf_entry->data = _ts_priv.buf_entry->mem;
	_ts_priv.buf_entry->data_len = _ts_priv.buf_entry->mem_len;

	return _ts_priv.buf_entry->mem;
}

static inline u_char *_get_bytes (uint len)
{
	char *ptr = _ts_priv.buf_entry->data;

	LOG (LOG_DEBUG, " ");
	_ts_priv.buf_entry->data += len;

	return ptr;
}

/**
 * get next TS cell, as well as set the offset after the TS header 
 * just leasve some error checks out, because the main buffer has to 
 * contain more than one cell	
 **/

static inline char *_get_next_cell (u_char * ptr)
{
	char *out_ptr;

	if (!ptr) {
		out_ptr = _ts_priv.buf_entry->data;
		_ts_priv.buf_entry->data += TS_HDR_LEN;
	} else {
		out_ptr = ptr + TS_LEN;
		_ts_priv.buf_entry->data = ptr + TS_LEN + TS_HDR_LEN;
	}

	if (_ts_priv.buf_entry->data >=
	    (_ts_priv.buf_entry->mem +
	     _ts_priv.buf_entry->mem_len)) return NULL;

	return out_ptr;
}

static inline void _set_cont_count (uint id, uint _cont_count)
{
	LOG (LOG_DEBUG, " ");
	_ts_priv.cont_count[id] = _cont_count;
}

static inline int _check_cont_count (uint id, uint _cont_count)
{
	LOG (LOG_DEBUG, " ");
	if (_ts_priv.cont_count[id] != _cont_count) {
		_ts_priv.cont_count[id] = _cont_count;
		return -1;
	}

	_ts_priv.cont_count[id] = (_ts_priv.cont_count[id] + 1) & 0x0F;

	return 0;
}

static inline void _put_data (buf_t * _buf, uint id)
{
	buf_entry_t *buf_entry = malloc (sizeof (buf_entry_t));

	_ts_priv.ts_buf[id].mem_len =
	    (_ts_priv.ts_buf[id].data - _ts_priv.ts_buf[id].mem) +
	    _ts_priv.ts_buf[id].data_len;

	memcpy (buf_entry, &_ts_priv.ts_buf[id], sizeof (buf_entry_t));
	buf_entry->mem = _ts_priv.ts_buf[id].mem;
	if (id == BUF_VIDEO)
		buf_entry->buf_id = BUF_VIDEO;
	if (id == BUF_AUDIO)
		buf_entry->buf_id = BUF_AUDIO;
	decaps.write (buf, buf_entry);

//data will be freed by the decoder
	_ts_priv.ts_buf[id].mem = NULL;
}

/**
 * PUSI ... packet unit start indicator
 **/

static inline int _do_pusi_stuff (buf_t * buf, uint id)
{
	pes_hdr_t *pes;
	pes_flag_t *pes_flag;
	uint pes_len;
	int hdr_len;

	LOG (LOG_DEBUG, " ");
	_ts_priv.ts_buf[id].flags |= BUF_FLAG_PUSI;

//      _set_cont_count (id, ts->continuity_counter);

	pes = (pes_hdr_t *) _get_bytes (PES_HDR_LEN);

	if (pes->start_code_prefix != PES_START_CODE_PREFIX_RAW) {
		LOG (LOG_DEBUG, "invalid PES start code prefix");
		//_check_cont_count (id, ts->continuity_counter);
		return -1;
	}
// simple skip pes flags right now
	pes_flag = (pes_flag_t *) _get_bytes (PES_FLAG_LEN);

	_get_bytes (pes_flag->pes_header_data_length);

/* bleh, everything would work out fine, if the standard would force encoders
 * to set the pes_len (maybe i should write them en email ;), but it states:
 * ISO/IEC 13818-1 p34:
 * PES_packet_length -- A 16 bit field specifying the number of bytes in the
 *	PES packet following the last byte of the field. A value of 0
 *	indicates that the PES packet length is neither specified nor bounded
 *	and is allowed only in PES packets whose payload is a video elementary
 *	stream contained in Transport stream packets.
 * - and we're dealing with VES inside TS here
 */
	pes_len = pes_get_len (pes);

/* we've some data left from the previous packet (see avobe the packet length
 * stuff - this can introduce serious jitter :( - but we've a buffer which
 * should smooth things out again
 */
	if (_ts_priv.ts_buf[id].mem)
		_put_data (buf, id);

	hdr_len = (_ts_priv.buf_entry->data - (u_char *) pes);

	if (pes_len)
		_ts_priv.ts_buf[id].mem_len = pes_len + hdr_len;
	else
		_ts_priv.ts_buf[id].mem_len = PES_MAX_BLOCK_LEN;

	_ts_priv.ts_buf[id].mem = malloc (_ts_priv.ts_buf[id].mem_len);
	_ts_priv.ts_buf[id].data = _ts_priv.ts_buf[id].mem + hdr_len;

// put pes header into mem, for HW decoder cards
	memcpy (_ts_priv.ts_buf[id].mem, (char *) pes, hdr_len);
	_ts_priv.ts_buf[id].data_len = 0;

	return 0;
}

static inline void _do_standard_stuff (buf_t * buf, uint id, u_char * ts)
{
	int data_len = TS_LEN - (_ts_priv.buf_entry->data - ts);
	u_char *ptr =

	    _ts_priv.ts_buf[id].data + _ts_priv.ts_buf[id].data_len;

	LOG (LOG_DEBUG, " ");
	if ((ptr + data_len) >
	    (_ts_priv.ts_buf[id].mem + _ts_priv.ts_buf[id].mem_len)) {
		_put_data (buf, id);
	} else {
// DENT: should alloc new buffer, and put this stuff in
		memcpy (ptr, _ts_priv.buf_entry->data, data_len);
		_ts_priv.ts_buf[id].data_len += data_len;
	}
}

static int _ts_decaps (buf_t * buf)
{
	ts_hdr_t *ts = NULL;
	uint pid;

	LOG (LOG_DEBUG, " ");
	if (!_get_next_block (buf))
		return -1;

	while ((ts = (ts_hdr_t *) _get_next_cell ((u_char *) ts))) {
		if (ts->sync_byte != TS_SYNC_BYTE) {
			LOG (LOG_DEBUG, "no sync found");
// TODO: try to resync here
			continue;
		}

		pid = _ts_get_pid (ts);
//fprintf (stderr, "pid: %x ", pid);

		switch (ts->adaptation_field_control) {
		case 0x00:	// reserved
		case 0x01:	// payload only
			break;

		case 0x02:	// adaptation_field only
			continue;

		case 0x03:{	// adaptation_filed + payload
				uint ts_adapt_len =

				    *((u_char *) _get_bytes (1));

				if (ts_adapt_len) {
					ts_adapt_fld_t *ts_adapt;

					ts_adapt =
					    (ts_adapt_fld_t *)
					    _get_bytes (ts_adapt_len);

					if (ts_adapt->
					    discontinuity_indicator) {
						LOG (LOG_DEBUG,
						     "TS: discontinuity error.");
						continue;
					}

					if (ts_adapt->pcr_flag) {	// PCR
//                                              ts_pcr_t *pcr =  _get_bytes (TS_PCR_LEN);
					}
				}
				break;
			}
		}

		if (pid == _ts_priv.filter_pid[BUF_VIDEO]) {
			if (ts->payload_unit_start_indicator)
				if (_do_pusi_stuff (buf, BUF_VIDEO))
					continue;

			if (_ts_priv.ts_buf[BUF_VIDEO].mem)
				_do_standard_stuff (buf, BUF_VIDEO,
						    (u_char *) ts);
		}
		if (pid == _ts_priv.filter_pid[BUF_AUDIO]) {
			if (ts->payload_unit_start_indicator)
				if (_do_pusi_stuff (buf, BUF_AUDIO))
					continue;

			if (_ts_priv.ts_buf[BUF_AUDIO].mem)
				_do_standard_stuff (buf, BUF_AUDIO,
						    (u_char *) ts);
		}
/*
 else if (pid == filter_pid_pcr) {
// DENT: fill here ...
		}
*/
	}

	return 0;
}

static int _ts_ctrl (void *self, uint ctrl_id, ...)
{
	LOG (LOG_DEBUG, "_ts_set_filter called");
	return 0;
}

int PLUGIN_INIT (decaps_ts) (char *whoami) {
	pluginRegister (whoami,
			PLUGIN_ID_DECAPS, "ts", "pes", NULL, &decaps);

	return 0;
}

void PLUGIN_EXIT (decaps_ts) (void) {
	if (_ts_priv.buf_entry) {
		if (_ts_priv.buf_entry->mem) {
			free (_ts_priv.buf_entry->mem);
		}
		free (_ts_priv.buf_entry);
	}
}
