// @reference:  http://www.jmcgowan.com/avi.html

/*
 * DUMMY DECAPSULATION
 *
 * Copyright (C) 2000  Thomas Mirlacher
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * The author may be reached as <dent@linuxvideo.org>
 *
 *------------------------------------------------------------
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <inttypes.h>
#include <sys/stat.h>
#include <bswap.h>

#include "riff.h"

#define AUDIO 1
#define VIDEO 2

#include <oms/oms.h>
#include <oms/plugin/decaps.h>

static int _riff_open (void *self, void *name);
static int _riff_close (void *self);
static int _riff_decaps (buf_t * buf);
static int _riff_ctrl (void *self, uint ctrl_id, ...);

static plugin_decaps_magic_t _riff_magic = {
	"avi,riff",
	4,
	"RIFF"
};

static plugin_decaps_t decaps = {
	open:_riff_open,
	close:_riff_close,
	decaps:_riff_decaps,
	ctrl:_riff_ctrl,
	magic:&_riff_magic,
	blocksize:0xF000,
	config:NULL,
};

static buf_t *buf;
static uint8_t priv_buf[0xfffff];

static int _riff_open (void *self, void *name)
{
	return 0;
}

static int _riff_close (void *self)
{
	return 0;
}

/**
 *
 **/

static uint8_t *_getbytes (uint8_t * _buf, uint32_t num)
{
	buf_entry_t buf_entry;

	buf_entry.mem = priv_buf;
	buf_entry.data = priv_buf;
	buf_entry.mem_len = num;
	buf_entry.data_len = num;

	num = decaps.read (NULL, buf_entry.mem, num);

	return num ? _buf : NULL;
}

/**
 *
 **/

static int get_write_bytes (buf_t * _buf, avi_hdr_t * avi_hdr)
{
	buf_entry_t *buf_entry;

	if (!(buf_entry = oms_buf_alloc (_buf, avi_hdr->len))) {
		_getbytes (priv_buf, avi_hdr->len);
		return -1;
	}

	buf_entry->mem_len = avi_hdr->len;
	if (
	    (buf_entry->mem_len =
	     decaps.read (NULL, buf_entry->mem, avi_hdr->len)) <= 0)
		return -1;

	buf_entry->flags = BUF_FLAG_NONE;
	buf_entry->buf_id = BUF_AUDIO;
	buf_entry->type = FOURCC_MP3;

	decaps.write (_buf, buf_entry);

// DENT: &nBlockAlign
	if (avi_hdr->len & 0x01)
		_getbytes (priv_buf, 1);

	return 0;
}

/**
 *
 **/

void print_char (uint32_t descr)
{
	unsigned char *ptr = (unsigned char *) &descr;
	unsigned int len = 4;

	while (len--)
		printf ("%c", *(ptr++));
	printf (" ");
}

/**
 *
 **/

void print_hex (unsigned char *ptr, unsigned int len)
{
	while (len--)
		printf ("%02x ", *(ptr++));
	printf (" ");
}

/**
 *
 **/

void print_len (uint32_t len, char *descr)
{
	printf ("%s: 0x%x\n", descr, len);
	printf (" ");
}

/**
 *
 **/

void print_hdr (avi_hdr_t * hdr)
{
	uint32_t *descr;

	print_char (hdr->descr_type);
	printf ("len: 0x%x ", hdr->len);
	descr = (uint32_t *) _getbytes (priv_buf, 4);
	print_char (*descr);
	printf ("\n");
}

/**
 *
 **/

int avi_skip_hdr (avi_hdr_t * avi_hdr)
{
	printf ("skip %d\n", avi_hdr->len);

	return _getbytes (priv_buf, avi_hdr->len) ? 0 : -1;
}

/**
 *
 **/

avi_hdr_t *avi_print_rec (avi_hdr_t * avi_hdr)
{
	uint32_t len = avi_hdr->len;

	print_char (avi_hdr->descr_type);
	print_len (avi_hdr->len, "len");

	//print_hex (_getbytes (priv_buf, len), len);
	//printf ("\n");
	_getbytes (priv_buf, len);

// DENT: &nBlockAlign
	if (len & 0x01)
		_getbytes (priv_buf, 1);

	return 0;
}

/**
 *
 **/

int avi_print_hdr (avi_hdr_t * avi_hdr)
{
	unsigned char *ptr = (unsigned char *) avi_hdr;

	//uint32_t *ptr32;

	print_char (avi_hdr->descr_type);
	print_len (avi_hdr->len, "len");

	ptr = _getbytes (priv_buf, avi_hdr->len);
	print_hex (ptr, avi_hdr->len);
	printf ("\n");

	return 0;
}

/**
 *
 **/

uint32_t get4byte (uint32_t * stream)
{
	return *stream++;
}

/**
 *
 **/

int avi_avih (uint32_t * stream)
{
	avi_avih_t *avih = (avi_avih_t *) _getbytes (priv_buf, 14 * 4);

	printf ("avih: (AVI File Header)\n");
	printf ("\tus_frame: %d\n", avih->us_frame);
	printf ("\tmax_bps: %d\n", avih->max_bps);
	printf ("\tpad_gran: %d\n", avih->pad_gran);
	printf ("\tflags: %d\n", avih->flags);
	printf ("\ttot_frames: %d\n", avih->tot_frames);
	printf ("\tinit_frames: %d\n", avih->init_frames);
	printf ("\tstreams: %d\n", avih->streams);
	printf ("\tsug_bsize: %d\n", avih->sug_bsize);
	printf ("\twidth: %d\n", avih->width);
	printf ("\theight: %d\n", avih->height);
	printf ("\tscale: %u\n", avih->scale);
	printf ("\trate: %d\n", avih->rate);
	printf ("\tstart: %d\n", avih->start);
	printf ("\tlength: %d\n", avih->length);

	return 0;
}

/**
 *
 **/

int avi_strf (avi_hdr_t * stream, uint32_t fourcc_type)
{
	uint32_t *ptr32 = (uint32_t *) stream;

	//uint16_t *ptr16;
	avi_vids_strf_t *vids_strf;
	avi_auds_strf_t *auds_strf;

	switch (bswap_32 (fourcc_type)) {
	case RIFF_VIDS:	// Video Stream
		//vids_strf = (avi_vids_strf_t *) _getbytes (priv_buf, 10*4);
		vids_strf =
		    (avi_vids_strf_t *) _getbytes (priv_buf, stream->len);

		printf ("strf: (Video Stream Format)\n");
		printf ("\tsize: %d\n", vids_strf->size);
		printf ("\twidth: %d\n", vids_strf->width);
		printf ("\theight: %d\n", vids_strf->height);
		printf ("\tplanes: %d\n", vids_strf->planes);
		printf ("\tbit depth: %d\n", vids_strf->bit_depth);
		printf ("\tcompression: ");
		print_char (vids_strf->compression);
		printf ("\timage_size: %d\n", vids_strf->image_size);
		printf ("\txpels_meter: %d\n", vids_strf->xpels_meter);
		printf ("\typels_meter: %d\n", vids_strf->ypels_meter);
		printf ("\tnum colors: %d\n", vids_strf->num_colors);
		printf ("\timp colors: %d\n", vids_strf->imp_colors);
		printf ("\n");

// read color table
		if (vids_strf->bit_depth == 40) {
			// thanks to xanim, we know this can happen if
			// someone fooled aoround with converting from
			// qt to avi
			vids_strf->bit_depth = 256;
			// create QT_Create_Gray_Cmap
		} else if (vids_strf->bit_depth <= 8) {
			int i;
			uint8_t *color = (uint8_t *) ptr32;

			for (i = 0; i < vids_strf->num_colors; i++) {
				printf ("\t\tb: %02x g: %02x r: %02x\n",
					*color++, *color++, *color++);
				color++;
			}
		}
		break;

	case RIFF_AUDS:	// Audio Stream
		//auds_strf = (avi_auds_strf_t *) _getbytes (priv_buf, 4*4);
		auds_strf =
		    (avi_auds_strf_t *) _getbytes (priv_buf, stream->len);

		printf ("strf: (Audio Stream Format)\n");
		printf ("\twFormatTag: %x\n", auds_strf->wFormatTag);
		printf ("\tchannels: %d\n", auds_strf->channels);
		printf ("\trate: %d\n", auds_strf->rate);
		printf ("\tAverage Bytes Per Second: %d\n",
			auds_strf->avg_Bps);
		printf ("\tnBlockAlign: %d\n", auds_strf->align);
		printf ("\tBits Per Sample: %x\n", auds_strf->bp_sample);
		break;
	}

	return 0;
}

/**
 *
 **/

uint32_t avi_strh ()
{
	avi_strh_t *strh = (avi_strh_t *) _getbytes (priv_buf, 14 * 4);

	printf ("strh: (Stream Header)\n");
	printf ("\tfourcc_type: ");
	print_char (strh->fourcc_type);
	printf ("\n\tfourcc_handler: ");
	print_char (strh->fourcc_handler);
	printf ("\n\tflags: %d\n", strh->flags);
	printf ("\tpriority: %d\n", strh->priority);
	printf ("\tinit_frames: %d\n", strh->init_frames);
	printf ("\tscale: %d\n", strh->scale);
	printf ("\trate: %d\n", strh->rate);
	printf ("\tstart: %d\n", strh->start);
	printf ("\tlen: %d\n", strh->len);
	printf ("\tsug_bsize: %d\n", strh->sug_bsize);
	printf ("\tquality: %d\n", strh->quality);
	printf ("\tsamp_size: %d\n", strh->samp_size);

	return strh->fourcc_type;
}

/**
 *
 **/

static int _riff_decaps (buf_t * _buf)
{
	avi_hdr_t *avi_hdr = (avi_hdr_t *) _buf;
	uint32_t fourcc_type = 0;

	buf = _buf;

//      while ((avi_hdr = _getbytes (priv_buf, 4*2))) {
	while (1) {
		avi_hdr = (avi_hdr_t *) _getbytes (priv_buf, 4 * 2);

		switch (bswap_32 (avi_hdr->descr_type)) {
		case mmioFOURCC ('R', 'I', 'F', 'F'):
		case mmioFOURCC ('L', 'I', 'S', 'T'):
			print_hdr (avi_hdr);
			break;
		case mmioFOURCC ('A', 'V', 'I', 'H'):	// AVI header
			avi_avih ((uint32_t *) avi_hdr);
			break;
		case mmioFOURCC ('s', 't', 'r', 'h'):	// Stream Header
			fourcc_type = avi_strh ((uint32_t *) avi_hdr);
			break;
		case mmioFOURCC ('s', 't', 'r', 'f'):	// Stream Format
			avi_strf (avi_hdr, fourcc_type);
			break;
		case mmioFOURCC ('J', 'U', 'N', 'K'):
			avi_skip_hdr (avi_hdr);
			break;
		default:
			switch (bswap_32 (avi_hdr->descr_type) & 0xFFFF) {
			case mmioTWOCC ('d', 'c'):	// Video Stream
			case mmioTWOCC ('d', 'b'):
// DENT: use 2 high bytes for filtering the video stream
//      the bytes are describing the stream in ascii)
				avi_print_rec (avi_hdr);
				break;
			case mmioTWOCC ('w', 'b'):{
					// Audio Stream
					get_write_bytes (buf, avi_hdr);
					return 0;
				}
			default:
				printf ("unknown ...\n");
				//avi_print_rec (avi_hdr);
				avi_print_hdr (avi_hdr);
			}

			break;
		}
	}

	return 0;
}

static int _riff_ctrl (void *self, uint ctrl_id, ...)
{
	LOG (LOG_DEBUG, "no ctrl available");

	return -1;
}

int PLUGIN_INIT (decaps_riff) (char *whoami) {
	pluginRegister (whoami,
			PLUGIN_ID_DECAPS,
			"riff;avi", "mpg2", NULL, &decaps);

	return 0;
}

void PLUGIN_EXIT (decaps_riff) (void) {
}
