//#define DEBUG

/*
 * RENDER MANAGER
 *
 * Copyright (C) 2000  Thomas Mirlacher
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * The author may be reached as <dent@linuxvideo.org>
 *
 *------------------------------------------------------------
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/time.h>

#include <oms/oms.h>
#include <oms/plugin/codec.h>	// CTRL_OUTPUT_INITIALIZED
#include <oms/plugin/output_audio.h>
#include <oms/plugin/output_video.h>

static int _audio_open (void *this, void *name);
static int _audio_close (void *this);
static int _audio_setup (plugin_output_audio_attr_t * attr);
static int _audio_write (const void *buf, size_t num_bytes);
static int _audio_check (void);

static int _video_open (void *this, void *name);
static int _video_close (void *this);
static int _video_setup (plugin_output_video_attr_t * attr);
static int _video_overlay (overlay_buf_t * overlay_buf, int id);
static int _video_draw_frame (frame_t * frame);
static int _video_draw_slice (uint8_t * src[], int slice_num);
static void _video_flip_page (void);
static frame_t *_video_allocate_image_buffer (int width, int height,
					      uint32_t format);
static void _video_free_image_buffer (frame_t * image);

static plugin_output_video_t *render_plugin_video = NULL;
static plugin_output_audio_t *render_plugin_audio = NULL;
static plugin_output_video_attr_t render_plugin_video_attr;
static plugin_output_audio_attr_t render_plugin_audio_attr;

plugin_entry_t render_list_entry[MAX_PLUGIN_ID];

static plugin_output_audio_t _render_audio = {
	open:			_audio_open,
	close:			_audio_close,
	setup:			_audio_setup,
	write:			_audio_write,
	check:			_audio_check
};

static plugin_output_video_t _render_video = {
	open:			_video_open,
	close:			_video_close,
	setup:			_video_setup,
	overlay:		_video_overlay,
	draw_frame:		_video_draw_frame,
	draw_slice:		_video_draw_slice,
	flip_page:		_video_flip_page,
	allocate_image_buffer:	_video_allocate_image_buffer,
	free_image_buffer:	_video_free_image_buffer
};

static int _render_load (uint32_t plugin_id, const char *fourcc_out, char *output_path)
{
	LOG (LOG_INFO, "plugin_id %d %s", plugin_id, fourcc_out);
	if (!pluginLoad (plugin_id, NULL, fourcc_out)) {
	  
	  //LOG (LOG_ERROR, "failed to open plugin_id #%d.", plugin_id);
	  switch(plugin_id){
	  case PLUGIN_ID_OUTPUT_VIDEO:
	    LOG (LOG_ERROR, "failed to open video output plugin");
	    break;
	  case PLUGIN_ID_OUTPUT_AUDIO:
	    LOG (LOG_ERROR, "failed to open audio output plugin");
	    break;
	  default:
	    LOG (LOG_ERROR, "failed to open unknown output plugin");
	  }
//		return -1;
		exit (-1);
	}
	// create new render entry
	memcpy (&render_list_entry[plugin_id],
		plugin_get_active (plugin_id), sizeof (plugin_entry_t));

	if (plugin_id == PLUGIN_ID_OUTPUT_VIDEO) {
		render_list_entry[plugin_id].ops = &_render_video;
		render_plugin_video = plugin_get_active_ops (plugin_id);
		plugin_set_active (plugin_id, &render_list_entry[plugin_id]);
		render_plugin_video->open (render_plugin_video, output_path);
		LOG (LOG_INFO, "opening video output plugin succeed.");
	} else if (plugin_id == PLUGIN_ID_OUTPUT_AUDIO) {
		render_list_entry[plugin_id].ops = &_render_audio;
		render_plugin_audio = plugin_get_active_ops (plugin_id);
		plugin_set_active (plugin_id, &render_list_entry[plugin_id]);
		render_plugin_audio->open (render_plugin_audio, output_path);
		LOG (LOG_INFO, "opening audio output plugin succeed.");
	} else {
		LOG (LOG_ERROR, "unhandled plugin_id %d", plugin_id);
		return -1;
	}

	//LOG (LOG_INFO, "opening plugin # %dsucceed.", plugin_id);

	return 0;
}

int render_video_open (const char *fourcc_out, char *output_path)
{
	plugin_output_video_t *plugin_output_video = 
	    plugin_get_active_ops (PLUGIN_ID_OUTPUT_VIDEO);

	if (render_plugin_video)
		plugin_output_video->close (render_plugin_video);

	_render_load (PLUGIN_ID_OUTPUT_VIDEO, fourcc_out, output_path);

	// start thread here
	return 0;
}

int render_video_open_func (void *func)
{
	plugin_output_video_t *plugin_output_video = 
	    plugin_get_active_ops (PLUGIN_ID_OUTPUT_VIDEO);

	if (render_plugin_video) 
		plugin_output_video->close (render_plugin_video);

//      render_plugin_video->open = &_render_video;
//      render_plugin_video = func;

//      render_plugin_video->open (render_plugin_video, output_path);

	return 0;
}

void render_video_close (void)
{
	// stop thread here
}

int render_audio_open (const char *fourcc_out, char *output_path)
{
	plugin_output_audio_t *plugin_output_audio =
	    plugin_get_active_ops (PLUGIN_ID_OUTPUT_AUDIO);

	if (render_plugin_audio)
		plugin_output_audio->close (render_plugin_audio);

	_render_load (PLUGIN_ID_OUTPUT_AUDIO, fourcc_out, output_path);

	// start thread here

	return 0;
}

void render_audio_close (void)
{
	// stop thread here
}

/****************************************/

static void _video_print_fps (int final)
{
	static struct timeval tv_beg,
	 tv_end,
	 tv_start;
	static uint32_t elapsed;
	static uint32_t total_elapsed;
	static uint32_t last_count = 0;
	static uint32_t frame_counter = 0;
	int fps,
	 tfps,
	 frames;

	gettimeofday (&tv_end, NULL);

	if (frame_counter++ == 0)
		tv_start = tv_beg = tv_end;

	elapsed = (tv_end.tv_sec - tv_beg.tv_sec) * 100 +
	    (tv_end.tv_usec - tv_beg.tv_usec) / 10000;
	total_elapsed = (tv_end.tv_sec - tv_start.tv_sec) * 100 +
	    (tv_end.tv_usec - tv_start.tv_usec) / 10000;

	if (final) {
		if (total_elapsed)
			tfps = frame_counter * 10000 / total_elapsed;
		else
			tfps = 0;

		fprintf (stderr, "\n%d frames decoded in %d.%02d "
			 "seconds (%d.%02d fps)\n", frame_counter,
			 total_elapsed / 100, total_elapsed % 100,
			 tfps / 100, tfps % 100);
	} else {
		if (elapsed < 50)	// only display every 0.50 seconds
			return;

		tv_beg = tv_end;
		frames = frame_counter - last_count;

		fps = frames * 10000 / elapsed;	// 100x
		tfps = frame_counter * 10000 / total_elapsed;	// 100x

		fprintf (stderr, "%d frames in %d.%02d sec (%d.%02d fps), "
			 "%d last %d.%02d sec (%d.%02d fps)\033[K\r",
			 frame_counter, total_elapsed / 100,
			 total_elapsed % 100, tfps / 100, tfps % 100,
			 frames, elapsed / 100, elapsed % 100,
			 fps / 100, fps % 100);

		last_count = frame_counter;
	}
}

static int _video_open (void *this, void *output_path)
{
	LOG (LOG_DEBUG, " ");
	render_plugin_video->open (render_plugin_video, output_path);
	return 0;
}

static int _video_close (void *this)
{
	int ret;
	plugin_codec_t *plugin_codec_video;

	plugin_codec_video = plugin_get_active_ops (PLUGIN_ID_CODEC_VIDEO);
	if (plugin_codec_video) {
		_video_print_fps (1);
		plugin_codec_video->ctrl (plugin_codec_video,
					  CTRL_VIDEO_INITIALIZED, 0);
	}

	ret = render_plugin_video->close (render_plugin_video);
	render_plugin_video = NULL;

	return ret;
}

static int _video_setup (plugin_output_video_attr_t * attr)
{
	LOG (LOG_DEBUG, " ");
	memcpy (&render_plugin_video_attr, attr,
		sizeof (plugin_output_video_attr_t));
	return render_plugin_video->setup (attr);
}

static int _video_overlay (overlay_buf_t * overlay_buf, int id)
{
	if (render_plugin_video->overlay)
		return render_plugin_video->overlay (overlay_buf, id);

	return -1;
}

static int _video_draw_slice (uint8_t * src[], int slice_num)
{
	LOG (LOG_DEBUG, " ");
	return render_plugin_video->draw_slice (src, slice_num);
}

static int _video_draw_frame (frame_t * frame)
{
	LOG (LOG_DEBUG, " ");
	return render_plugin_video->draw_frame (frame);
}

static void _video_flip_page (void)
{
	LOG (LOG_DEBUG, " ");
	_video_print_fps (0);

	render_plugin_video->flip_page ();
}

static frame_t *_video_allocate_image_buffer (int width, int height,
					      uint32_t format)
{
	return render_plugin_video->allocate_image_buffer (height, width,
							   format);
}

static void _video_free_image_buffer (frame_t * image)
{
	render_plugin_video->free_image_buffer (image);
}

/****************************************/

static int _audio_open (void *this, void *output_path)
{
	LOG (LOG_DEBUG, "dummy function");
	return render_plugin_audio->open (render_plugin_audio,
					  output_path);
}

static int _audio_close (void *this)
{
	int ret;
	plugin_codec_t *p;

	LOG (LOG_DEBUG, "dummy function");

	p = plugin_get_active_ops (PLUGIN_ID_CODEC_AUDIO);
	if (p)
		p->ctrl (p, CTRL_AUDIO_INITIALIZED, 0);

	ret = render_plugin_audio->close (render_plugin_audio);
	render_plugin_audio = NULL;

	return ret;
}

static int _audio_setup (plugin_output_audio_attr_t * attr)
{
	memcpy (&render_plugin_audio_attr, attr,
		sizeof (plugin_output_audio_attr_t));
	return render_plugin_audio->setup (attr);
}

static int _audio_write (const void *buf, size_t num_bytes)
{
	return render_plugin_audio->write (buf, num_bytes);
}

static int _audio_check (void)
{
	return render_plugin_audio->check ();
}
