/*
 *
 * Copyright (C) 1999-2000  Thomas Mirlacher, Yoann Vandoorselaere
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Authors:    Thomas 'Dent' Mirlacher <dent@linuxvideo.org>
 *              Yoann Vandoorselaere <yoann@mandrakesoft.com>
 *
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include <oms/buf.h>
#include <oms/log.h>

#define BUF_MINFILL 50

buf_t *oms_buf_init (void)
{
	buf_t *buf;

	if (!(buf = malloc (sizeof (buf_t)))) {
		LOG (LOG_ERROR, "memory exhausted");
		return NULL;
	}

	buf->entries_valid = 0;
	pthread_mutex_init (&buf->freemutex, NULL);
	pthread_mutex_init (&buf->validmutex, NULL);

	INIT_LIST_HEAD (&buf->freelist);
	INIT_LIST_HEAD (&buf->validlist);

	return buf;
}


void oms_buf_exit (buf_t *buf)
{
        buf_entry_t *entry;
        struct list_head *tmp;
        
	oms_buf_flush (buf, BUF_ANY);

        list_for_each(tmp, &buf->freelist) {
                entry = list_entry(tmp, buf_entry_t, list);
                free(entry->mem);
                free(entry);
        }
        
        pthread_mutex_destroy(&buf->freemutex);
        pthread_mutex_destroy(&buf->validmutex);

        free(buf);
}


/**
 * Ever used a toilet? - then you should be familiar with the concept
 * behind this function
 **/

void oms_buf_flush (buf_t * buf, uint16_t buf_id)
{
	buf_entry_t *entry;

        assert(buf);
        
	while ((entry = oms_buf_get (buf, buf_id)))
		oms_buf_free (buf, entry);
}


int oms_buf_needs_refill (buf_t *buf)
{
	return (buf->entries_valid < (BUF_MINFILL/2));
}


int oms_buf_isempty (buf_t * buf)
{
	return list_empty (&buf->validlist);
}


int oms_buf_isfull (buf_t * buf)
{
	return (buf->entries_valid >= BUF_MINFILL);
}


buf_entry_t *oms_buf_alloc (buf_t * buf, size_t size)
{
	buf_entry_t *buf_entry = NULL;
	uint8_t *mem;

	if (!list_empty (&buf->freelist)) {
		pthread_mutex_lock(&buf->freemutex);
		buf_entry = list_entry (buf->freelist.next, buf_entry_t, list);
		list_del (&buf_entry->list);
		pthread_mutex_unlock(&buf->freemutex);
		mem = buf_entry->mem;
	} else {
		buf_entry = malloc (sizeof (buf_entry_t));
		if (!buf_entry) {
			LOG (LOG_ERROR, "memory exhausted.");
			return NULL;
		}
		mem = valloc (size);
		if (!mem) {
			free (buf_entry);
			LOG (LOG_ERROR, "memory exhausted.");
			return NULL;
		}
	}

	memset (buf_entry, 0, sizeof (buf_entry_t));

	buf_entry->data = buf_entry->mem = mem;
	buf_entry->data_len = buf_entry->mem_len = size;
	buf_entry->state = ALLOCATED;

	return buf_entry;
}

int oms_buf_free (buf_t * buf, buf_entry_t * buf_entry)
{
        assert(buf && buf_entry);
        
	switch (buf_entry->state) {
	case FREE:
		break;

	case ALLOCATED:
		pthread_mutex_lock(&buf->freemutex);
		list_add (&buf_entry->list, &buf->freelist);
		pthread_mutex_unlock(&buf->freemutex);
		break;

	case VALID:
		pthread_mutex_lock(&buf->validmutex);
                buf->entries_valid--;
		list_del (&buf_entry->list);
		pthread_mutex_unlock(&buf->validmutex);
		pthread_mutex_lock(&buf->freemutex);
		list_add (&buf_entry->list, &buf->freelist);
		pthread_mutex_unlock(&buf->freemutex);
		break;

	case USED:
		pthread_mutex_lock(&buf->freemutex);
		list_add (&buf_entry->list, &buf->freelist);
		pthread_mutex_unlock(&buf->freemutex);
		break;
	}

	buf_entry->state = FREE;
	return 0;
}

int oms_buf_validate (buf_t * buf, buf_entry_t * buf_entry)
{
        assert(buf && buf_entry);
        
	pthread_mutex_lock(&buf->validmutex);
	list_add_tail (&buf_entry->list, &buf->validlist);
	buf->entries_valid++;
	pthread_mutex_unlock(&buf->validmutex);
	buf_entry->state = VALID;

	return 0;
}

buf_entry_t *oms_buf_get (buf_t *buf, uint buf_id)
{
	buf_entry_t *entry, *ret = NULL;
	struct list_head *tmp;

	pthread_mutex_lock(&buf->validmutex);

        list_for_each(tmp, &buf->validlist) {
            
		entry = list_entry(tmp, buf_entry_t, list);

		if (entry->buf_id & buf_id) {
                        list_del(&entry->list);
			buf->entries_valid--;
			entry->state = USED;
                        ret = entry;
			break;
		}
	}
        
	pthread_mutex_unlock(&buf->validmutex);

	return ret;
}




