/*
 * Copyright (C) 1998-2000  Thomas Mirlacher
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 * The author may be reached as <dent@linuxvideo.org>
 *
 *------------------------------------------------------------
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <inttypes.h>
#include <unistd.h>
#include <sys/mman.h>

#include <bswap.h>
#include "ifo.h"
#include "misc.h"

#define ASSERT(x) if(x) return;
#define ASSERT_VOID ASSERT
#define ASSERT_NULL(x) if(x) return NULL;
#define ASSERT_NEG(x) if(x) return -1;
#define ASSERT_0(x) if(x) return 0;

#define PGCI_SUB_LEN 8

typedef struct {
	uint16_t id	: 16;	// Language
	uint16_t	: 16;	// don't know
	uint32_t start	: 32;	// Start of unit
} pgci_sub_t;

typedef struct {
	uint32_t zero	: 32;	// 0x00000000 ?
	uint8_t  zero1	: 8;	// 0x00 ?
	uint8_t  num	: 8;	// number of SPUs
} spu_hdr_t;

#define PTT_LEN 4

typedef struct {
	uint16_t pgc	: 16;	// Program Chain number
	uint16_t pg	: 16;	// Program number
} ptt_t;

#define SPU_HDR_LEN    6

#define OFFSET_IFO		0x0000
#define OFFSET_VTS		0x0000
#define OFFSET_LEN		0x00C0
#define IFO_OFFSET_TAT		0x00C0
#define OFFSET_VTSI_MAT		0x0080
#define IFO_OFFSET_VIDEO	0x0100
#define IFO_OFFSET_AUDIO	0x0200
#define IFO_OFFSET_SUBPIC	0x0250
#define IFO_OFFSET_FIRST_PGC	0x0400


#define PGCI_COLOR_LEN 4


#define CADDR_HDR_LEN 8

typedef struct {
	uint16_t num	: 16;	// Number of Video Objects
	uint16_t unknown: 16;	// don't know
	uint32_t len	: 32;	// length of table
} caddr_hdr_t;



/*
 * audio_struct [1..8] audio_table_struct
 */

typedef struct {
	uint	foo		: 16;	// ???
	uint	num		: 16;	// number of subchannels
} audio_hdr_t;

#define AUDIO_HDR_LEN 4


ifo_t *ifoOpen (const int fd, const off_t pos)
{
	ifo_t *ifo;

	LOG (LOG_DEBUG, "offset: %llx", pos);

	if (fd < 0) {
		LOG (LOG_ERROR, "invalid file descriptor");
		return NULL;
	}

	if (!(ifo = (ifo_t *) calloc (sizeof (ifo_t), 1))) {
		LOG (LOG_ERROR, "memory squeeze (ifo)");
		return NULL;
	}

	if (!(ifo->tbl[ID_MAT] = (uint8_t *) calloc (DVD_VIDEO_LB_LEN/4, 4))) {
		LOG (LOG_ERROR, "memory squeeze (data)");
		free (ifo);
		return NULL;
	}

	ifo->fd = fd;
	ifo->pos = pos; 

	if (lseek (fd, pos, SEEK_SET) == -1) {
		LOG (LOG_ERROR, "error in lseek (pos: 0x%llx)", pos);
		free (ifo->tbl[ID_MAT]);
		free (ifo);
		return NULL;
	}

	if (read (fd, ifo->tbl[ID_MAT], DVD_VIDEO_LB_LEN) < 0) { 
		LOG (LOG_ERROR, "error reading file");
		free (ifo->tbl[ID_MAT]);
		free (ifo);
		return NULL;
	}

	ifo->num_menu_vobs	= OFF_VTSM_VOBS;
	ifo->vob_start		= OFF_VTSTT_VOBS;

	LOG (LOG_DEBUG, "num of vobs: %x vob_start %x", ifo->num_menu_vobs, ifo->vob_start);

	if (!ifoIsVMG (ifo)) {
		ifoReadTBL (ifo, OFF_VMG_PTT_SRPT,	ID_PTT_SRPT);
		ifoReadTBL (ifo, OFF_VMGM_PGCI_UT,	ID_MENU_PGCI);
		ifoReadTBL (ifo, OFF_VMG_PTL_MAIT,	ID_PTL_MAIT);
		ifoReadTBL (ifo, OFF_VMG_VTS_ATRT,	ID_VTS_ATRT);
		ifoReadTBL (ifo, OFF_VMG_TXTDT_MGI,	ID_TXTDT_MGI);
		ifoReadTBL (ifo, OFF_VMGM_C_ADT,	ID_TITLE_CELL_ADDR);
		ifoReadTBL (ifo, OFF_VMGM_VOBU_ADMAP,	ID_TITLE_VOBU_ADMAP);
	} else if (!ifoIsVTS (ifo)) {
		ifoReadTBL (ifo, OFF_VTS_PTT_SRPT,	ID_PTT_SRPT);
		ifoReadTBL (ifo, OFF_VTS_PGCIT,		ID_TITLE_PGCI);
		ifoReadTBL (ifo, OFF_VTSM_PGCI_UT,	ID_MENU_PGCI);
		ifoReadTBL (ifo, OFF_VTS_TMAPT,		ID_TMAPT);
		ifoReadTBL (ifo, OFF_VTSM_C_ADT,	ID_MENU_CELL_ADDR);
		ifoReadTBL (ifo, OFF_VTSM_VOBU_ADMAP,	ID_MENU_VOBU_ADMAP);
		ifoReadTBL (ifo, OFF_VTS_C_ADT,		ID_TITLE_CELL_ADDR);
		ifoReadTBL (ifo, OFF_VTS_VOBU_ADMAP,	ID_TITLE_VOBU_ADMAP);
	}

	return ifo;
}


int ifoClose (ifo_t *ifo)
{
	int i;

	for (i=0; i <= ID_MAX; i++)
		if (ifo->tbl[i])
			free (ifo->tbl[i]);

	free (ifo);

	return 0;
}

int ifoGetAudio (uint8_t *_hdr, uint8_t **ptr)
{
	audio_hdr_t *hdr = (audio_hdr_t *) _hdr;

	ASSERT_NEG (!_hdr);

	*ptr = _hdr + AUDIO_HDR_LEN;

	return be2me_16 (hdr->num);
}	


const ifo_caddr_t *ifoGetCellAddr (const ifo_t *ifo)
{
	uint8_t *ptr;

	ASSERT_NULL (!ifo);

	ptr = ifo->tbl[ID_TITLE_CELL_ADDR];

	ASSERT_NULL (!ptr);

	ptr += CADDR_HDR_LEN;

	return (const ifo_caddr_t *) ptr;
}


uint16_t ifoGetCellAddrNum (const ifo_t *ifo, const uint8_t title_or_menu)
{
	caddr_hdr_t *hdr;

	switch (title_or_menu) {
		case IFO_TITLE:
			hdr = (caddr_hdr_t *) ifo->tbl[ID_TITLE_CELL_ADDR];
			break;
		case IFO_MENU:
			hdr = (caddr_hdr_t *) ifo->tbl[ID_MENU_CELL_ADDR];
			break;
		default:
			return 0;
	}

	ASSERT_0 (!hdr);

	return be2me_32 (hdr->len)/sizeof (ifo_caddr_t);
}


int ifoIsVTS (const ifo_t *ifo)
{
        if (!strncmp (ifo->tbl[ID_MAT], "DVDVIDEO-VTS", 12)) 
		return 0;

	return -1;
}


int ifoIsVMG (const ifo_t *ifo)
{
        if (!strncmp (ifo->tbl[ID_MAT], "DVDVIDEO-VMG", 12))
		return 0;

	return -1;
}


uint32_t ifoGetVOBStart (const ifo_t *ifo)
{
	return ifo->vob_start;
}


const pgc_t *ifoGetPGCI (const ifo_t *ifo, const uint8_t title_or_menu, const int title)
{
	pgci_sub_t *pgci_sub;
	ifo_hdr_t *hdr = NULL;
	uint8_t *ptr;

	switch (title_or_menu) {
		case IFO_TITLE:
			hdr = (ifo_hdr_t *) ifo->tbl[ID_TITLE_PGCI];
			break;
		case IFO_MENU:
			hdr = (ifo_hdr_t *) ifo->tbl[ID_MENU_PGCI];
			break;
	}

	ASSERT_NULL (!hdr);
	ASSERT_NULL (title > hdr->num);

	ptr = (uint8_t *) hdr;
	ptr += IFO_HDR_LEN;

	pgci_sub = (pgci_sub_t *) ptr + title;

	ptr = (uint8_t *) hdr + be2me_32 (pgci_sub->start);

	return (const pgc_t *) ptr;
}


int ifoGetMiscPGCI (ifo_hdr_t *hdr, int title, uint8_t **ptr)
{
	pgci_sub_t *pgci_sub;

	ASSERT_0 (!hdr);
	ASSERT_0 (title > hdr->num);

	*ptr = (uint8_t *) hdr;
	*ptr += IFO_HDR_LEN;
	pgci_sub = (pgci_sub_t *) *ptr + title;

	*ptr = (uint8_t *) hdr + be2me_32 (pgci_sub->start);

	return 0;
}


uint16_t ifoGetNumberOfTitles (const ifo_t *ifo)
{
	ifo_hdr_t *hdr = (ifo_hdr_t *) ifo->tbl[ID_PTT_SRPT];

	ASSERT_0 (!ifo->tbl[ID_PTT_SRPT]);

	return be2me_16 (hdr->num);
}


uint16_t ifoGetNumberOfParts (const ifo_t *ifo)
{
	ifo_hdr_t *hdr = (ifo_hdr_t *) ifo->tbl[ID_PTT_SRPT];

	ASSERT_0 (!ifo->tbl[ID_PTT_SRPT]);

	return be2me_16 (hdr->num);
}


ifo_ptt_t *ifoGetPTT (const ifo_t *ifo)
{
	uint8_t *ptr;
	ifo_hdr_t *hdr;
	ifo_ptt_t *ifo_ptt;
	ifo_ptt_sub_t *ifo_ptt_sub;
	ptt_t *ptt;
	int i, s;
	uint16_t prev_start = 0;
	uint num;

	ASSERT_NULL (!ifo);

	hdr = (ifo_hdr_t *) ifo->tbl[ID_PTT_SRPT];

	ASSERT_NULL (!hdr);

	ptr = (uint8_t *) hdr;

	if (!(ifo_ptt = (ifo_ptt_t *) malloc (sizeof (ifo_ptt_t)))) {
		LOG (LOG_ERROR, "memory squeeze");
		return NULL;
	}

	ifo_ptt->num = be2me_16 (hdr->num);

	if (!(ifo_ptt_sub = (ifo_ptt_sub_t *) calloc (ifo_ptt->num, sizeof (ifo_ptt_sub_t)))) {
		LOG (LOG_ERROR, "memory squeeze");
		return NULL;
	}

	ifo_ptt->title = ifo_ptt_sub;

	ptr += IFO_HDR_LEN;

	prev_start = get4bytes (ptr);

	ptr += 4;
	for (i=1; i<be2me_16 (hdr->num); i++) {
		ifo_ptt_sub = ifo_ptt->title+i-1;

		num = (get4bytes (ptr) - prev_start)/4;

		ptt = (ptt_t *) ((uint8_t *) ifo->tbl[ID_PTT_SRPT] + prev_start);

		if (!(ifo_ptt_sub->data = (ifo_ptt_data_t *) calloc (num, sizeof (ifo_ptt_data_t)))) {
			LOG (LOG_ERROR, "memory squeeze (num: %d)", num);
			return NULL;
		}

		ifo_ptt_sub->num = num;

		for (s=0; s<num; s++) {
			ifo_ptt_sub->data[s].pg = be2me_16 (ptt->pg);
			ifo_ptt_sub->data[s].pgc = be2me_16 (ptt->pgc);

			ptt++;
		}

		prev_start = get4bytes (ptr);

		ptr += 4;
	}

	ifo_ptt_sub = ifo_ptt->title+i-1;
	num = (be2me_32 (hdr->len) - prev_start + 1)/4;

	ptt = (ptt_t *) ((uint8_t *) ifo->tbl[ID_PTT_SRPT] + prev_start);

//DENT: deallocate elements when failing here
	if (!(ifo_ptt_sub->data = (ifo_ptt_data_t *) calloc (num, sizeof (ifo_ptt_data_t)))) {
		LOG (LOG_ERROR, "memory squeeze");
		return NULL;
	}

	ifo_ptt_sub->num = num;

	for (s=0; s<num; s++) {
		ifo_ptt_sub->data[s].pg = be2me_16 (ptt->pg);
		ifo_ptt_sub->data[s].pgc = be2me_16 (ptt->pgc);

		ptt++;
	}

	return ifo_ptt;
}


int ifoGetSPU (uint8_t *_hdr, uint8_t **ptr)
{
	spu_hdr_t *hdr = (spu_hdr_t *) _hdr;

	ASSERT_NEG (!_hdr);

	*ptr = _hdr + SPU_HDR_LEN;
	
	return hdr->num;
}	


off_t ifoGetTSPoffset (const ifo_t *ifo, const int index)
{
	uint8_t *ptr;
	ifo_tsp_t *tsp;

	ASSERT_0 (!ifo);
	ASSERT_0 (!ifo->tbl[ID_PTT_SRPT]);

	ptr = ((uint8_t *) ifo->tbl[ID_PTT_SRPT]) + IFO_HDR_LEN;
	tsp = ((ifo_tsp_t *) ptr) + index;

	return (off_t) be2me_32 (tsp->offset);
}


uint8_t *ifoGetFirstPGC (const ifo_t *ifo)
{
	return &((ifo->tbl[ID_MAT])[IFO_OFFSET_FIRST_PGC]);
}


const clut_t *ifoGetCLUT (const pgc_t *pgc)
{
	ASSERT_NULL (!pgc);

	return (clut_t *) &(pgc->clut);
}


uint8_t ifoGetProgramMapNum (const pgc_t *pgc)
{
	ASSERT_0 (!pgc);

	return pgc->nr_of_programs;
}


const uint8_t *ifoGetProgramMap (const pgc_t *pgc)
{
	const uint8_t *ptr = (uint8_t *) pgc;

	ASSERT_NULL (!pgc);

	return ptr + be2me_16 (pgc->pgc_program_map_offset);
}


uint8_t ifoGetCellPlayInfoNum (const pgc_t *pgc)
{
	ASSERT_0 (!pgc);

	return pgc->nr_of_cells;
}


const ifo_pgci_caddr_t *ifoGetCellPlayInfo (const pgc_t *pgc)
{
	const uint8_t *ptr = (uint8_t *) pgc;

	ASSERT_NULL (!pgc);

	return (ifo_pgci_caddr_t *) (ptr + be2me_16 (pgc->cell_playback_tbl_offset));
}


uint8_t ifoGetCellPosNum (const pgc_t *pgc)
{
	ASSERT_0 (!pgc);

	return pgc->nr_of_cells;
}


ifo_pgc_cpos_t *ifoGetCellPos (const pgc_t *pgc)
{
	const uint8_t *ptr = (uint8_t *) pgc;

	ASSERT_NULL (!pgc);

	return (ifo_pgc_cpos_t *) (ptr + be2me_16 (pgc->cell_position_tbl_offset));
}
