
/*
 * IFO VIRTUAL MACHINE
 *
 * Copyright (C) 2000  Thomas Mirlacher
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 * The author may be reached as <dent@linuxvideo.org>
 *
 *------------------------------------------------------------
 *
 */


#include <stdio.h>
#include <unistd.h>
#include <ctype.h>

#include "ifo.h"

#include "misc.h"
#include "decode.h"

typedef struct {
#if BYTE_ORDER == BIG_ENDIAN
	uint8_t type		: 3;
	uint8_t direct		: 1;
	uint8_t cmd		: 4;

	uint8_t dir_cmp		: 1;
	uint8_t cmp		: 3;
	uint8_t sub_cmd		: 4;
#else
	uint8_t cmd		: 4;
	uint8_t direct		: 1;
	uint8_t type		: 3;

	uint8_t sub_cmd		: 4;
	uint8_t cmp		: 3;
	uint8_t dir_cmp		: 1;
#endif
	union {
		uint8_t  data_8[6];
		uint16_t data_16[3];
	} val;
} op_t;	

#define OP_VAL_16(i) (be2me_16 (op->val.data_16[i]))
#define OP_VAL_8(i) ((op->val.data_8[i]))

static void _cmd_goto		(op_t *op);
static void _cmd_lnk		(op_t *op);
static void _cmd_setsystem	(op_t *op);
static void _cmd_set		(op_t *op);
static void _cmd_4		(op_t *op);
static void _cmd_5		(op_t *op);
static void _cmd_unknown	(op_t *op);

static void _do_lnk		(op_t *op);
static void _do_jmp		(op_t *op);
static void _do_setsystem	(op_t *op);
static void _do_misc		(op_t *op);
static void _do_set		(op_t *op);

static struct {
	void (*cmd) (op_t *op);
} cmd_switch[] = {
	{_cmd_goto},		// goto
	{_cmd_lnk},		// link, jump
	{_cmd_setsystem},
	{_cmd_set},
	{_cmd_4},
	{_cmd_5},
	{_cmd_5},
	{_cmd_unknown},
	{_cmd_unknown},
	{_cmd_unknown},
	{_cmd_unknown},
	{_cmd_unknown},
	{_cmd_unknown},
	{_cmd_unknown},
	{_cmd_unknown},
	{_cmd_unknown},
	{_cmd_unknown},
	{_cmd_unknown},
	{_cmd_unknown},
	{_cmd_unknown},
};


static char reg_name[][80]={
	"Menu_Language_Code",
	"Audio_Stream_#",
	"SubPicture_Stream_#",
	"Angle_#",
	"VTS_#",
	"VTS_Title_#",
	"PGC_#",
	"PTT_#",
	"Highlighted_Button_#",
	"Nav_Timer",
	"TimedPGC",
	"Karaoke_audio_mixing_mode",
	"Parental_mgmt_country_code",
	"Parental_Level",
	"Player_Video_Cfg",
	"Player_Audio_Cfg",
	"Audio_language_code_setting",
	"Audio_language_extension_code",
	"SPU_language_code_setting",
	"SPU_language_extension_code",
	"?Player_Regional_Code",
	"Reserved_21",
	"Reserved_22",
	"Reserved_23"
};


static void _PrintRegister (uint16_t data, uint8_t direct)
{
	if (direct) {
		if (isalpha (data>>8&0xff))
			printf("'%c%c'", data>>8&0xff, data&0xff);
		else
			printf("0x%02x", data);
	} else {
		if (data&0x80) {
			data &= 0x1f;

			if (data > 0x17)
				printf("s[ILL]");
			else
				printf("s[%s]", reg_name[data]);
		} else {
			data &= 0x1f;

			if (data > 0xf)
				printf("r[ILL]");
			else
				printf("r[0x%02x]", data);
		}
	}
}


static void _hexdump (uint8_t *ptr, uint8_t num)
{
	int i;

	for (i=8-num; i<8; i++)
		printf ("%02x ", ptr[i]);
	printf ("\t");	
}


static char *_decode_calc (char val)
{
	static char calc_op[][10] = {
		"none",
		"=", 
		"<->",	// swap
		"+=",
		"-=",
		"*=",
		"/=",
		"%=",
		"rnd",	// rnd
		"&=",
		"|=",
		"^=",
		"??",	// invalid
		"??",	// invalid
		"??",	// invalid
		"??"	// invalid
};

	return (char *) calc_op[val&0x0f];
}

	
char cmp_op[][10] = {
	"none",
	"&&",
	"==",
	"!=",
	">=",
	">",
	"<",
	"<="
};


static void advanced (uint8_t *opcode)
{
	uint8_t cmd = opcode[0];

	printf(" { ");

	if (opcode[1]>>2)
		printf (" Highlight button %d; ", opcode[1]>>2);

	if (cmd == 0xff)
		printf (" Illegal ");

	if (cmd == 0x00) {
		printf ("ReSuME %d", opcode[7]);
	} else if ((cmd & 0x06) == 0x02) {	// XX01Y
		printf ("Link to %s cell ", (cmd & 0x01) ? "prev" : "next");
	} else {
		printf ("advanced (0x%02x) ", cmd);
	}
	printf(" } ");
}


void _do_setsystem (op_t *op)
{
        switch(op->cmd) {
        case 1: {
		int i;

		for (i=1; i<=3; i++) {
                        if (OP_VAL_8(i)&0x80) {
                		if(op->direct)
                                	printf ("s[%s] = 0x%02x;", reg_name[i], OP_VAL_8(i)&0xf);
				else
                                	printf ("s[%s] = r[0x%02x];", reg_name[i], OP_VAL_8(i)&0xf);
			}
		}	

#if 0
                if(op->direct) {
                        if(OP_VAL_8(1]&0x80)
                                printf ("s[%s] = 0x%02x;", reg_name[1], OP_VAL_8(1]&0xf);
                        if(OP_VAL_8(2)&0x80)
//DENT: lwhat about 0x7f here ???
                                printf ("s[%s] = 0x%02x;", reg_name[2], OP_VAL_8(2)&0x7f);
                        if(OP_VAL_8(3)&0x80)
                                printf ("s[%s] = 0x%02x;", reg_name[3], OP_VAL_8(3)&0xf);
                } else {
                        if(OP_VAL_8(1)&0x80)
                                printf ("s[%s] = r[0x%02x];", reg_name[1], OP_VAL_8(1)&0xf);
                        if(OP_VAL_8(2)&0x80)
                                printf ("s[%s] = r[0x%02x];", reg_name[2], OP_VAL_8(2)&0xf);
                        if(OP_VAL_8(3)&0x80)
                                printf ("s[%s] = r[0x%02x];", reg_name[3], OP_VAL_8(3)&0xf);
                }
#endif
		}
                break;
        case 2:
                if(op->direct)
                        printf ("s[%s] = 0x%02x", reg_name[9], OP_VAL_16(0));
                else
                        printf ("s[%s] = r[0x%02x]", reg_name[9], OP_VAL_8(1)&0x0f);

		printf ("s[%s] = (s[%s]&0x7FFF)|0x%02x", reg_name[10], reg_name[10], OP_VAL_16(1)&0x8000);
                break;
        case 3:
                if(op->direct)
                        printf ("r[0x%02x] = 0x%02x", OP_VAL_8(3)&0x0f, OP_VAL_16(0));
                else
                        printf ("r[r[0x%02x]] = r[0x%02x]", OP_VAL_8(3)&0x0f, OP_VAL_8(1)&0x0f);
                break;
        case 4:
                //actually only bits 00011100 00011100 are set
                if(op->direct)
                        printf ("s[%s] = 0x%02x", reg_name[11], OP_VAL_16(1));
                else
                        printf ("s[%s] = r[0x%02x]", reg_name[11], OP_VAL_8(3)&0x0f );
                break;
        case 6:
                //actually,
                //s[%s]=(r[%s]&0x3FF) | (0x%02x << 0xA);
                //but it is way too ugly
                if(op->direct)
                        printf ("s[%s] = 0x%02x", reg_name[8], OP_VAL_8(2)>>2);
                else
                        printf ("s[%s] = r[0x%02x]", reg_name[8], OP_VAL_8(3)&0x0f);
                break;
        default:
                printf ("unknown");
        }
}


char parental[][10] = {
	"0",
	"G",
	"2",
	"PG",
	"PG-13",
	"5",
	"R",
	"NC-17"
};


void _do_misc (op_t *op)
{
	switch(op->sub_cmd) {
	case 1:
		printf ("goto Line 0x%02x", OP_VAL_16(2));
		break;
	case 2:
		printf ("stop VM");
		break;
	case 3:
		printf ("Set Parental Level To %s and goto Line 0x%02x", parental[OP_VAL_8(4)&0x7], OP_VAL_8(5));
		break;
	default:
		printf("Illegal");
		break;
	}
}


static void _cmd_goto (op_t *op)
{
	uint8_t *opcode = (uint8_t *) op;

	if (!opcode[1]) {
		printf("nop");
	} else {
		if (op->cmp) {
			printf ("if (r[0x%02x] %s ", OP_VAL_8(1)&0x0f, cmp_op[op->cmp]);
			_PrintRegister (OP_VAL_16(1), op->dir_cmp);
			printf (") ");
		}
		_do_misc (op);
	}
}


static void _do_lnk (op_t *op)
{
	uint16_t button=OP_VAL_8(4)>>2;

	printf ("lnk to ");

	switch (op->sub_cmd) {
		case 0x01:
			advanced (&OP_VAL_8(4));
			break;

		case 0x04:
			printf ("PGC 0x%02x", OP_VAL_16(2));
			break; 

		case 0x05:
			printf ("PTT 0x%02x", OP_VAL_16(2));
			break; 

		case 0x06:
			printf ("Program 0x%02x this PGC", OP_VAL_8(5));
			break;

		case 0x07:
			printf ("Cell 0x%02x this PGC", OP_VAL_8(5));
			break;
		default:
			return;
	}

	if (button)
		printf (", Highlight 0x%02x", OP_VAL_8(4)>>2);
}


static void _cmd_lnk (op_t *op)
{
	uint8_t *opcode = (uint8_t *) op;

	if (!opcode[1]) {
		printf("nop");
	} else {
		if (op->direct) {
			if (op->cmp) {
				printf ("if (r[0x%02x] %s ", OP_VAL_8(4)&0x0f, cmp_op[op->cmp]);
				_PrintRegister (OP_VAL_8(5), 0);
				printf (") ");
			}
			_do_jmp (op);
		} else {
			if (op->cmp) {
				printf ("if (r[0x%02x] %s ", OP_VAL_8(1)&0x0f, cmp_op[op->cmp]);
				_PrintRegister (OP_VAL_16(1), op->dir_cmp);
				printf (") ");
			}
			_do_lnk (op);
        	}
	}
}


static void _do_jmp (op_t *op)
{

	printf ("jmp ");

	switch (op->sub_cmd) {
		case 0x01:
			printf ("Exit");
			break;
		case 0x02:
			printf ("VTS 0x%02x", OP_VAL_8(3));
			break;
		case 0x03:
			printf ("This VTS Title 0x%02x", OP_VAL_8(3));
			break;
		case 0x05:
			printf ("This VTS Title 0x%02x Part 0x%04x", OP_VAL_8(3), OP_VAL_8(0)<<8|OP_VAL_8(1));
			break;
		case 0x06:
#if 0
			printf ("in SystemSpace ");
			switch (OP_VAL_8(3)>>4) {
				case 0x00:
					printf ("to play first PGC");
					break;
				case 0x01: {
					printf ("to menu \"%s\"", decode_menuname (OP_VAL_8(3)));
				}
					break;
				case 0x02:
					printf ("to VTS 0x%02x and TTN 0x%02x", OP_VAL_8(1), OP_VAL_8(2));
					break;
				case 0x03:
					printf ("to VMGM PGC number 0x%02x", OP_VAL_8(0)<<8 | OP_VAL_8(1));
					break;
				case 0x08:
					printf ("vts 0x%02x lu 0x%02x menu \"%s\"", OP_VAL_8(2), OP_VAL_8(1), decode_menuname (OP_VAL_8(3)));
					break;
#else
			switch (OP_VAL_8(3)>>6) {
				case 0x00:
					printf("to play first PGC");
					break;				
				case 0x01:
					printf ("to VMG title menu (?)");
					break;
				case 0x02:
					printf ("vts 0x%02x lu 0x%02x menu \"%s\"", OP_VAL_8(2), OP_VAL_8(1), decode_menuname (OP_VAL_8(3)&0xF));
					break;				
				case 0x03:
					printf ("vmg pgc 0x%04x (?)", (OP_VAL_8(0)<<8)|OP_VAL_8(1));
					break;
#endif
			}
			break;
		case 0x08:
#if 0
			switch(OP_VAL_8(3)>>4) {
				case 0x00:
					printf ("system first pgc");
					break;
				case 0x01:
					printf ("system title menu");
					break;
				case 0x02:
					printf ("system menu \"%s\"", decode_menuname (OP_VAL_8(3)));
					break;
				case 0x03:
					printf ("system vmg pgc %02x ????", OP_VAL_8(0)<<8|OP_VAL_8(1));
					break;
				case 0x08:
					printf ("system lu 0x%02x menu \"%s\"", OP_VAL_8(2), decode_menuname (OP_VAL_8(3)));
					break;
				case 0x0c:
					printf ("system vmg pgc 0x%02x", OP_VAL_8(0)<<8|OP_VAL_8(1));
					break;
			}
#else
			// OP_VAL_8(2) is number of cell
			// it is processed BEFORE switch
			// under some conditions, it is ignored
			// I don't understand exactly what it means
			printf(" ( spec cell 0x%02X ) ", OP_VAL_8(2)); 

			switch(OP_VAL_8(3)>>6) {
				case 0:
					printf("to FP PGC");
					break;
				case 1:
					printf("to VMG root menu (?)");
					break;
				case 2:
					printf("to VTS menu \"%s\" (?)", decode_menuname(OP_VAL_8(3)&0xF));
					break; 
				case 3:
					printf("vmg pgc 0x%02x (?)", (OP_VAL_8(0)<<8)|OP_VAL_8(1));
					break;
			}	
#endif
			break;
	}
}


static void _do_set (op_t *op)
{
	_PrintRegister (OP_VAL_16(0), 0);
	printf (" %s ", _decode_calc (op->cmd));
	_PrintRegister (OP_VAL_16(1), op->direct);
}


static void _cmd_set (op_t *op)
{
	uint8_t *opcode = (uint8_t *) op;

	if (!opcode[1]) {
		_do_set (op);
	} else if (op->cmp && !op->sub_cmd) {
		printf ("if (r[0x%02x] %s ", OP_VAL_8(0)&0x0f, cmp_op[op->cmp]);
		_PrintRegister (OP_VAL_16(2), op->dir_cmp);
		printf (") ");
		_do_set (op);
	} else if (!op->cmp && op->sub_cmd) {
		printf ("if (");
		_do_set (op);
		printf (") ");
		_do_lnk (op);
	} else
		printf ("nop");
}


static void _cmd_setsystem (op_t *op)
{
	uint8_t *opcode = (uint8_t *) op;

	if (!opcode[1]) {
		_do_setsystem (op);
	} else if (op->cmp && !op->sub_cmd) {
		printf ("if (r[0x%02x] %s ", OP_VAL_8(4)&0x0f, cmp_op[op->cmp]);
		_PrintRegister (OP_VAL_8(5), 0);
		printf (") ");
		_do_setsystem (op);
	} else if (!op->cmp && op->sub_cmd) {
		printf ("if (");
		_do_setsystem (op);
		printf (") ");
		_do_lnk (op);
	} else
		printf("nop");
}


static void _cmd_unknown (op_t *op)
{
	printf ("unknown");
}


void _cmd_4 (op_t *op)
{
        // math command on r[opcode[1]] and direct?be2me_16(OP_VAL_8(0)):reg[OP_VAL_8(1)] is executed
        // ( unless command is swap; then r[opcode[1]] and r[OP_VAL_8(1)] are swapped )
        //
        //  boolean operation cmp on r[opcode[1]] and dir_cmp?be2me_16(OP_VAL_8(1)[1]):reg[OP_VAL_8(3)] is executed
        //  on true result, buttons(c[6], c[7]) is called
        //  problem is 'what is buttons()'
        //
	uint8_t *opcode = (uint8_t *) op;

	printf("r[0x%X] ", opcode[1]);
	printf (" %s ", _decode_calc (op->cmd));
	if (op->cmd == 2)
		printf("r[0x%X] ", OP_VAL_8(1));
	else
		_PrintRegister (OP_VAL_16(0), op->direct);
	printf("; ");

        printf("if ( r[%d] %s ", opcode[1], cmp_op[op->cmp]);
	_PrintRegister (OP_VAL_8(1), op->dir_cmp);
	printf(" )  then {");
	advanced (&OP_VAL_8(4));
	printf("}");
}


void _cmd_5 (op_t *op)
{
        //
        // opposite to dump_4: boolean, math and buttons
        //
	uint8_t *opcode = (uint8_t *) op;

	printf("if (");
		
	if (!op->direct && op->dir_cmp)
		printf("0x%X", OP_VAL_16(1));
	else {
		_PrintRegister (OP_VAL_8(3), 0);
		if(OP_VAL_8(3)&0x80)
			printf("s[%s]", reg_name[OP_VAL_8(3)&0x1F]);
		else
			printf("r[0x%X]", OP_VAL_8(3)&0x1F);	// 0x1F is either not a mistake,
								// or Microsoft programmer's mistake!!!
	}

	printf(" %s r[0x%X] ", cmp_op[op->cmp], op->direct?OP_VAL_8(2):OP_VAL_8(1));
	printf(" )  then {");
	printf("r[0x%X] ", opcode[1]&0xF);
	printf (" %s ", _decode_calc (op->cmd));

	if (op->cmd == 0x02)	// swap
		printf("r[0x%X] ", OP_VAL_8(0)&0x1F);
	else {
		if(op->direct)
			printf("0x%X", OP_VAL_16(0));
		else {
			if(OP_VAL_8(0)&0x80)
				printf("s[%s]", reg_name[OP_VAL_8(0)&0x1F]);
			else
				printf("r[0x%X]", OP_VAL_8(0)&0x1F);
		}
	}

	printf("; ");
	advanced (&OP_VAL_8(4));
	printf("}");
}


void ifoPrintVMOP (uint8_t *opcode)
{
	op_t *op = (op_t *) opcode;

	_hexdump (opcode, 8);

	cmd_switch [op->type].cmd (op);
	printf (";");
}
