
/*
 * Copyright (C) 1998-2000  Thomas Mirlacher
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 * The author may be reached as <dent@linuxvideo.org>
 *
 *------------------------------------------------------------
 *
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <inttypes.h>

#include <bswap.h>
#include <ifo.h>

#include "misc.h"
#include "decode.h"

#define ASSERT(x) if(x) {printf ("%s in %d\n", #x, __LINE__); return; }
#define ASSERT_VOID ASSERT
#define ASSERT_NULL(x) if(x) {printf ("%s in %d\n", #x, __LINE__); return NULL;}
#define ASSERT_0(x) if(x) {printf ("%s in %d\n", #x, __LINE__); return 0;}

#define OUT_H(arg...)	{printf ("\n"); printf (arg); printf ("\n---\n");}
#define OUT_H1(arg...)	{printf ("\n\t"); printf (arg); printf ("\n\t---\n");}
#define OUT_B(arg...)	{printf (arg);}
#define OUT_B1(arg...)	{printf ("\t"); printf (arg);}
#define OUT_B2(arg...)	{printf ("\t\t"); printf (arg);}
#define OUT_B3(arg...)	{printf ("\t\t\t"); printf (arg);}
#define OUT_B1N(arg...)	{printf ("\n\t"); printf (arg);}
#define OUT_B2N(arg...)	{printf ("\n\t\t"); printf (arg);}
#define OUT_B3N(arg...)	{printf ("\n\t\t\t"); printf (arg);}

#define DEBUG_HEX(ptr,x) {int i; for (i=0; i<(x); i++) printf ("%02x ", *ptr++);printf ("\n");}
#define DEBUG_CHAR(ptr,x) {int i; for (i=0; i<(x); i++) printf ("%c%s", *ptr++, i%30 ?"":"\n");printf ("\n");}

#define CADDR_HDR_LEN 8
#define PGCI_SUB_LEN 8
#define VOBU_ADMAP_HDR_LEN 4

typedef struct {
	uint32_t len		: 32;	// length of table
} vobu_addr_map_hdr_t;


typedef struct {
	uint16_t num		: 16;	// Number of Video Objects
	uint16_t unknown	: 16;	// don't know
	uint32_t len		: 32;	// length of table
} caddr_hdr_t;


#define OFFSET_START_TBL_CMD (12+8*2+32*4+8+16*PGCI_COLOR_LEN)

#define PGCI_COLOR_LEN 4

void ifoPrint_cell_position (uint8_t *ptr, uint8_t num);
typedef struct {
	uint16_t id		: 16;	// Language
	uint16_t		: 16;	// don't know
	uint32_t start		: 32;	// Start of unit
} pgci_sub_t;


#define MLU_SUB_LEN 8

typedef struct {
        uint16_t lang		: 16;   // Language
        uint16_t foo		: 16;   // don't know
        uint32_t start		: 32;   // Start of unit
} mlu_sub_t;

#define LU_SUB_LEN 8

//TODO check if BIG_ENDIAN and LITTLE_ENDIAN are correct

typedef struct {
#if BYTE_ORDER == BIG_ENDIAN
        uint16_t foo1		: 4;	// don't know
	uint8_t menu_id		: 4;	// 0=off, 3=root, 4=spu,
					// 5=audio, 6=angle, 7=ptt
#else
	uint8_t menu_id		: 4;	// 0=off, 3=root, 4=spu,
					// 5=audio, 6=angle, 7=ptt
        uint16_t foo1		: 4;	// don't know
#endif
        uint16_t foo2		: 8;	// don't know
        uint16_t bar		: 16;	// don't know
        uint32_t start		: 32;	// Start of unit
} lu_sub_t;

static void _ifoPrint_mlu (uint8_t *ptr);
static void _ifoPrint_caddr (uint8_t *ptr);
static void _ifoPrint_vobu_addr_map (uint8_t *ptr);


void ifoPrintAudio (uint8_t *ptr, uint8_t num)
{
	int i;

	ASSERT (!ptr);

	OUT_H ("AUDIO");
	OUT_B ("number of streams:\t%d\n", num);

	for (i=0; i<num; i++) {
		ifo_audio_t audio;
		uint8_t *appl_mode;
		uint8_t *caption;

		memcpy (&audio, ptr, sizeof (ifo_audio_t));

#if 1
{		int i;
		uint8_t *ptr = (uint8_t *) &audio;

		for (i=0; i<IFO_AUDIO_LEN; i++)
			OUT_B ("%02x ", *ptr++);
}
#endif
//		if (audio.type) // language included
			OUT_B1 ("lang_code:\t%s\n", ifoDecodeLang (audio.lang_code));
//		else
//			OUT_B1 ("no language specified in stream\n");

		OUT_B1 ("mode:\t\t%s(%dCh)%s coding: %d\n",
			decode_audiomode (audio.coding_mode),
			audio.num_channels+1,
			audio.multichannel_extension ? "+multichannel" : "",
			audio.coding_mode
			);

		OUT_B1 ("sampling:\t%dkHz\n", audio.sample_freq ? 96 : 48);

		if ((appl_mode = decode_audiomodeappl (audio.appl_mode)))
			OUT_B1 ("appl_mode:\t%s\n", appl_mode);
		if ((caption = decode_caption (audio.caption)))
			OUT_B1 ("caption:\t%s\n", caption);

		OUT_B1 ("quant:\t\t");
		switch (audio.quantization) {
			case 0x00:
				printf ("16bit");
				break;
			case 0x01:
				printf ("20bit");
				break;
			case 0x02:
				printf ("24bit");
				break;
			case 0x03:
				printf ("DRC");
				break;
		}
		printf ("\n");

		ptr += IFO_AUDIO_LEN;
	}
}


void ifoPrint_vts_caddr (ifo_t *ifo, uint8_t title_or_menu)
{
	switch (title_or_menu) {
	case IFO_TITLE:
		if (ifo->tbl[ID_TITLE_CELL_ADDR]) {
			OUT_H ("TitleSet Cell Address");
			_ifoPrint_caddr (ifo->tbl[ID_TITLE_CELL_ADDR]);
		}
		break;
	case IFO_MENU:
		if (ifo->tbl[ID_MENU_CELL_ADDR]) {
			OUT_H ("Menu Cell Address");
			_ifoPrint_caddr (ifo->tbl[ID_MENU_CELL_ADDR]);
		}
	}
}


static void _ifoPrint_caddr (uint8_t *ptr)
{
	caddr_hdr_t *caddr_hdr = (caddr_hdr_t *) ptr;
	ifo_caddr_t *caddr;
	int i;

	ASSERT (!ptr);

	OUT_B ("number of VOBs: %d\n", be2me_16 (caddr_hdr->num));	
	OUT_B ("???: 0x%x\n", be2me_32 (caddr_hdr->unknown));	
	OUT_B ("len: 0x%x\n", be2me_32 (caddr_hdr->len));	

	ptr += CADDR_HDR_LEN;
	caddr = (ifo_caddr_t *) ptr;

	for (i=0; i<be2me_32 (caddr_hdr->len)/sizeof (ifo_caddr_t); i++) {
		OUT_B1 ("VOB ID: 0x%x Cell ID: 0x%x Start: 0x%x End: 0x%x\n",
			be2me_16 (caddr->vob_id), caddr->cell_id,
			be2me_32 (caddr->start), be2me_32 (caddr->end));

		caddr ++;
	}

	printf ("\n\n\n");
}	


void ifoPrint_pgc (const pgc_t *pgc)
{
	int num;
	int i;

	ASSERT (!pgc);

	OUT_H1 ("Program Chain");
	OUT_B1 ("foo1: 0x%x\n", pgc->zero_1);
	OUT_B1 ("num_programs: %d  num_cells: %d\n",
		pgc->nr_of_programs, pgc->nr_of_cells);

	OUT_B2 ("chain time: %02x:%02x:%02x:%02x\t%sframes/s\n",
		pgc->playback_time.hour,
		pgc->playback_time.minute,
		pgc->playback_time.second,
		pgc->playback_time.frame_u & 0x3f,
		pgc->playback_time.frame_u & 0x80 ? "30" : "25 non-drop");
	OUT_B2 ("prohibited ops: 0x%x\n", be2me_32 (pgc->prohibited_ops));
	decode_user_op (be2me_32 (pgc->prohibited_ops));

/*****/
	OUT_B2 ("AUDIO\n");
	for (i=0; i<8; i++) {		// 8 subaudio streams
		audio_status_t audio = pgc->audio_status[i];

		if (audio.available) {
			OUT_B3 ("0x%02x->0x%02x unknown:0x%02x\n",
				i,
				audio.link,
				audio.foo);
		}
	}
	
/*****/
	OUT_B2N ("SUBPICTURE\n");
	for (i=0; i<32; i++) {
		subp_status_t subp = pgc->subp_status[i];

		if (subp.available) {
			OUT_B3 ("0x%02x 4:3 0x%02x wide 0x%02x letter 0x%02x pan 0x%02x\n",
				i,
				subp.format4_3,
				subp.wide,
				subp.letter,
				subp.pan);
		}
	}

/*****/
	OUT_B2 ("next pgc nr: 0x%x\n", be2me_16 (pgc->next_pgc_nr));
	OUT_B2 ("prev pgc nr: 0x%x\n", be2me_16 (pgc->prev_pgc_nr));
	OUT_B2 ("goup pgc nr: 0x%x\n", be2me_16 (pgc->goup_pgc_nr));
	OUT_B2 ("still time : 0x%x\n", pgc->still_time);
	OUT_B2 ("program playback mode: 0x%x\n", pgc->pg_playback_mode);

	ifoPrintCLUT ((uint8_t *) ifoGetCLUT (pgc));
	ifoPrint_pgc_cmd ((uint8_t *) pgc);

	if ((num = ifoGetProgramMapNum (pgc)))
		ifoPrintProgramMap ((uint8_t *) ifoGetProgramMap (pgc), num);

	if ((num = ifoGetCellPlayInfoNum (pgc)))
		ifoPrintCellInfo ((uint8_t *) ifoGetCellPlayInfo (pgc), num);

	if ((num = ifoGetCellPosNum (pgc)))
		ifoPrintCellPos ((uint8_t *) ifoGetCellPos (pgc), num);
}	


void ifoPrintCellPos (uint8_t *ptr, uint8_t num)
{
	int i;

	OUT_B1N ("Cell Position (num: %d)\n", num);

	for (i=0; i<num; i++) {
		OUT_B2 ("VOB ID: %02x ", get2bytes (ptr));
		ptr += 2;
		OUT_B ("Cell ID: %02x\n", get2bytes (ptr));
		ptr += 2;
	}
}


static void _pgc_cmd (command_data_t *ptr, int num)
{
	int i;

	for (i=1; i<=num; i++) {
		OUT_B3N ("0x%04x: ", i);
		ifoPrintVMOP ((uint8_t *) ptr);
		ptr++;
	}
}


void ifoPrint_pgc_cmd (uint8_t *pgc_ptr)
{
	pgc_command_tbl_t *_cmd_tbl;
	pgc_command_tbl_t cmd_tbl;
	uint16_t offset;

	offset = get2bytes (pgc_ptr + OFFSET_START_TBL_CMD);

	ASSERT (!offset);

// setup
	_cmd_tbl = (pgc_command_tbl_t *) (pgc_ptr + offset);

	cmd_tbl.nr_of_pre  = be2me_16 (_cmd_tbl->nr_of_pre);
	cmd_tbl.nr_of_post = be2me_16 (_cmd_tbl->nr_of_post);
	cmd_tbl.nr_of_cell = be2me_16 (_cmd_tbl->nr_of_cell);

	cmd_tbl.pre_commands = (command_data_t *) &(_cmd_tbl->pre_commands);
	cmd_tbl.post_commands = cmd_tbl.pre_commands + cmd_tbl.nr_of_pre;
	cmd_tbl.cell_commands = cmd_tbl.post_commands + cmd_tbl.nr_of_post;
// setup_end

	if (cmd_tbl.nr_of_pre) {
		OUT_B2N ("pre commands: (num: %d)", cmd_tbl.nr_of_pre);
		_pgc_cmd (cmd_tbl.pre_commands, cmd_tbl.nr_of_pre);
	}
	if (cmd_tbl.nr_of_post) {
		OUT_B2N ("post commands: (num: %d)", cmd_tbl.nr_of_post);
		_pgc_cmd (cmd_tbl.post_commands, cmd_tbl.nr_of_post);
	}
	if (cmd_tbl.nr_of_cell) {
		OUT_B2N ("cell commands: (num: %d)", cmd_tbl.nr_of_cell);
		_pgc_cmd (cmd_tbl.cell_commands, cmd_tbl.nr_of_cell);
	}
}


void ifoPrintCellInfo (uint8_t *ptr, uint8_t num_cell_play_info)
{
	int i;

	OUT_B1N ("Cell Play Info (num: %d)\n", num_cell_play_info);

	for (i=0; i<num_cell_play_info; i++) {
		ifo_pgci_caddr_t caddr;

		memcpy (&caddr, ptr, sizeof (ifo_pgci_caddr_t));

		OUT_B2 ("cell: %d chain info: 0x%x foo: 0x%x\n", i, caddr.chain_info, caddr.foo);

		OUT_B2 ("still_time: %02xs\t cell_cmd: 0x%x\tlen: %02x:%02x:%02x:%02x %sframes/s\n",
			caddr.still_time,
			caddr.cell_cmd,
			caddr.playback_time.hour,
			caddr.playback_time.minute,
			caddr.playback_time.second,
			caddr.playback_time.frame_u,
			caddr.playback_time.frame_u & 0x80 ? "30" : "25 non-drop");
		OUT_B2 ("1st_vobu_start : 0x%08X  1st_ilvu_end : 0x%08X\n\t\tlast_vobu_start: 0x%08X  last_vobu_end: 0x%08X\n",
			be2me_32 (caddr.vobu_start),
			be2me_32 (caddr.ilvu_end),
			be2me_32 (caddr.vobu_last_start),
			be2me_32 (caddr.vobu_last_end));
		ptr += PGCI_CELL_ADDR_LEN;
	}
}		


void ifoPrintCLUT (uint8_t *_clut)
{
	clut_t *clut = (clut_t *) _clut;
	int num = 16;

	ASSERT (!clut);

	OUT_B2N ("Color LookUp Table:\n");

	while (num--) {
		OUT_B3 ("foo:0x%x  Y:0x%x Cr:0x%x Cb:0x%x\n",
			clut->foo, clut->y, clut->cr, clut->cb);

		clut++;
	}
}


void ifoPrintProgramMap (uint8_t *ptr, uint8_t num)
{
	int i;

	OUT_B2N ("cell links: (num: %d)\n", num);

	for (i=0; i<num; i++) {
		OUT_B3 ("0x%02x -> 0x%02x\n", i, *ptr);
		ptr++;
	}
}


void ifoPrint_pgci (ifo_t *ifo, uint8_t title_or_menu)
{
	ifo_hdr_t *hdr;
	int i;

	switch (title_or_menu) {
	case IFO_TITLE:
 		hdr = (ifo_hdr_t *) ifo->tbl[ID_TITLE_PGCI];

		ASSERT (!hdr);

		OUT_H ("Title Program Chain Information");
		OUT_B ("number of units: %d\n", be2me_16 (hdr->num));	

		for (i=0; i<be2me_16(hdr->num); i++) {
			const pgc_t *pgc;

			if ((pgc = ifoGetPGCI (ifo, IFO_TITLE, i)))
				ifoPrint_pgc (pgc);
		}
		break;
	case IFO_MENU:
		hdr = (ifo_hdr_t *) ifo->tbl[ID_MENU_PGCI];

		ASSERT (!hdr);

		OUT_H ("Menu Program Chain Information");
		OUT_B ("number of units: %d\n", be2me_16 (hdr->num));	

		for (i=0; i<be2me_16(hdr->num); i++) {
			const uint8_t *ptr;

			if ((ptr = (uint8_t *) ifoGetPGCI (ifo, IFO_MENU, i)))
				_ifoPrint_mlu ((uint8_t *) ptr);
		}
		break;
	default:
		return;
	}
}	


static void _ifoPrint_mlu (uint8_t *ptr)
{
	ifo_hdr_t *hdr = (ifo_hdr_t *) ptr;
	uint16_t num;
	int i;

	ASSERT (!hdr);
	OUT_B1N ("Language Unit\n");

	ptr += IFO_HDR_LEN;
	num = be2me_16(hdr->num);

	for (i=1; i<=num; i++) {
		lu_sub_t *lu_sub = (lu_sub_t *) ptr;

		if (lu_sub->foo1) {
			OUT_B2 ("%d: menu_id: %s\tmenu: 0x%x parental: 0x%x\n",
				i,
				decode_menuname (lu_sub->menu_id),
				lu_sub->foo2,
                        	be2me_16 (lu_sub->bar));
		} else {
			OUT_B2 ("%d:\t\t\t\tmenu: 0x%x parental: 0x%x\n",
				i,
				lu_sub->foo2,
                        	be2me_16 (lu_sub->bar));
		}

		ptr += LU_SUB_LEN;
	}

	for (i=0; i<be2me_16(hdr->num); i++) {
		uint8_t *ptr;

		if (ifoGetMiscPGCI (hdr, i, &ptr) >= 0)
			ifoPrint_pgc ((pgc_t *) ptr);
	}
}


void ifoPrint_ptt (ifo_ptt_t *ptt)
{
	int i, s;

	ASSERT (!ptt);

	OUT_H ("PART OF TITLE");
	OUT_B ("number of titles: %d\n", ptt->num);	

	for (i=0; i<ptt->num; i++) {
		ifo_ptt_sub_t *title = ptt->title+i;

		OUT_B1N ("Title: 0x%x\tnumber of chapters: %d\n", i+1, title->num);

		for (s=0; s<title->num; s++) {
			OUT_B2 ("Chapter: 0x%02x\tPGC: 0x%02x\tPG: 0x%02x\n",
				s+1, title->data[s].pgc, title->data[s].pg);
		}
	}
}	


void ifoPrintSPU (ifo_spu_t *spu, uint8_t num)
{
	int i;

	ASSERT (!spu);

	OUT_H ("SUBPICTURE");
	OUT_B ("number of streams:\t%d", num);

	for (i=0; i<num; i++) {
		uint8_t *caption;

		OUT_B1N ("lang_code:\t%s\n",   ifoDecodeLang (spu->lang_code));
		if ((caption = decode_caption (spu->caption)))
			OUT_B1 ("caption:\t%s\n", caption);

		spu++;
	}

	printf ("\n");
}


void ifoPrint_TMAPT (ifo_t *ifo)
{
	uint32_t time;

	uint32_t *ptr = (uint32_t *) ifo->tbl[ID_TMAPT];
	ifo_hdr_t *hdr = (ifo_hdr_t *) ptr;
	uint16_t num;
	int i = 0;

	ASSERT (!ptr);
	num = be2me_16 (hdr->num);

	OUT_H ("Time Map Table");
	OUT_B ("number of maps: %d\n", num);	
	OUT_B ("length of table: %d\n", be2me_32 (hdr->len));	
	OUT_B ("('*' specifies last entry in chapter)\n");

	ptr += IFOQ_HDR_LEN;
	ptr += 2 + num;

// FIXME
	//for (i=0; i<be2me_16(hdr->num); i++) {
	{
		uint32_t row;
		uint32_t tab = 0;
		uint32_t foo = be2me_32 (*ptr++);
		uint32_t len = foo & 0x00fffff;

		OUT_B ("--- %d start of 0x%x\n", i, foo>>24);

		for (row=1; row<=len; row++) {
			time = be2me_32 (*ptr++);

			if (time & 0x80000000) {
				time &= 0x7fffffff;
				OUT_B ("%8.8x*\n", time);
				tab = 0;
			} else {
				OUT_B ("%8.8x", time);
				tab++;
				OUT_B ("%s", (tab % 8 == 0) ? "\n" : " ");
			}
		}
	}
}	


void ifoPrintVideo (uint8_t *ptr)
{
	ifo_video_info_t *video = (ifo_video_info_t *) ptr;

	OUT_H ("VIDEO");
	OUT_B1 ("compression : MPEG-%d\n", video->compression+1);
	OUT_B1 ("TV system   : %s Hz\n", video->system ? "PAL 625/50" : "NTSC 525/60");

	OUT_B1 ("Aspect ratio: ");
	switch (video->ratio) {
		case 0:
			printf ("4:3\n");
			break;
		case 3:
			printf ("16:9\n");
			break;
		default:
			printf ("undefined\n");
	}

{
	char perm_displ[][23] = {
		"pan-scan & letterboxed",
		"pan-scan",
		"letterboxed",
		"not specified"
	};

	OUT_B1 ("Display Mode: %s\n", perm_displ[video->perm_displ]);
}

	OUT_B1 ("Line21-1    : %s\n", video->line21_1 ? "data present in GOP" : "");
	OUT_B1 ("Line21-2    : %s\n", video->line21_2 ? "data present in GOP" : "");

{
	char source_res[][28] = {
		"720x480 NTSC or 720x576 PAL",
		"704x480 NTSC or 704x576 PAL",
		"352x480 NTSC or 352x576 PAL",
		"352x240 NTSC or 352x288 PAL"
	};

	OUT_B1 ("Source Res  : %s\n", source_res[video->source_res]);
}
	OUT_B1 ("Letterboxed : %s\n", video->letterboxed ? "yes" : "no");
	OUT_B1 ("Mode        : %s\n", video->mode ? "Film (625/50 only)" : "Camera");
}


void ifoPrint_VOBU_ADMAP (ifo_t *ifo, uint8_t title_or_menu)
{
	switch (title_or_menu) {
	case IFO_TITLE:
		if (ifo->tbl[ID_TITLE_VOBU_ADMAP]) {
			OUT_H ("Video Title Set VOBU Address Map Table");
			_ifoPrint_vobu_addr_map (ifo->tbl[ID_TITLE_VOBU_ADMAP]);
		}
		break;
	case IFO_MENU:
		if (ifo->tbl[ID_MENU_VOBU_ADMAP]) {
			OUT_H ("Video Title Set Menu VOBU Address Map Table");

			_ifoPrint_vobu_addr_map (ifo->tbl[ID_MENU_VOBU_ADMAP]);
		}
	}

	printf ("\n\n\n");
}


static void _ifoPrint_vobu_addr_map (uint8_t *ptr)
{
	vobu_addr_map_hdr_t *vobu_addr_map_hdr = (vobu_addr_map_hdr_t *) ptr;
	int i;
	uint8_t num;

	num = be2me_32 (vobu_addr_map_hdr->len)  / 4;
	OUT_B ("number of units: %d\n\n", num);	
	OUT_B ("len of units: %d\n\n", be2me_32 (vobu_addr_map_hdr->len));	

	ptr += VOBU_ADMAP_HDR_LEN;

	for (i=0; i<num; i++) {
		OUT_B ("0x%07x ", get4bytes (ptr));

		ptr += 4;
	}
}	


void ifoPrint_PTT_SRPT (ifo_t *ifo)
{
	uint8_t *ptr = ifo->tbl[ID_PTT_SRPT];
	ifo_hdr_t *hdr = (ifo_hdr_t *) ptr;
	ifo_tsp_t *tsp;
	int i;

	ASSERT (!ptr);

	OUT_H ("Title Search Pointer");
	OUT_B ("number of maps: %d\n", be2me_16 (hdr->num));
	OUT_B ("length of table: %d\n", be2me_32 (hdr->len));

	ptr = (uint8_t *) (hdr + 1);
	tsp = (ifo_tsp_t *) ptr;

	for (i=0; i<be2me_16 (hdr->num); i++) {
		OUT_B ("%d:\n", i);

		{
		if (tsp->pgc_seq)
			OUT_B ("Sequential Program Chain\n");
		if (tsp->pgc_rnd)
			OUT_B ("Random Program Chain\n");

		if (tsp->ljci_cmd_force)
			OUT_B ("Link/Jump/Call Instr exist in Cell cmd or Button cmd as forced button\n");
		if (tsp->ljci_cmd_pre_post)
			OUT_B ("Link/Jump/Call Instr exist in Cell cmd or Button cmd as Pre or Postcmd\n");
		if (tsp->ljci_button)
			OUT_B ("Link/Jump/Call Instr exist in Button cmd\n");
		if (tsp->ljci_tspd)
			OUT_B ("Link/Jump/Call Instr exist in Title Search Pointer Domain\n");

		if (tsp->uop0)
			OUT_B ("User Operation 0 permitted\n");
		if (tsp->uop1)
			OUT_B ("User Operation 1 permitted\n");
                }

//		OUT_B1 ("playback_type:    0x%x\n", tsp->playback_type);
		OUT_B1 ("num_angles:       0x%x\n", tsp->num_angles);
		OUT_B1 ("num_ptt:          0x%x\n", be2me_16 (tsp->num_ptt));

		OUT_B1 ("parental_control: 0x%x\n", be2me_16 (tsp->parental_control));
		OUT_B1 ("title_set_num:    0x%x\n", tsp->title_set_num);
		OUT_B1 ("vts_ttn:          0x%x\n", tsp->vts_ttn);

		OUT_B1 ("offset:           0x%x\n", be2me_32 (tsp->offset));

		tsp++;
	}
}


void ifoPrint_TXTDT_MGI (ifo_t *ifo)
{
	uint8_t *ptr = ifo->tbl[ID_TXTDT_MGI];
	ifo_hdr_t *hdr = (ifo_hdr_t *) ptr;
	int i;
	int num;

	ASSERT (!ptr);

	OUT_H ("Text Data Manager information");
	OUT_B ("number of maps: %d\n", be2me_16 (hdr->num));
	OUT_B ("length of table: %d\n", be2me_32 (hdr->len));

	ptr = (uint8_t *) (hdr + 1);

	OUT_B ("Language: %c%c\n", ptr[0], ptr[1]);
	ptr+=2;

	DEBUG_HEX(ptr,7);
	num = *ptr++;
	OUT_B ("number of entries: %d\n", num);
	DEBUG_HEX(ptr,7);

	for (i=0; i<num; i++) {
		OUT_B1 ("%d:\t", i);
		DEBUG_HEX(ptr,8);
#if 0
		{
		uint8_t *seg_start;
		//int seg_len;

		ptr-=3;
		seg_start = (uint8_t *) hdr;
		seg_start += ptr[0]<<16|ptr[1];
		ptr+=3;

		DEBUG_CHAR(seg_start, 100);
		}
#endif
	}

	//DEBUG_CHAR(ptr, 4000);
}


void ifo_print_audiosub (uint8_t *ptr)
{
	ifo_hdr_t *hdr = (ifo_hdr_t *) ptr;
	int i;
	uint32_t *start_list;

	OUT_H ("AUDIOSUB");
	OUT_B ("number of units: %d\n", be2me_16 (hdr->num));
	OUT_B ("length of table: 0x%x\n", be2me_32 (hdr->len));

	DEBUG_HEX (ptr, 500);

	start_list = (uint32_t *) calloc (be2me_16 (hdr->num), sizeof (uint32_t));

	ptr += IFO_HDR_LEN;

	for (i=0; i<be2me_16 (hdr->num); i++) {
		start_list[i] = get4bytes (ptr);
		ptr+=4;
	}

	for (i=0; i<be2me_16 (hdr->num); i++) {
		uint32_t len;
		uint8_t *start_ptr;
		uint8_t *_ptr;
		int num;

		start_ptr = ptr = (uint8_t *) hdr + start_list[i];

		len = get4bytes (ptr);
		OUT_B ("length of table: 0x%x\n", len);
		ptr += 4;

		DEBUG_HEX (ptr, 32*8+4);

// <AUDIO>
		if ((num = ifoGetAudio (ptr, &_ptr)) < 0)
			break;
		ifoPrintAudio (_ptr, num);
// </AUDIO>

		ptr += 4+8*8;
		DEBUG_HEX (ptr, 12);

// <SUBPIC>
		if ((num = ifoGetSPU (ptr, &_ptr)) < 0)
			break;
		ifoPrintSPU ((ifo_spu_t *) _ptr, num);
// </SUBPIC>

		ptr += 4+32*6;

		DEBUG_HEX (ptr, start_ptr + len - ptr);
	}
}

