/*****
*
* This file is part of the OMS program.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by 
* the Free Software Foundation; either version 2, or (at your option)
* any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; see the file COPYING.  If not, write to
* the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*
*****/

#ifndef __IFO_H__
#define __IFO_H__

#include <inttypes.h>

#ifndef DVD_VIDEO_LB_LEN
#define DVD_VIDEO_LB_LEN 2048
#endif

#define IFO_TITLE	0x01
#define IFO_MENU	0x02

#include <unistd.h>
#include <fcntl.h>


//-------------------------------------------------------

typedef struct {
	uint8_t	hour;
	uint8_t	minute;
	uint8_t	second;
	uint8_t	frame_u;		// The two high bits are the frame rate.
}  __attribute__ ((packed)) dvd_time_t;

typedef uint8_t __attribute__ ((packed)) command_data_t[8];
#define COMMAND_DATA_SIZE 8
 
typedef struct { // PGC Command Table
	uint16_t nr_of_pre;
	uint16_t nr_of_post;
	uint16_t nr_of_cell;
	uint16_t tbl_len;
	command_data_t *pre_commands;
	command_data_t *post_commands;
	command_data_t *cell_commands;
} __attribute__ ((packed)) pgc_command_tbl_t;
#define PGC_COMMAND_TBL_SIZE 8

typedef uint8_t __attribute__ ((packed)) pgc_program_map_t;

// ifo_pgci_caddr_t == cell_playback_tbl_t

typedef struct { // Cell Playback Information
	uint8_t	chain_info	: 8; // 0x5e 0xde(2 angles, no overlay), 0x5f 0x9f 0x9f 0xdf(4 angles overlay), 0x2 0xa 0x8(1 angle)
	uint8_t foo;				// parental control ??
	uint8_t still_time;
	uint8_t cell_cmd;

        dvd_time_t playback_time;
        uint32_t vobu_start;  			// 1st vobu start
        uint32_t ilvu_end;
        uint32_t vobu_last_start;
        uint32_t vobu_last_end;
} __attribute__ ((packed)) ifo_pgci_caddr_t;

typedef struct { // Cell Position Information
	uint16_t vob_id		: 16;	// Video Object Identifier
	uint8_t  foo		: 8;	// Unknown
	uint8_t  cell_id	: 8;	// Cell Identifier
} __attribute__ ((packed)) ifo_pgc_cpos_t;

#ifndef CLUT_T
#define CLUT_T

typedef struct { // CLUT == Color LookUp Table
	uint8_t foo		: 8;    // UNKNOWN: 0x00?
	uint8_t y		: 8;
	uint8_t cr		: 8;
	uint8_t cb		: 8;
} __attribute__ ((packed)) clut_t;
#endif

typedef struct { // Audio Status
#if BYTE_ORDER == BIG_ENDIAN
	uint8_t available	: 1;
	uint8_t link		: 7;
#else
	uint8_t link		: 7;
	uint8_t available	: 1;
#endif
	uint8_t foo		: 8;	// UNKNOWN
} __attribute__ ((packed)) audio_status_t;


typedef struct { // Subpicture status
#if BYTE_ORDER == BIG_ENDIAN
	uint8_t available	: 1;
	uint8_t format4_3	: 7;
#else
	uint8_t format4_3	: 7;
	uint8_t available	: 1;
#endif
	uint8_t wide		: 8;
	uint8_t letter		: 8;
	uint8_t pan		: 8;
} __attribute__ ((packed)) subp_status_t;


typedef struct { // Program Chain Information
	uint16_t zero_1;
	uint8_t	 nr_of_programs;
	uint8_t	 nr_of_cells;
	dvd_time_t playback_time;
	uint32_t prohibited_ops;	// New type?
	audio_status_t audio_status[8];
	subp_status_t subp_status[32];
	uint16_t next_pgc_nr;
	uint16_t prev_pgc_nr;
	uint16_t goup_pgc_nr;
	uint8_t	 still_time;
	uint8_t	 pg_playback_mode;
	clut_t   clut[16];
	uint16_t pgc_command_tbl_offset;
	uint16_t pgc_program_map_offset;
	uint16_t cell_playback_tbl_offset;
	uint16_t cell_position_tbl_offset;
	pgc_command_tbl_t	*pgc_command_tbl;
	pgc_program_map_t	*pgc_program_map;
	ifo_pgci_caddr_t	*cell_playback_tbl;
	ifo_pgc_cpos_t		*cell_position_tbl;
} __attribute__ ((packed)) pgc_t;
#define PGC_SIZE 236
 
//-------------------------------------------------------


/**
 * Video Info Table 
 **/

typedef struct {
#if BYTE_ORDER == BIG_ENDIAN
	uint8_t compression      	: 2;
	uint8_t system           	: 2;
	uint8_t ratio            	: 2;
	uint8_t perm_displ       	: 2;

	uint8_t line21_1         	: 1;
	uint8_t line21_2         	: 1;
	uint8_t source_res       	: 2;
	uint8_t letterboxed      	: 1;
	uint8_t mode             	: 1;
#elif BYTE_ORDER == LITTLE_ENDIAN
	uint8_t perm_displ       	: 2;
	uint8_t ratio            	: 2;
	uint8_t system           	: 2;
	uint8_t compression      	: 2;

	uint8_t mode             	: 1;
	uint8_t letterboxed      	: 1;
	uint8_t source_res       	: 2;
	uint8_t line21_2         	: 1;
	uint8_t line21_1         	: 1;
#else
#error unhandled byteorder
#endif
} ifo_video_info_t;

/**
 * Audio Table
 **/

// DENT: recheck ... but looks almost as it should be ...

typedef struct {
#if BYTE_ORDER == BIG_ENDIAN
	uint8_t coding_mode		: 3;
	uint8_t multichannel_extension	: 1;
	uint8_t type			: 2;    // audio type (language included?)
	uint8_t appl_mode		: 1;    // audio application mode

	uint8_t quantization		: 2;    // quantization
	uint8_t sample_freq		: 2;    // sampling frequency
	uint8_t				: 1;
	uint8_t num_channels		: 3;    // number of channels (n+1)
#else
	uint8_t appl_mode		: 2;
	uint8_t type			: 2;
	uint8_t multichannel_extension	: 1;
	uint8_t coding_mode		: 3;

	uint8_t num_channels		: 3;
	uint8_t				: 1;
	uint8_t sample_freq		: 2;
	uint8_t quantization		: 2;
#endif
	uint16_t lang_code		: 16;   // <char> description
	uint8_t  foo			: 8;    // 0x00000000 ?
	uint8_t  caption		: 8;
	uint8_t  bar			: 8;    // 0x00000000 ?
} ifo_audio_t;

#define IFO_AUDIO_LEN 7

/**
 * Subpicture Table
 **/

typedef struct {
	uint16_t prefix		: 16;	// 0x0100 ?
	uint16_t lang_code		: 16;	// <char> description
	uint8_t foo			: 8;	// dont know
	uint8_t caption		: 8;	// 0x00 ?
} ifo_spu_t;


/**
 * Time Map Table header entry
 **/

typedef struct {
	uint32_t			: 24;	// don't know
	uint8_t  tu			: 8;	// time unit (in seconds)
} ifoq_tmt_hdr_t;

#define IFOQ_TMT_HDR_LEN 1

/**
 * Title Search Pointer table
 **/

typedef struct {
#if BYTE_ORDER == BIG_ENDIAN
	uint8_t  pgc_rnd		: 1;	// Random Program Chain
	uint8_t  pgc_seq		: 1;	// Sequential Progam Chain
// Link/Jump/Call instruction exist in
	uint8_t  ljci_cmd_force	: 1;	// Cell Command or Button Command as forced button
	uint8_t  ljci_cmd_pre_post	: 1;	// as Pre- or Post-Command
	uint8_t  ljci_button		: 1;	// in Button Command
	uint8_t  ljci_tspd		: 1;	// in Title Search Pointer Domain
	uint8_t  uop0			: 1;	// User Operation 0 permitted
	uint8_t  uop1			: 1;	// User Operation 1 permitted
#else
	uint8_t  uop1			: 1;	// User Operation 1 permitted
	uint8_t  uop0			: 1;	// User Operation 0 permitted

// Link/Jump/Call instruction exist in
	uint8_t  ljci_tspd		: 1;	// in Title Search Pointer Domain
	uint8_t  ljci_button		: 1;	// in Button Command
	uint8_t  ljci_cmd_pre_post	: 1;	// as Pre- or Post-Command
	uint8_t  ljci_cmd_force	: 1;	// Cell Command or Button Command as forced button
	uint8_t  pgc_seq		: 1;	// Sequential Progam Chain
	uint8_t  pgc_rnd		: 1;	// Random Program Chain
#endif
	uint8_t  num_angles		: 8;
	uint16_t num_ptt		: 16;
	uint16_t parental_control	: 16;
	uint8_t  title_set_num		: 8;
	uint8_t  vts_ttn		: 8;
	uint32_t offset			: 32;
} ifo_tsp_t;


/**
 * hmm
 **/

typedef struct {
        uint16_t vob_id		: 16;	// Video Object Identifier
        uint8_t  cell_id	: 8;	// Cell Identifier
	uint8_t  foo		: 8;	// Unknown
        uint32_t start		: 32;	// Cell start
        uint32_t end		: 32;	// Cell end
} ifo_caddr_t;


/**
 * Part of Title AND Title set Cell Address
 **/

typedef struct {
	uint16_t pgc			: 16;	// Program Chain (PTT)
	uint16_t pg			: 16;	// Program (PTT)
	uint32_t start			: 32;	// Start of VOBU (VTS? CADDR)
	uint32_t end			: 32;	// End of VOBU (VTS? CADDR)
} ifo_ptt_data_t;

typedef struct {
	uint16_t num			: 16;	// Number of Chapters
	ifo_ptt_data_t *data		    ;	// Data
} ifo_ptt_sub_t;

typedef struct {
	uint16_t num			: 16;	// Number of Titles
	ifo_ptt_sub_t *title		    ;	// Titles
} ifo_ptt_t;


#define PGCI_CELL_ADDR_LEN	24

#define ID_MAT			0
#define ID_PTT_SRPT		1
#define ID_TITLE_PGCI		2
#define ID_MENU_PGCI		3
#define ID_TMAPT		4
#define ID_MENU_CELL_ADDR	5
#define ID_MENU_VOBU_ADMAP	6
#define ID_TITLE_CELL_ADDR	7
#define ID_TITLE_VOBU_ADMAP 	8

#define ID_PTL_MAIT		2
#define ID_VTS_ATRT		4
#define ID_TXTDT_MGI		5

#define ID_MAX 8


/**
 * Information Table - for internal use only
 **/
 
typedef struct {
	uint32_t num_menu_vobs;
	uint32_t vob_start;

	uint8_t *tbl[ID_MAX+1];
	
	int fd;		// file descriptor
	off_t pos;	// offset of ifo file on device 
} ifo_t;


/**
 * Generic header
 **/

#define IFO_HDR_LEN 8
#define IFOQ_HDR_LEN 2

typedef struct {
        uint16_t num		: 16;   // number of entries
        uint16_t		: 16;   // UNKNOWN
        uint32_t len		: 32;   // length of table
} ifo_hdr_t;


/**
 * Prototypes
 **/

ifo_t *ifoOpen				(int fd, off_t pos);
int ifoClose				(ifo_t *ifo);

int ifoIsVTS				(const ifo_t *ifo);
int ifoIsVMG				(const ifo_t *ifo);
uint32_t ifoGetVOBStart			(const ifo_t *ifo);
uint16_t ifoGetNumberOfTitles		(const ifo_t *ifo);
uint16_t ifoGetNumberOfParts		(const ifo_t *ifo);
ifo_ptt_t *ifoGetPTT			(const ifo_t *ifo);

uint8_t *ifoGetFirstPGC			(const ifo_t *ifo);
const pgc_t *ifoGetPGCI			(const ifo_t *ifo, const uint8_t title_or_menu, const int num);
off_t ifoGetTSPoffset			(const ifo_t *ifo, const int index);

const uint8_t *ifoGetProgramMap		(const pgc_t *pgc);
uint8_t ifoGetProgramMapNum		(const pgc_t *pgc);
	
const ifo_caddr_t *ifoGetCellAddr	(const ifo_t *ifo);
uint16_t ifoGetCellAddrNum		(const ifo_t *ifo, const uint8_t title_or_menu);

const ifo_pgci_caddr_t *ifoGetCellPlayInfo(const pgc_t *pgc);
uint8_t ifoGetCellPlayInfoNum		(const pgc_t *pgc);

ifo_pgc_cpos_t *ifoGetCellPos		(const pgc_t *pgc);
uint8_t ifoGetCellPosNum		(const pgc_t *pgc);


const clut_t *ifoGetCLUT		(const pgc_t *pgc);

//-----------------
int ifoGetAudio				(uint8_t *hdr, uint8_t **ptr);
int ifoGetSPU				(uint8_t *hdr, uint8_t **ptr);


char *ifoDecodeLang			(uint16_t descr);
int ifoGetMiscPGCI			(ifo_hdr_t *hdr, int title, uint8_t **ptr);

//#ifdef PARSER
void ifoPrintVideo			(uint8_t *ptr);

void ifoPrintCellPlayInfo		(uint8_t *ptr, uint8_t num);
void ifoPrintCellInfo			(uint8_t *ptr, uint8_t num);
void ifoPrintCellPos			(uint8_t *ptr, uint8_t num);
void ifoPrintCLUT			(uint8_t *clut); 
void ifoPrintProgramMap			(uint8_t *ptr, uint8_t num);

void ifoPrintAudio			(uint8_t *ptr, uint8_t num);
void ifoPrintSPU			(ifo_spu_t *ptr, uint8_t num);

void ifoPrintVMOP			(uint8_t *opcode);

void ifoPrint_ptt			(ifo_ptt_t *ptt);
void ifoPrint_vts_caddr			(ifo_t *ifo, uint8_t title_or_menu);
void ifoPrint_pgci			(ifo_t *ifo, uint8_t title_or_menu);
void ifoPrint_pgc_cmd			(uint8_t *pgc_ptr);
void ifoPrint_pgc			(const pgc_t *pgc_t);

void ifoPrint_VOBU_ADMAP		(ifo_t *ifo, uint8_t title_or_menu);
void ifoPrint_TMAPT			(ifo_t *ifo);
void ifoPrint_PTT_SRPT			(ifo_t *ifo);
void ifoPrint_TXTDT_MGI			(ifo_t *ifo);
//#endif

#include "misc.h"

#define OFF_VMGM_VOBS           get4bytes (ifo->tbl[ID_MAT] + 192)
#define OFF_VMG_PTT_SRPT        get4bytes (ifo->tbl[ID_MAT] + 196)
#define OFF_VMGM_PGCI_UT        get4bytes (ifo->tbl[ID_MAT] + 200)
#define OFF_VMG_PTL_MAIT        get4bytes (ifo->tbl[ID_MAT] + 204)
#define OFF_VMG_VTS_ATRT        get4bytes (ifo->tbl[ID_MAT] + 208)
#define OFF_VMG_TXTDT_MGI       get4bytes (ifo->tbl[ID_MAT] + 212)
#define OFF_VMGM_C_ADT          get4bytes (ifo->tbl[ID_MAT] + 216)
#define OFF_VMGM_VOBU_ADMAP     get4bytes (ifo->tbl[ID_MAT] + 220)

#define OFF_VTSM_VOBS           get4bytes (ifo->tbl[ID_MAT] + 192)
#define OFF_VTSTT_VOBS          get4bytes (ifo->tbl[ID_MAT] + 196)
#define OFF_VTS_PTT_SRPT        get4bytes (ifo->tbl[ID_MAT] + 200)
#define OFF_VTS_PGCIT           get4bytes (ifo->tbl[ID_MAT] + 204)
#define OFF_VTSM_PGCI_UT        get4bytes (ifo->tbl[ID_MAT] + 208)
#define OFF_VTS_TMAPT           get4bytes (ifo->tbl[ID_MAT] + 212)
#define OFF_VTSM_C_ADT          get4bytes (ifo->tbl[ID_MAT] + 216)
#define OFF_VTSM_VOBU_ADMAP     get4bytes (ifo->tbl[ID_MAT] + 220)
#define OFF_VTS_C_ADT           get4bytes (ifo->tbl[ID_MAT] + 224)
#define OFF_VTS_VOBU_ADMAP      get4bytes (ifo->tbl[ID_MAT] + 228)

#endif

