
/*
 * PROGRAM STREAM
 *
 * Copyright (C) 1998  Thomas Mirlacher
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 * The author may be reached as <dent@linuxvideo.org>
 *
 *------------------------------------------------------------
 *
 */


#ifndef __PS_H__
#define __PS_H__

#include <inttypes.h>

#define PS_SYSTEM_HDR_LEN 12

typedef struct {
	uint8_t header_length_hi		: 8;
	uint8_t header_length_lo		: 8;

#if BYTE_ORDER == BIG_ENDIAN
	uint8_t 				: 1;
	uint8_t rate_bound_1			: 7;
#else
	uint8_t rate_bound_1			: 7;
	uint8_t 				: 1;
#endif

	uint8_t rate_bound_2			: 8;

#if BYTE_ORDER == BIG_ENDIAN
	uint8_t rate_bound_3			: 7;
	uint8_t 				: 1;

	uint8_t audio_bound			: 6;
	uint8_t fixed_flag			: 1;
	uint8_t CSPS_flag			: 1;

	uint8_t system_audio_lock_flag		: 1;
	uint8_t system_video_lock_flag		: 1;
	uint8_t					: 1;
	uint8_t video_bound			: 5;

	uint8_t packet_rate_restriction_flag	: 1;
	uint8_t reserved_byte			: 7;
#else
	uint8_t 				: 1;
	uint8_t rate_bound_3			: 7;

	uint8_t CSPS_flag			: 1;
	uint8_t fixed_flag			: 1;
	uint8_t audio_bound			: 6;

	uint8_t video_bound			: 5;
	uint8_t					: 1;
	uint8_t system_video_lock_flag		: 1;
	uint8_t system_audio_lock_flag		: 1;

	uint8_t reserved_byte			: 7;
	uint8_t packet_rate_restriction_flag	: 1;
#endif
} ps_system_hdr_t;

struct ps_system_hdr_2_struct {
	uint8_t stream_id			: 8;

#if BYTE_ORDER == BIG_ENDIAN
	uint8_t					: 2;	// has to be 11b
	uint8_t PSTD_buffer_bound_scale		: 1;
	uint8_t PSDT_buffer_size_bound_hi	: 5;
#else
	uint8_t PSDT_buffer_size_bound_hi	: 5;
	uint8_t PSTD_buffer_bound_scale		: 1;
	uint8_t					: 2;	// has to be 11b
#endif

	uint8_t PSDT_buffer_size_bound_lo	: 8;
} ps_system_hdr_2_t;

#define PS_STREAM_PACK_HDR	0xBA
#define PS_STREAM_SYSTEM_HDR	0xBB
#define PS_STREAM_PMT		0xBC
#define PS_STREAM_PRIVATE_1	0xBD
#define PS_STREAM_PADDING	0xBE
#define PS_STREAM_PRIVATE_2	0xBF
// C0-DF MPEG audio
// E0-EF MPEG video
#define PS_STREAM_ECM_PRIVATE	0xF0
#define PS_STREAM_EMM_PRIVATE	0xF1
#define PS_STREAM_DSMCC		0xF2
#define PS_STREAM_ITU_T_type_A	0xF4
#define PS_STREAM_ITU_T_type_B	0xF5
#define PS_STREAM_ITU_T_type_C	0xF6
#define PS_STREAM_ITU_T_type_D	0xF7
#define PS_STREAM_ITU_T_type_E	0xF8
#define PS_STREAM_ANCILLARY	0xF9
// FA-FE reserved
#define PS_STREAM_DIRECTORY	0xFF


// Program Stream Elementary Stream Types
#define PS_PMT_EST_RESERVED	0x00
#define PS_PMT_EST_MPEG1_VIDEO	0x01
#define PS_PMT_EST_MPEG2_VIDEO	0x02
#define PS_PMT_EST_MPEG1_AUDIO	0x03
#define PS_PMT_EST_MPEG2_AUDIO	0x04
#define PS_PMT_EST_PRIV_SECT	0x05
#define PS_PMT_EST_PES_PRIV_SECT	0x06
#define PS_PMT_EST_MHEG		0x07
#define PS_PMT_EST_DSMCC	0x08
#define PS_PMT_EST_H2221	0x09
#define PS_PMT_EST_TYPE_A	0x0A
#define PS_PMT_EST_TYPE_B	0x0B
#define PS_PMT_EST_TYPE_C	0x0C
#define PS_PMT_EST_TYPE_D	0x0C
#define PS_PMT_EST_AUXILIARY	0x0E
// 0x0F-0x7F reserved
// 0x80-0xFF user private


#if BYTE_ORDER == BIG_ENDIAN
#define PS_START_CODE_PREFIX_RAW	0x000010
#else
#define PS_START_CODE_PREFIX_RAW	0x010000
#endif

#define PS_START_CODE_PREFIX		PS_START_CODE_PREFIX_RAW

#define PS_HDR_LEN 4

typedef struct {
	uint32_t start_code_prefix		: 24;
	uint8_t stream_id			: 8;
} ps_hdr_t;

typedef struct {
#if BYTE_ORDER == BIG_ENDIAN
	uint8_t					: 2;	// should be 01b
	uint8_t system_clock_reference_1	: 3;
	uint8_t					: 1;
	uint8_t system_clock_reference_2	: 2;
#else
	uint8_t system_clock_reference_2	: 2;
	uint8_t					: 1;
	uint8_t system_clock_reference_1	: 3;
	uint8_t					: 2;	// should be 01b
#endif

	uint8_t system_clock_reference_3	: 8;

#if BYTE_ORDER == BIG_ENDIAN
	uint8_t system_clock_reference_4	: 5;
	uint8_t					: 1;
	uint8_t system_clock_reference_5	: 2;
#else	
	uint8_t system_clock_reference_5	: 2;
	uint8_t					: 1;
	uint8_t system_clock_reference_4	: 5;
#endif

	uint8_t system_clock_reference_6	: 8;

#if BYTE_ORDER == BIG_ENDIAN
	uint8_t system_clock_reference_7	: 5;
	uint8_t					: 1;
	uint8_t system_clock_reference_ext_1	: 2;

	uint8_t system_clock_reference_ext_2	: 7;
	uint8_t					: 1;
#else
	uint8_t system_clock_reference_ext_1	: 2;
	uint8_t					: 1;
	uint8_t system_clock_reference_7	: 5;

	uint8_t					: 1;
	uint8_t system_clock_reference_ext_2	: 7;
#endif

	uint8_t program_mux_rate_1		: 8;
	uint8_t program_mux_rate_2		: 8;

#if BYTE_ORDER == BIG_ENDIAN
	uint8_t program_mux_rate_3		: 6;
	uint8_t					: 2;

	uint8_t					: 5;	// reserved
	uint8_t pack_stuffing_length		: 3;
#else
	uint8_t					: 2;
	uint8_t program_mux_rate_3		: 6;

	uint8_t pack_stuffing_length		: 3;
	uint8_t					: 5;	// reserved
#endif
} ps_pack_t;

#define PS_PACK_LEN 10

/***** access functions *****/

static inline uint16_t _ps_get_hdr_len (ps_system_hdr_t *ps_system)
{
	return ((ps_system->header_length_hi << 8) |
		ps_system->header_length_lo);
}


static inline uint64_t _ps_get_scr (ps_pack_t *ps_pack)
{
	return	((ps_pack->system_clock_reference_1 << 30) |
		(ps_pack->system_clock_reference_2 << 28) |
		(ps_pack->system_clock_reference_3 << 20) |
		(ps_pack->system_clock_reference_4 << 15) |
		(ps_pack->system_clock_reference_5 << 13) |
		(ps_pack->system_clock_reference_6 << 5) |
		(ps_pack->system_clock_reference_7)) * 300 +
		((ps_pack->system_clock_reference_ext_1 << 7) |
		(ps_pack->system_clock_reference_ext_2));
}

#endif
