
/*
 * PACKETIZED ELEMENTARY STREAM
 *
 * Copyright (C) 1998  Thomas Mirlacher
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 * The author may be reached as <dent@linuxvideo.org>
 *
 *------------------------------------------------------------
 *
 */

#ifndef __PES_H__
#define __PES_H__

#include <inttypes.h>
#include <bswap.h>

#if BYTE_ORDER == BIG_ENDIAN
#define PES_START_CODE_PREFIX_RAW	0x000001
#else
#define PES_START_CODE_PREFIX_RAW	0x010000
#endif

#define PES_START_CODE_PREFIX	0x010000

typedef struct {
	uint32_t start_code_prefix		: 24;	// 000001
	uint8_t stream_id;
	uint16_t pes_packet_length;
} pes_hdr_t;    

#define PES_HDR_LEN 6

typedef struct {
#if BYTE_ORDER == BIG_ENDIAN
	uint8_t start_code_prefix		: 2;	// 0x02
	uint8_t pes_scrambling_control  	: 2;
	uint8_t pes_priority	       		: 1;
	uint8_t data_alignment_indicator	: 1;
	uint8_t copyright			: 1;
	uint8_t original_or_copy		: 1;

	uint8_t pts_dts_flag	       		: 2;
	uint8_t escr_flag			: 1;
	uint8_t es_rate_flag	       		: 1;
	uint8_t dsm_trick_mode_flag		: 1;
	uint8_t additional_copy_info_flag	: 1;
	uint8_t pes_crc_flag	       		: 1;
	uint8_t pes_extension_flag		: 1;
#else
	uint8_t original_or_copy		: 1;
	uint8_t copyright			: 1;
	uint8_t data_alignment_indicator	: 1;
	uint8_t pes_priority	       		: 1;
	uint8_t pes_scrambling_control  	: 2;
	uint8_t start_code_prefix		: 2;	// 0x02

	uint8_t pes_extension_flag		: 1;
	uint8_t pes_crc_flag	       		: 1;
	uint8_t additional_copy_info_flag	: 1;
	uint8_t dsm_trick_mode_flag		: 1;
	uint8_t es_rate_flag	       		: 1;
	uint8_t escr_flag			: 1;
	uint8_t pts_dts_flag	       		: 2;
#endif
		
	uint8_t pes_header_data_length		: 8;
} pes_flag_t;

#define PES_FLAG_LEN 3

//TODO: check ENDIANs and convert pts stuff into bytes

struct pts_dts_flags_2_struct {
#if BYTE_ORDER == BIG_ENDIAN
	uint8_t marker_bit0			: 1;  
	uint8_t pts_32_30			: 3;
	uint8_t					: 4;	// has to be 0010b
#else
	uint8_t					: 4;	// has to be 0010b
	uint8_t pts_32_30			: 3;
	uint8_t marker_bit0			: 1;  
#endif

	uint16_t pts_29_15			: 15;
	uint8_t marker_bit1			: 1;  

	uint16_t pts_14_0			: 15;
	uint8_t marker_bit2			: 1;  
};

struct PTS_DTS_flags_3__struct {
	uint8_t					: 4;	// has to be 0011b
	uint8_t pts_32_30			: 3;
	uint8_t marker_bit0			: 1;

	uint16_t pts_29_15			: 15;
	uint8_t marker_bit1			: 1;

	uint16_t pts_14_0			: 15;
	uint8_t marker_bit2			: 1;

	uint8_t					: 2;	// has to be 0001b
	uint8_t dtp_32_30			: 3;
	uint8_t marker_bit3			: 1;
	uint16_t dtp_29_15			: 15;
	uint8_t marker_bit4			: 1;
	uint16_t dtp_14_0			: 15;
	uint8_t marker_bit5			: 1;
};

struct escr_flag_struct {
	uint8_t 				: 2;
	uint8_t escr_base_32_30			: 3;
	uint8_t marker_bit0			: 1;
	uint16_t escr_base_29_15		: 15;
	uint8_t marker_bit1			: 1;
	uint16_t escr_base_14_0			: 15;
	uint8_t marker_bit2			: 1;
	uint16_t escr_extension			: 9;
	uint8_t marker_bit3			: 1;
};

struct es_rate_flag_struct {
	uint8_t marker_bit0			: 1;
	uint es_rate				: 22;
	uint8_t marker_bit1			: 1;
};

struct dsm_trick_mode_flag_struct {
	uint8_t trick_mode_control		: 3;
	union {
		struct {
			uint8_t field_id		: 2;
			uint8_t intra_slice_refresh	: 1;
			uint8_t frequency_truncation	: 2;
		} fast_forward;
		
		struct {
			uint8_t rep_cntrl		: 5;
		} slow_motion;
		
		struct {
			uint8_t field_id		: 2;
			uint8_t 			: 3;
		} freeze_frame;
		
		struct {
			uint8_t field_id            	: 2;
			uint8_t intra_slice_refresh 	: 1;
			uint8_t frequency_truncation	: 2;
		} fast_reverse;
		
		struct {
			uint8_t rep_cntrli		: 5;
		} slow_reverse;
	} mode;
};

struct additional_copy_info_flag_struct {
	uint8_t marker_bit				: 1;
	uint8_t additional_copy_info			: 7;
};

struct pes_crc_flag_struct {
	uint16_t previous_pes_packet_crc		: 16;
};

struct pes_extension_flag_struct {
	uint8_t pes_private_data_flag			: 1;
	uint8_t pack_header_field_flag			: 1;
	uint8_t program_packet_sequence_counter_flag	: 1;
	uint8_t psdt_buffer_flag			: 1;
	uint8_t					: 3;
	uint8_t pes_extension_flag_2			: 1;
/* DENT: tjo ... just continue here to fill structs in ... */
}; 

#define PES_STREAM_PROGRAM_MAP	0xBC
#define PES_STREAM_PRIVATE_1	0xBD
#define PES_STREAM_AUDIO_SPU	0xBD
#define PES_STREAM_PRIVATE_2	0xBF
#define PES_STREAM_PCI_DSI	0xBF
#define PES_STREAM_PADDING	0xBE
#define PES_STREAM_AUDIO	0xBC	// 110X XXXX
#define PES_STREAM_VIDEO	0xE0	// 1110 XXXX
#define PES_STREAM_ECM		0xF0
#define PES_STREAM_EMM		0xF1
#define PES_STREAM_DSMCC	0xF2
#define PES_STREAM_ITU_A	0xF4	
#define PES_STREAM_ITU_B	0xF5	
#define PES_STREAM_ITU_C	0xF6	
#define PES_STREAM_ITU_D	0xF7	
#define PES_STREAM_ITU_E	0xF8	
#define PES_STREAM_MASK_AUDIO	0xE0
#define PES_STREAM_REST_AUDIO	0xC0
#define PES_STREAM_MASK_VIDEO	0xF0
#define PES_STREAM_REST_VIDEO	0xE0

#define PES_SUBSTREAM_AUDIO_LPCM	0xA0
#define PES_SUBSTREAM_AUDIO_AC3		0x80

/***** access functions *****/

static inline uint16_t pes_get_len (pes_hdr_t *pes_hdr)
{
	return be2me_16 (pes_hdr->pes_packet_length);
}


static inline uint64_t _pes_get_pts (pes_flag_t *flag)
{
	uint8_t *ptr = (uint8_t *) (((uint8_t *) (flag)) + PES_FLAG_LEN);
	uint64_t pts;

	pts  = (*ptr++ & 0x0E) << 29; // >> 1 << 30
	pts |= (*ptr++       ) << 22;
	pts |= (*ptr++ & 0xFE) << 14; // >> 1 << 15
	pts |= (*ptr++       ) <<  7;
	pts |= (*ptr++ & 0xFE) >>  1;

	return pts;
}

static inline int _pes_is_startcode (uint32_t data)
{
	return ((data & 0x00FFFFFF) == PES_START_CODE_PREFIX_RAW);
}

#endif
