/*****
*
* This file is part of the OMI program.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by 
* the Free Software Foundation; either version 2, or (at your option)
* any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; see the file COPYING.  If not, write to
* the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*
*****/

/*
 *
 * the playlist
 *
 * Copyright (C) 1999-2000  Thomas Mirlacher, Guenter Bartsch
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *  Changes:
 *    Dominik Schnitzer <dominik@schnitzer.at> - December 21, 2000.
 *    - rewrote playlist load, save and file adding functions
 *    - file adding now checks for the file type and accessability
 *    - Commented the source :)
 *    - major cleanups
 *    Dominik Schnitzer <dominik@schnitzer.at> - December 26, 2000.
 *    - rewrote ~ 80% of the playlist stuff
 *    - cleanups, we free allocated memory, on exit.
 *    - code should be stable now, playlist behaves quite normal now :)
 *    - fixed playlist saving/loading
 *    - playlist format changed
 *    John McCullough <jcmdev0@netscape.net> - December 29, 2000
 *    - Added popup menu
 */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <gtk/gtk.h>

#include <oms/oms.h>

#include "main.h"
#include "panel_gui.h"
#include "playlist.h"
#include "playlist_misc.h"
#include "playlist_op.h"
#include "config_gui.h"

extern playlist_global_t playlist_global;
extern struct player_global_struct player_global;



/**
 * Add an entry to the playlist and device view list
 *
 *    params : *type == URI of file or dvd chapter
 *             *location == title of pice to play
 *             *row_number == playlist position
 *   returns : nothing
 **/

gint playlist_add_entry (gchar *type, gchar *location, gchar *title, gchar *chapter, gint titlenr, gint chapternr, gint row_number)
{
	struct omsitem_info *navitem_info;
	gchar *playlist_text[2];
	
	playlist_text[0] = type;
	playlist_text[1] = chapter;
  
	/* generate a navitem for a file */
	navitem_info = (struct omsitem_info*) malloc (sizeof (struct omsitem_info));
	navitem_info->input = g_strconcat(type, ":", location, NULL);
	navitem_info->title = g_strdup(title);
	navitem_info->chapter = g_strdup(chapter);
	navitem_info->titlenr = titlenr;
	navitem_info->chapternr = chapternr;
	
	/* add the file to the flatlist */
	if (row_number < 0)
		row_number = gtk_clist_append (GTK_CLIST(playlist_global.flatlist), playlist_text);
	else
		gtk_clist_insert (GTK_CLIST(playlist_global.flatlist), row_number, playlist_text);		
		
	gtk_clist_set_row_data_full (GTK_CLIST(playlist_global.flatlist), row_number, navitem_info, GTK_SIGNAL_FUNC(omsitem_free));
	
	/* adjust selected and playing row values */
	if (row_number <= playlist_global.playing_row)
		playlist_global.playing_row++;
	if (row_number <= playlist_global.selected_row)
		playlist_global.selected_row++;
	LOG (LOG_DEBUG, "playlist: playing_row: %d selected_row: %d", playlist_global.playing_row, playlist_global.selected_row);

	if (!playlist_global.first_set) {
		playlist_sync_player(navitem_info);
		playlist_global.first_set = TRUE;
		playlist_global.selected_row = 0;
		playlist_global.playing_row = 0;
		gtk_clist_set_background(GTK_CLIST(playlist_global.flatlist), 0, &playlist_global.color_playing);
		playlist_global.selection_inprogress = TRUE;
		gtk_clist_select_row (GTK_CLIST(playlist_global.flatlist), 0, 0);
		playlist_global.selection_inprogress = FALSE;
	}

	return row_number;
}


/**
 * playlist receives drar and drop events
 *
 *    params : *widget == widget which is responsible for the call
 *             context == 
 *             x, y == X and Y position of drag event
 *             *event == what happened to out entry
 *             user_data == user data passed to the function
 *   returns : nothing
 **/

void playlist_drag_data_received  (GtkWidget *widget, GdkDragContext *context, gint x, gint y, GtkSelectionData *data, guint info, guint time)
{
	int row, col;
	gchar *file_string = data->data;
	GtkAdjustment* row_adjustment = gtk_scrolled_window_get_vadjustment (playlist_global.flat_scroller);

	if ((data->length < 0) || (data->format != 8))
		return;

	gtk_clist_get_selection_info (GTK_CLIST(widget), x, y + row_adjustment->value, &row, &col);

	while (file_string && *file_string) {
		gchar *next_string = NULL;

		if ((next_string = strchr (file_string, '\n'))) {
			if (*(next_string - 1) == '\r')
				*(next_string - 1) = '\0';
			else
				*next_string = '\0';

			next_string++;
		}

		LOG (LOG_DEBUG, "playlist: dropped: %s at %d", file_string, row);
		
		playlist_add_entry ("file", strchr(file_string, ':') + 1, "file", strrchr(file_string, '/') + 1, 0, 0, row++);
		gtk_drag_finish (context, TRUE, FALSE, time);
		file_string = next_string;
	}
  
	gtk_drag_finish (context, FALSE, FALSE, time);
}


/**
 * a playlist item is selected, we gotta play this
 *
 *    params : *widget == widget which is responsible for the call
 *             row, column == row and column of selected playlist item
 *             *event == what happened to out entry
 *             user_data == user data passed to the function
 *   returns : nothing
 **/

void playlist_selection_made (GtkWidget *widget, gint row, gint column, GdkEventButton *event, gpointer user_data)
{
	if (playlist_global.selection_inprogress) 
		return;

	playlist_global.selection_inprogress = TRUE;

	playlist_global.selected_row = row;

	/* we're now playing another item */
	if ((event->type == GDK_2BUTTON_PRESS) || (event->type == GDK_3BUTTON_PRESS) )
		playlist_play_item (row, TRUE);

	playlist_global.selection_inprogress = FALSE;
}


/**
 * move a playlist item up
 *
 *    params : button == button which is responsible for the call
 *             user_data == user data passed to the function
 *   returns : nothing
 **/

void playlist_move_up (GtkButton *button, gpointer user_data) 
{
	if (playlist_global.selected_row > 0) {
		gtk_clist_row_move ((GtkCList *) playlist_global.flatlist,
			playlist_global.selected_row, playlist_global.selected_row - 1);

		/* if the selected row moves up, and just passes playing_row 
		 * we need to increase playing_row by one */
		if ((playlist_global.selected_row - 1) == playlist_global.playing_row)
			playlist_global.playing_row++;
			
		/* if the selected row, which is beeing moved up is our playing_row
		 * we need to decrease the playing row too */
		else if (playlist_global.selected_row == playlist_global.playing_row)
			playlist_global.playing_row--;

		playlist_global.selected_row--;

		LOG (LOG_DEBUG, "playlist: playing_row: %d selected_row: %d", playlist_global.playing_row, playlist_global.selected_row);

		/* check if we need to scroll by */
		if ((gtk_clist_row_is_visible(GTK_CLIST(playlist_global.flatlist), playlist_global.selected_row) == GTK_VISIBILITY_NONE) ||
			(gtk_clist_row_is_visible(GTK_CLIST(playlist_global.flatlist), playlist_global.selected_row) == GTK_VISIBILITY_PARTIAL)) {
			gtk_clist_moveto (GTK_CLIST(playlist_global.flatlist), playlist_global.selected_row, -1, 0.7, 0.7);
		}
	}
}


/**
 * move a playlist entry down
 *
 *    params : button == button which is responsible for the call
 *             user_data == user data passed to the function
 *   returns : nothing
 **/

void playlist_move_down (GtkButton *button, gpointer user_data) 
{
	if (playlist_global.selected_row < (GTK_CLIST(playlist_global.flatlist)->rows - 1)) {
		gtk_clist_row_move (GTK_CLIST(playlist_global.flatlist),
			playlist_global.selected_row, playlist_global.selected_row + 1);

		/* if the selected row moves down, and just passes playing_row 
		 * we need to decrease playing_row by one */
		if ((playlist_global.selected_row + 1) == playlist_global.playing_row)
			playlist_global.playing_row--;
			
		/* if the selected row, which is beeing moved down is our playing_row
		 * we need to increase the playing row too */
		else if (playlist_global.selected_row == playlist_global.playing_row)
			playlist_global.playing_row++;

		playlist_global.selected_row++;

		LOG (LOG_DEBUG, "playlist: playing_row: %d selected_row: %d", playlist_global.playing_row, playlist_global.selected_row);

		/* check if we need to scroll by */
		if ((gtk_clist_row_is_visible(GTK_CLIST(playlist_global.flatlist), playlist_global.selected_row) == GTK_VISIBILITY_NONE) ||
			(gtk_clist_row_is_visible(GTK_CLIST(playlist_global.flatlist), playlist_global.selected_row) == GTK_VISIBILITY_PARTIAL)) {
			gtk_clist_moveto (GTK_CLIST(playlist_global.flatlist), playlist_global.selected_row, -1, 0.7, 0.7);
		}
	}
}


/**
 * remove an entry form the playlist
 *
 *    params : button == button which is responsible for the call
 *             user_data == user data passed to the function
 *   returns : nothing
 **/

void playlist_remove (GtkButton *button, gpointer user_data) 
{
	
	/* check if we actually have rows to select */
	if (playlist_global.selected_row >= 0) {

		LOG (LOG_DEBUG, "playlist: removing row nr: %d", playlist_global.selected_row);
		
		/* if the chapter currently played is removed, play next */
		if ((playlist_global.playing_row == playlist_global.selected_row))
			playlist_play_next();

		gtk_clist_remove (GTK_CLIST(playlist_global.flatlist), playlist_global.selected_row);
		
		/* we need to decrease the currently playing row, if the row which was
		 * deleted was over the playing row. */
		if (playlist_global.selected_row < playlist_global.playing_row)
			playlist_global.playing_row--;

		LOG (LOG_DEBUG, "playlist: playing_row: %d selected_row: %d", playlist_global.playing_row, playlist_global.selected_row);
			
		/* select same row (which is the next item now) if possible */
		if (playlist_global.selected_row < GTK_CLIST(playlist_global.flatlist)->rows) {

			playlist_global.selection_inprogress = TRUE;
			gtk_clist_select_row (GTK_CLIST(playlist_global.flatlist),playlist_global.selected_row, 0);
			playlist_global.selection_inprogress = FALSE;
		}
		
		panel_update_ui();
	}
}


/**
 * add a file to the playlist, confirm if it's really a file
 *
 *    params : w == the gtkwidget the event was toggled
 *             fs == the fileselection dialog
 *   returns : nothing
 **/

void playlist_file_ok_sel (GtkWidget *w, GtkFileSelection *fs)
{
	gchar *filename = gtk_file_selection_get_filename (fs);
	struct stat *file_stat;
	
	file_stat = (struct stat*) malloc( sizeof( struct stat ) );

	/* check if we have a valid file and _no_ directory */	
	if ((stat(filename, file_stat) == 0) && (file_stat->st_mode & S_IFREG)) {

		playlist_add_entry ("file", filename, "file", strrchr(filename, '/') + 1, 0, 0, -1);
		LOG (LOG_DEBUG, "playlist: added file %s", filename);
		gtk_widget_destroy (GTK_WIDGET(fs));
	}
	
	free (file_stat);
}


/**
 * add a file to the playlist
 *
 *    params : button == the gtkbutton the event was toggled
 *             user_data == user data passed to the event
 *   returns : nothing
 **/

void playlist_add_file (GtkButton *button, gpointer user_data) 
{
	GtkWidget *file_selection = gtk_file_selection_new ("Select a file");

	/* Connect the ok_button to file_ok_sel function */
	gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (file_selection)->ok_button),
		"clicked", GTK_SIGNAL_FUNC (playlist_file_ok_sel), 
		file_selection);
         
	/* Connect the cancel_button to destroy the widget */
	gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION
		(file_selection)->cancel_button),
		"clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy),
		GTK_OBJECT (file_selection));
        
	gtk_widget_show (file_selection);
}


/**
 * clear the playlist
 *
 *    params : button == the gtkbutton the event was toggled
 *             user_data == user data passed to the event
 *   returns : nothing
 **/

void playlist_clear (GtkButton *button, gpointer user_data) 
{
	if (oms_get_status () != STATUS_STOP) {
		oms_set_status (STATUS_STOP);
	}
	
	gtk_clist_clear (GTK_CLIST(playlist_global.flatlist));
	
	playlist_sync_player(NULL);
}


/**
 * save the playlist to the home directory
 *
 *    params : none
 *   returns : nothing
 **/

void playlist_save (void)
{
	GString *playlist_file;
	FILE *playlist;
	gint i;
	struct omsitem_info *navitem_info;
	
	LOG(LOG_DEBUG, "playlist: saving ...");

	playlist_file = g_string_new (getenv ("HOME"));
	g_string_append (playlist_file, "/.oms/playlist");
	if (!(playlist = fopen (playlist_file->str, "w"))) {
		LOG (LOG_WARNING, "playlist: couldnt save your playlist in ~/.oms/playlist");
		return;
	}

	for (i = 0; i < GTK_CLIST(playlist_global.flatlist)->rows ; i++) {

		navitem_info = (struct omsitem_info *) gtk_clist_get_row_data ((GtkCList *) playlist_global.flatlist, i);
		if (navitem_info)
			fprintf (playlist, "%s:%s:%s:%d:%d\n", navitem_info->input, navitem_info->title,
				navitem_info->chapter, navitem_info->titlenr, navitem_info->chapternr);
	}
  
	fclose(playlist);
	g_string_free(playlist_file, 1);
}


/**
 * load the playlist from the home directory
 *
 *    params : none
 *   returns : nothing
 **/

void playlist_load (void)
{
	GString *playlist_file;
	FILE *playlist;
	gchar **playlist_text;
	gchar buf[2048];

	LOG (LOG_DEBUG, "playlist: loading...");

	/* clear the playlist before loading it */
	playlist_clear(NULL, NULL);

	playlist_file = g_string_new (getenv ("HOME"));
	g_string_append (playlist_file, "/.oms/playlist");
	if (!(playlist = fopen (playlist_file->str, "r"))) {
		LOG (LOG_INFO, "playlist: couldnt load your playlist: ~/.oms/playlist");
		return;
	}

	while (fgets(buf, sizeof(buf), playlist)) {
		buf[strlen(buf) - 1] = 0;
		playlist_text = g_strsplit(buf, ":", 6);
		
		if (playlist_text[0] && playlist_text[1] && playlist_text[2] && playlist_text[3]
			 && playlist_text[4] && playlist_text[5])
			playlist_add_entry (playlist_text[0], playlist_text[1], playlist_text[2],
				playlist_text[3], strtol(playlist_text[4], NULL, 10), strtol(playlist_text[5], NULL, 10), -1);
				
		if (playlist_text)
			g_strfreev(playlist_text);

	}
        
	fclose(playlist);
	g_string_free(playlist_file, 1);
}


#if 0
static GtkItemFactoryEntry menu_items[] = {
	{ "/Add _File",		"<control>F", playlist_add_file, 0, NULL },
	{ "/_Remove File",	"<control>R", playlist_remove, 0, NULL },
	{ "/_Clear List",	"<control>C", playlist_clear, 0, NULL },
	{ "/sep1",		NULL,         NULL, 0, "<Separator>" },
	{ "/_Save List",	"<control>S", playlist_remove, 0, NULL },
	{ "/_Load List",	"<control>L", playlist_clear, 0, NULL },
	{ "/sep2",		NULL,         NULL, 0, "<Separator>" },
};
#endif

gint playlist_item_press (GtkWidget *hbox, GdkEvent *event)
{
        GtkWidget *PlMenu,*PlMenuItem;
        GList *pDevList, *pDevCur;
        GList *selection, *selection_end;
   	PlMenu = gtk_menu_new();    
#if 0 
	GtkItemFactory *item_factory;
	GtkAccelGroup *accel_group;
	gint nmenu_items = sizeof (menu_items) / sizeof (menu_items[0]);

	accel_group = gtk_accel_group_new ();

	item_factory = gtk_item_factory_new (GTK_TYPE_MENU_BAR, "<main>", accel_group);

  gtk_item_factory_create_items (item_factory, nmenu_items, menu_items, NULL);

  if (menubar)
    /* Finally, return the actual menu bar created by the item factory. */
    *menubar = gtk_item_factory_get_widget (item_factory, "<main>");

#else
	PlMenuItem = gtk_menu_item_new_with_label ("Add File");
	gtk_signal_connect_object (GTK_OBJECT (PlMenuItem), "activate",
		GTK_SIGNAL_FUNC (playlist_add_file),
		GTK_OBJECT (hbox));
	gtk_menu_append ( GTK_MENU (PlMenu), PlMenuItem);
	
	PlMenuItem = gtk_menu_item_new_with_label ("Remove");
	gtk_signal_connect_object (GTK_OBJECT (PlMenuItem), "activate",
		GTK_SIGNAL_FUNC (playlist_remove), 
		GTK_OBJECT (hbox)); 
  	gtk_menu_append ( GTK_MENU (PlMenu), PlMenuItem);

	PlMenuItem = gtk_menu_item_new_with_label ("Clear");
	gtk_signal_connect_object (GTK_OBJECT (PlMenuItem), "activate",
		GTK_SIGNAL_FUNC (playlist_clear), 
		GTK_OBJECT (hbox)); 
	gtk_menu_append ( GTK_MENU (PlMenu), PlMenuItem);
	
	/* Seperator */
	gtk_menu_append(GTK_MENU(PlMenu), gtk_menu_item_new());

	PlMenuItem = gtk_menu_item_new_with_label ("Save list");
	gtk_signal_connect_object (GTK_OBJECT (PlMenuItem), "activate",
		GTK_SIGNAL_FUNC (playlist_save), 
		GTK_OBJECT (hbox)); 
	gtk_menu_append ( GTK_MENU (PlMenu), PlMenuItem);

	PlMenuItem = gtk_menu_item_new_with_label ("Load list");
	gtk_signal_connect_object (GTK_OBJECT (PlMenuItem), "activate",
		GTK_SIGNAL_FUNC (playlist_load), 
		GTK_OBJECT (hbox)); 
	gtk_menu_append ( GTK_MENU (PlMenu), PlMenuItem);

	/* Seperator */
	gtk_menu_append(GTK_MENU(PlMenu), gtk_menu_item_new());

// one scan button per device:
	pDevList = get_dev_list () ; // from config_devices

        LOG (LOG_DEBUG, "Setup Device\n");
	for (pDevCur = g_list_first (pDevList); pDevCur; pDevCur = g_list_next (pDevCur)) {
		gchar  str[80];
		gchar *strDevName;
		strDevName = (gchar *) pDevCur->data ;

		LOG (LOG_DEBUG, "Scan %s", strDevName);
		LOG (LOG_DEBUG, "Have--> %s", strDevName);
		snprintf (str, sizeof(str), "Scan %s", strDevName);

		PlMenuItem = gtk_menu_item_new_with_label (str);
      
		gtk_signal_connect (GTK_OBJECT(PlMenuItem), "activate",
			GTK_SIGNAL_FUNC(playlist_scan_device),
			strDevName);
      
		gtk_menu_append ( GTK_MENU (PlMenu), PlMenuItem);
	}
#endif
	gtk_widget_show_all ( GTK_WIDGET(PlMenu) );

	if (event->type == GDK_BUTTON_PRESS) {
	   	GdkEventButton *bevent = (GdkEventButton *) event;
	   	selection = GTK_CLIST (hbox)->selection;
	     	selection_end = GTK_CLIST (hbox)->selection_end;
	     	//insert selection handling
	     	if (bevent->button == 3)
			gtk_menu_popup (GTK_MENU (PlMenu),NULL, NULL, NULL, NULL, bevent->button, bevent->time);
	    	return TRUE;
	     }
	return FALSE;
}
