/*****
*
* This file is part of the OMI program.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by 
* the Free Software Foundation; either version 2, or (at your option)
* any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; see the file COPYING.  If not, write to
* the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*
*****/

/* Open Media System - oms
 *
 * the playlist
 *
 * Copyright (C) 1999-2000  Thomas Mirlacher, Guenter Bartsch
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *  Changes:
 *    Dominik Schnitzer <dominik@schnitzer.at> - December 26, 2000.
 *    - distinction between playing row and selected row
 *    - big cleanups
 *    - gtk fixes
 *    - todo: chapter change callback doesn't work
 */


#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <inttypes.h>
#include <sys/stat.h>
#include <gdk/gdktypes.h>

#include "main.h"
#include "config_gui.h"
#include "panel_gui.h"
#include "messagebox.h"
#include "playlist.h"
#include "playlist_op.h"

#include "oms/oms.h"

void cb_chapter_change (void *data, int my_part);
void scan_devices (void);
struct omsitem_info *playlist_get_omsitem (gint which);

extern struct player_global_struct player_global;
extern GtkWidget *widget_playlist;

extern playlist_global_t playlist_global;

#define NAVITEM_CURRENT 0
#define NAVITEM_NEXT    1
#define NAVITEM_PREV   -1


/**
 * sync playlist and player
 *
 *    params : navitem_info == navigation information, to uptadte the playlist and
 *              player gui
 *   returns : nothing
 **/

void playlist_sync_player (struct omsitem_info *navitem_info)
{
	if (navitem_info){
		playlist_global.current_navitem = navitem_info;

		player_global.nCurTitle = navitem_info->titlenr;
		player_global.nCurChapter = navitem_info->chapternr;

		if (player_global.strCurSourceName)
			g_free (player_global.strCurSourceName);
		player_global.strCurSourceName = g_strdup(navitem_info->title);

		if (player_global.strCurChapter)
			g_free (player_global.strCurChapter);
		player_global.strCurChapter = g_strdup (navitem_info->chapter);
		
	} else {
		playlist_global.current_navitem = NULL;
		playlist_global.first_set = FALSE;
		playlist_global.selected_row = -1;
		playlist_global.playing_row = -1;
	
		if (player_global.strCurSourceName)
			g_free (player_global.strCurSourceName);
		player_global.strCurSourceName = g_strdup("none");

		if (player_global.strCurChapter)
			g_free (player_global.strCurChapter);
		player_global.strCurChapter = g_strdup("");
		
		player_global.nCurChapter = 0;
		player_global.nCurTitle = 0;
	}

	panel_update_ui();
	return;
}


/**
 * play desired playlist item
 *
 *    params : which == number of item to play
 *             start_playing == start playing now, false if player is stopped
 *   returns : nothing
 **/

gboolean playlist_play_item (gint which, gboolean start_playing)
{
	struct omsitem_info *navitem_info = playlist_get_omsitem (which);

	/* cancel if no data is passed */
	if (!navitem_info)
		return FALSE;
		
	/* If our currently opened source differs from the desired one open desired one */
	if ((!player_global.strCurSource || strcmp (player_global.strCurSource, navitem_info->input)) && start_playing) {
		LOG (LOG_DEBUG, "playlist: switching input form %s to %s", player_global.strCurSource, navitem_info->input);

		/* DENT: free the nav_tree !!! */
		if (omsOpenInput (navitem_info->input) < 0) {
			show_message("Error", MB_OK, "Error switching input to %s", navitem_info->input);
			return FALSE;
		}

		/* register callbacks */

		if (player_global.strCurSource)
			g_free (player_global.strCurSource);
		player_global.strCurSource = g_strdup(navitem_info->input);
	}

	/* select the row currently played */
	gtk_clist_set_background(GTK_CLIST(playlist_global.flatlist), playlist_global.playing_row, NULL);
	gtk_clist_set_background(GTK_CLIST(playlist_global.flatlist), which, &playlist_global.color_playing);
	playlist_global.playing_row = which;
	playlist_sync_player (navitem_info);

	if (start_playing)
		omsPartPlay (navitem_info->titlenr, navitem_info->chapternr);
		
	panel_update_ui();

	return TRUE;
}


/**
 * get the omsitem info of a playlist item
 *
 *    params : which ==
 *   returns : and omsitem_info structure, filled with the desired 
 **/

struct omsitem_info *playlist_get_omsitem (gint which)
{
	/* check if which is a valid playlist item */
	if ((which >= 0) && (which < GTK_CLIST(playlist_global.flatlist)->rows)) {
		struct omsitem_info* navitem_info;
	  
		navitem_info = (struct omsitem_info *) gtk_clist_get_row_data (GTK_CLIST(playlist_global.flatlist), which);
		return navitem_info;
	}

	return NULL;
}


/**
 * play previous, next and currently selected chapter
 *
 *    params : none
 *   returns : nothing
 **/

gboolean playlist_play_next ()
{
	if (oms_get_status () == STATUS_PLAY)
		return playlist_play_item (playlist_global.playing_row + 1, TRUE);
	else
		return playlist_play_item (playlist_global.playing_row + 1, FALSE);
}

gboolean playlist_play_prev ()
{
	if (oms_get_status () == STATUS_PLAY)
		return playlist_play_item (playlist_global.playing_row - 1, TRUE);
	else
		return playlist_play_item (playlist_global.playing_row - 1, FALSE);
}

gboolean playlist_play_current () 
{
	if (!playlist_global.current_navitem) {
		scan_devices ();
		panel_update_ui ();
	}

	return playlist_play_item (playlist_global.playing_row, TRUE);
}


/**
 * Callback: chapter changes
 *
 *    params : *data == 
 *             my_part == 
 *   returns : nothing
 **/

void cb_chapter_change (void *data, int my_part)
{
//	THREADSAFE(if (!playlist_play_next ()) oms_set_status (STATUS_STOP));
}


/**
 * The button scan device is pressed
 *
 *    params : *button == button that caused the action
 *             *dev_name == the device we need to scan
 *   returns : nothing
 **/

void playlist_scan_device (GtkButton *button, gchar *dev_name)
{
	gint i, s;
	nav_tree_t *nav_tree;
	
	LOG (LOG_DEBUG, "playlist: scanning the device: %s", dev_name);

	gtk_widget_hide (playlist_global.flatlist);
	
	playlist_clear(NULL, NULL);

	/* DENT: free the nav_tree !!! */
	omsCloseInput ();

	if (omsOpenInput (dev_name) < 0) { 
		THREADSAFE (show_message("Error", MB_OK, "Error opening device %s for scanning", dev_name));
		gtk_widget_show (playlist_global.flatlist);
		return;
	}

	nav_tree = omsGetInfo (OMS_INFO_PROGRAM);
  
	/* i is the Title number */
	for (i = 0; nav_tree && i < nav_tree->num; i++) {
	
		nav_subtree_t *nav_sub = nav_tree->title + i;
    
		/* s is the Chapternumber */
		for (s = 0; s < nav_sub->num; s++) {
		
			nav_tree_data_t *oms_data = nav_sub->sub + s;
			gchar **devicetype = g_strsplit(dev_name, ":", 2);
			
			/* We append our playlist entries */ 
			playlist_add_entry (devicetype[0], devicetype[1], devicetype[0], oms_data->name, i, s, -1);
		}
	}

	player_global.strCurSource = g_strdup(dev_name);

	gtk_widget_show (playlist_global.flatlist);
}


/**
 * Scan all available devices for AV
 *
 *    params : none
 *   returns : nothing
 **/

void scan_devices (void)
{
	GList      *dev_list;
	playlist_global.first_set = FALSE;

	/* get a list of all devices from the config, dev_list memory is not handled by us */
	dev_list = g_list_first (get_dev_list ());

	/* scan all devices for AV */
	while (dev_list) {
  		gchar *dev_name = (gchar *) dev_list->data ;
		playlist_scan_device (NULL, dev_name);
		dev_list = g_list_next (dev_list);
	}
}


/**
 * Scan all available devices for AV
 *
 *    params : omsitem_info == the infostruture to free
 *   returns : nothing
 **/

void omsitem_free(struct omsitem_info *navitem_info)
{
	
	if (navitem_info) {
		if (navitem_info->input) g_free(navitem_info->input);
		if (navitem_info->title) g_free(navitem_info->title);
		if (navitem_info->chapter) g_free(navitem_info->chapter);
		LOG (LOG_DEBUG, "playlist: freeing navitem_info");
		free (navitem_info);
	}
}
